// Copyright 2025 The BoringSSL Authors
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     https://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

#include <gtest/gtest.h>

#include <openssl/bytestring.h>
#include <openssl/xwing.h>

#include "../test/test_util.h"

namespace {

TEST(XWingTest, EncapsulateDecapsulate) {
  uint8_t public_key[XWING_PUBLIC_KEY_BYTES];
  XWING_private_key private_key;
  ASSERT_TRUE(XWING_generate_key(public_key, &private_key));

  uint8_t ciphertext[XWING_CIPHERTEXT_BYTES];
  uint8_t shared_secret[XWING_SHARED_SECRET_BYTES];
  ASSERT_TRUE(XWING_encap(ciphertext, shared_secret, public_key));

  uint8_t decapsulated[XWING_SHARED_SECRET_BYTES];
  ASSERT_TRUE(XWING_decap(decapsulated, ciphertext, &private_key));
  EXPECT_EQ(Bytes(decapsulated), Bytes(shared_secret));
}

TEST(XWingTest, PublicFromPrivate) {
  uint8_t public_key[XWING_PUBLIC_KEY_BYTES];
  XWING_private_key private_key;
  ASSERT_TRUE(XWING_generate_key(public_key, &private_key));

  uint8_t public_key2[XWING_PUBLIC_KEY_BYTES];
  ASSERT_TRUE(XWING_public_from_private(public_key2, &private_key));
  EXPECT_EQ(Bytes(public_key2), Bytes(public_key));
}

TEST(XWingTest, MarshalParsePrivateKey) {
  uint8_t public_key[XWING_PUBLIC_KEY_BYTES];
  XWING_private_key private_key;
  ASSERT_TRUE(XWING_generate_key(public_key, &private_key));

  // Serialize private key.
  uint8_t encoded_private_key[XWING_PRIVATE_KEY_BYTES];
  CBB cbb;
  CBB_init_fixed(&cbb, encoded_private_key, XWING_PRIVATE_KEY_BYTES);
  ASSERT_TRUE(XWING_marshal_private_key(&cbb, &private_key));
  ASSERT_EQ(CBB_len(&cbb), (size_t)XWING_PRIVATE_KEY_BYTES);

  // Parse private key.
  XWING_private_key parsed_private_key;
  CBS cbs;
  CBS_init(&cbs, encoded_private_key, XWING_PRIVATE_KEY_BYTES);
  ASSERT_TRUE(XWING_parse_private_key(&parsed_private_key, &cbs));

  // Check that both have a consistent behavior.
  uint8_t ciphertext[XWING_CIPHERTEXT_BYTES];
  uint8_t shared_secret[XWING_SHARED_SECRET_BYTES];
  ASSERT_TRUE(XWING_encap(ciphertext, shared_secret, public_key));

  uint8_t decapsulated[XWING_SHARED_SECRET_BYTES];
  ASSERT_TRUE(XWING_decap(decapsulated, ciphertext, &parsed_private_key));
  EXPECT_EQ(Bytes(decapsulated), Bytes(shared_secret));
}

TEST(XWingTest, TestVector) {
  // Taken from
  // https://datatracker.ietf.org/doc/html/draft-connolly-cfrg-xwing-kem-06,
  // Appendix C.
  const uint8_t kPrivateKey[XWING_PRIVATE_KEY_BYTES] = {
      0x7f, 0x9c, 0x2b, 0xa4, 0xe8, 0x8f, 0x82, 0x7d, 0x61, 0x60, 0x45,
      0x50, 0x76, 0x05, 0x85, 0x3e, 0xd7, 0x3b, 0x80, 0x93, 0xf6, 0xef,
      0xbc, 0x88, 0xeb, 0x1a, 0x6e, 0xac, 0xfa, 0x66, 0xef, 0x26};
  const uint8_t kExpectedPublicKey[XWING_PUBLIC_KEY_BYTES] = {
      0xe2, 0x23, 0x6b, 0x35, 0xa8, 0xc2, 0x4b, 0x39, 0xb1, 0x0a, 0xa1, 0x32,
      0x3a, 0x96, 0xa9, 0x19, 0xa2, 0xce, 0xd8, 0x84, 0x00, 0x63, 0x3a, 0x7b,
      0x07, 0x13, 0x17, 0x13, 0xfc, 0x14, 0xb2, 0xb5, 0xb1, 0x9c, 0xfc, 0x3d,
      0xa5, 0xfa, 0x1a, 0x92, 0xc4, 0x9f, 0x25, 0x51, 0x3e, 0x0f, 0xd3, 0x0d,
      0x6b, 0x16, 0x11, 0xc9, 0xab, 0x96, 0x35, 0xd7, 0x08, 0x67, 0x27, 0xa4,
      0xb7, 0xd2, 0x1d, 0x34, 0x24, 0x4e, 0x66, 0x96, 0x9c, 0xf1, 0x5b, 0x3b,
      0x2a, 0x78, 0x53, 0x29, 0xf6, 0x1b, 0x09, 0x6b, 0x27, 0x7e, 0xa0, 0x37,
      0x38, 0x34, 0x79, 0xa6, 0xb5, 0x56, 0xde, 0x72, 0x31, 0xfe, 0x4b, 0x7f,
      0xa9, 0xc9, 0xac, 0x24, 0xc0, 0x69, 0x9a, 0x00, 0x18, 0xa5, 0x25, 0x34,
      0x01, 0xba, 0xcf, 0xa9, 0x05, 0xca, 0x81, 0x65, 0x73, 0xe5, 0x6a, 0x2d,
      0x2e, 0x06, 0x7e, 0x9b, 0x72, 0x87, 0x53, 0x3b, 0xa1, 0x3a, 0x93, 0x7d,
      0xed, 0xb3, 0x1f, 0xa4, 0x4b, 0xac, 0xed, 0x40, 0x76, 0x99, 0x23, 0x61,
      0x00, 0x34, 0xae, 0x31, 0xe6, 0x19, 0xa1, 0x70, 0x24, 0x51, 0x99, 0xb3,
      0xc5, 0xc3, 0x98, 0x64, 0x85, 0x9f, 0xe1, 0xb4, 0xc9, 0x71, 0x7a, 0x07,
      0xc3, 0x04, 0x95, 0xbd, 0xfb, 0x98, 0xa0, 0xa0, 0x02, 0xcc, 0xf5, 0x6c,
      0x12, 0x86, 0xce, 0xf5, 0x04, 0x1d, 0xed, 0xe3, 0xc4, 0x4c, 0xf1, 0x6b,
      0xf5, 0x62, 0xc7, 0x44, 0x85, 0x18, 0x02, 0x6b, 0x3d, 0x8b, 0x99, 0x40,
      0x68, 0x0a, 0xbd, 0x38, 0xa1, 0x57, 0x5f, 0xd2, 0x7b, 0x58, 0xda, 0x06,
      0x3b, 0xfa, 0xc3, 0x2c, 0x39, 0xc3, 0x08, 0x69, 0x37, 0x4c, 0x05, 0xc1,
      0xae, 0xb1, 0x89, 0x8b, 0x6b, 0x30, 0x3c, 0xc6, 0x8b, 0xe4, 0x55, 0x34,
      0x6e, 0xe0, 0xaf, 0x69, 0x96, 0x36, 0x22, 0x4a, 0x14, 0x8c, 0xa2, 0xae,
      0xa1, 0x04, 0x63, 0x11, 0x1c, 0x70, 0x9f, 0x69, 0xb6, 0x9c, 0x70, 0xce,
      0x85, 0x38, 0x74, 0x66, 0x98, 0xc4, 0xc6, 0x0a, 0x9a, 0xef, 0x00, 0x30,
      0xc7, 0x92, 0x4c, 0xee, 0xc4, 0x2a, 0x5d, 0x36, 0x81, 0x6f, 0x54, 0x5e,
      0xae, 0x13, 0x29, 0x34, 0x60, 0xb3, 0xac, 0xb3, 0x7e, 0xa0, 0xe1, 0x3d,
      0x70, 0xe4, 0xaa, 0x78, 0x68, 0x6d, 0xa3, 0x98, 0xa8, 0x39, 0x7c, 0x08,
      0xea, 0xf9, 0x68, 0x82, 0x11, 0x3f, 0xe4, 0xf7, 0xba, 0xd4, 0xda, 0x40,
      0xb0, 0x50, 0x1e, 0x1c, 0x75, 0x3e, 0xfe, 0x73, 0x05, 0x3c, 0x87, 0x01,
      0x4e, 0x86, 0x61, 0xc3, 0x30, 0x99, 0xaf, 0xe8, 0xbe, 0xde, 0x41, 0x4a,
      0x5b, 0x1a, 0xa2, 0x7d, 0x83, 0x92, 0xb3, 0xe1, 0x31, 0xe9, 0xa7, 0x0c,
      0x10, 0x55, 0x87, 0x82, 0x40, 0xca, 0xd0, 0xf4, 0x0d, 0x5f, 0xe3, 0xcd,
      0xf8, 0x52, 0x36, 0xea, 0xd9, 0x7e, 0x2a, 0x97, 0x44, 0x83, 0x63, 0xb2,
      0x80, 0x8c, 0xaa, 0xfd, 0x51, 0x6c, 0xd2, 0x50, 0x52, 0xc5, 0xc3, 0x62,
      0x54, 0x3c, 0x25, 0x17, 0xe4, 0xac, 0xd0, 0xe6, 0x0e, 0xc0, 0x71, 0x63,
      0x00, 0x9b, 0x64, 0x25, 0xfc, 0x32, 0x27, 0x7a, 0xce, 0xe7, 0x1c, 0x24,
      0xba, 0xb5, 0x3e, 0xd9, 0xf2, 0x9e, 0x74, 0xc6, 0x6a, 0x0a, 0x35, 0x64,
      0x95, 0x59, 0x98, 0xd7, 0x6b, 0x96, 0xa9, 0xa8, 0xb5, 0x0d, 0x16, 0x35,
      0xa4, 0xd7, 0xa6, 0x7e, 0xb4, 0x2d, 0xf5, 0x64, 0x4d, 0x33, 0x04, 0x57,
      0x29, 0x3a, 0x80, 0x42, 0xf5, 0x3c, 0xc7, 0xa6, 0x92, 0x88, 0xf1, 0x7e,
      0xd5, 0x58, 0x27, 0xe8, 0x2b, 0x28, 0xe8, 0x26, 0x65, 0xa8, 0x6a, 0x14,
      0xfb, 0xd9, 0x66, 0x45, 0xec, 0xa8, 0x17, 0x2c, 0x04, 0x4f, 0x83, 0xbc,
      0x0d, 0x8c, 0x0b, 0x4c, 0x86, 0x26, 0x98, 0x56, 0x31, 0xca, 0x87, 0xaf,
      0x82, 0x90, 0x68, 0xf1, 0x35, 0x89, 0x63, 0xcb, 0x33, 0x36, 0x64, 0xca,
      0x48, 0x27, 0x63, 0xba, 0x3b, 0x3b, 0xb2, 0x08, 0x57, 0x7f, 0x9b, 0xa6,
      0xac, 0x62, 0xc2, 0x5f, 0x76, 0x59, 0x27, 0x43, 0xb6, 0x4b, 0xe5, 0x19,
      0x31, 0x77, 0x14, 0xcb, 0x41, 0x02, 0xcb, 0x7b, 0x2f, 0x9a, 0x25, 0xb2,
      0xb4, 0xf0, 0x61, 0x5d, 0xe3, 0x1d, 0xec, 0xd9, 0xca, 0x55, 0x02, 0x6d,
      0x6d, 0xa0, 0xb6, 0x51, 0x11, 0xb1, 0x6f, 0xe5, 0x2f, 0xee, 0xd8, 0xa4,
      0x87, 0xe1, 0x44, 0x46, 0x2a, 0x6d, 0xba, 0x93, 0x72, 0x8f, 0x50, 0x0b,
      0x6f, 0xfc, 0x49, 0xe5, 0x15, 0x56, 0x9e, 0xf2, 0x5f, 0xed, 0x17, 0xaf,
      0xf5, 0x20, 0x50, 0x73, 0x68, 0x25, 0x35, 0x25, 0x86, 0x0f, 0x58, 0xbe,
      0x3b, 0xe6, 0x1c, 0x96, 0x46, 0x04, 0xa6, 0xac, 0x81, 0x4e, 0x69, 0x35,
      0x59, 0x64, 0x02, 0xa5, 0x20, 0xa4, 0x67, 0x0b, 0x3d, 0x28, 0x43, 0x18,
      0x86, 0x65, 0x93, 0xd1, 0x5a, 0x4b, 0xb0, 0x1c, 0x35, 0xe3, 0xe5, 0x87,
      0xee, 0x0c, 0x67, 0xd2, 0x88, 0x0d, 0x6f, 0x24, 0x07, 0xfb, 0x7a, 0x70,
      0x71, 0x2b, 0x83, 0x8d, 0xeb, 0x96, 0xc5, 0xd7, 0xbf, 0x2b, 0x44, 0xbc,
      0xf6, 0x03, 0x8c, 0xcb, 0xe3, 0x3f, 0xbc, 0xf5, 0x1a, 0x54, 0xa5, 0x84,
      0xfe, 0x90, 0x08, 0x3c, 0x91, 0xc7, 0xa6, 0xd4, 0x3d, 0x4f, 0xb1, 0x5f,
      0x48, 0xc6, 0x0c, 0x2f, 0xd6, 0x6e, 0x0a, 0x8a, 0xad, 0x4a, 0xd6, 0x4e,
      0x5c, 0x42, 0xbb, 0x88, 0x77, 0xc0, 0xeb, 0xec, 0x2b, 0x5e, 0x38, 0x7c,
      0x8a, 0x98, 0x8f, 0xdc, 0x23, 0xbe, 0xb9, 0xe1, 0x6c, 0x87, 0x57, 0x78,
      0x1e, 0x0a, 0x14, 0x99, 0xc6, 0x1e, 0x13, 0x8c, 0x21, 0xf2, 0x16, 0xc2,
      0x9d, 0x07, 0x69, 0x79, 0x87, 0x1c, 0xaa, 0x69, 0x42, 0xba, 0xfc, 0x09,
      0x05, 0x44, 0xbe, 0xe9, 0x9b, 0x54, 0xb1, 0x6c, 0xb9, 0xa9, 0xa3, 0x64,
      0xd6, 0x24, 0x6d, 0x9f, 0x42, 0xcc, 0xe5, 0x3c, 0x66, 0xb5, 0x9c, 0x45,
      0xc8, 0xf9, 0xae, 0x92, 0x99, 0xa7, 0x5d, 0x15, 0x18, 0x0c, 0x3c, 0x95,
      0x21, 0x51, 0xa9, 0x1b, 0x7a, 0x10, 0x77, 0x24, 0x29, 0xdc, 0x4c, 0xba,
      0xe6, 0xfc, 0xc6, 0x22, 0xfa, 0x80, 0x18, 0xc6, 0x34, 0x39, 0xf8, 0x90,
      0x63, 0x0b, 0x99, 0x28, 0xdb, 0x6b, 0xb7, 0xf9, 0x43, 0x8a, 0xe4, 0x06,
      0x5e, 0xd3, 0x4d, 0x73, 0xd4, 0x86, 0xf3, 0xf5, 0x2f, 0x90, 0xf0, 0x80,
      0x7d, 0xc8, 0x8d, 0xfd, 0xd8, 0xc7, 0x28, 0xe9, 0x54, 0xf1, 0xac, 0x35,
      0xc0, 0x6c, 0x00, 0x0c, 0xe4, 0x1a, 0x05, 0x82, 0x58, 0x0e, 0x3b, 0xb5,
      0x7b, 0x67, 0x29, 0x72, 0x89, 0x0a, 0xc5, 0xe7, 0x98, 0x8e, 0x78, 0x50,
      0x65, 0x71, 0x16, 0xf1, 0xb5, 0x7d, 0x08, 0x09, 0xaa, 0xed, 0xec, 0x0b,
      0xed, 0xe1, 0xae, 0x14, 0x81, 0x48, 0x31, 0x1c, 0x6f, 0x7e, 0x31, 0x73,
      0x46, 0xe5, 0x18, 0x9f, 0xb8, 0xcd, 0x63, 0x5b, 0x98, 0x6f, 0x8c, 0x0b,
      0xdd, 0x27, 0x64, 0x1c, 0x58, 0x4b, 0x77, 0x8b, 0x3a, 0x91, 0x1a, 0x80,
      0xbe, 0x1c, 0x96, 0x92, 0xab, 0x8e, 0x1b, 0xbb, 0x12, 0x83, 0x95, 0x73,
      0xcc, 0xe1, 0x9d, 0xf1, 0x83, 0xb4, 0x58, 0x35, 0xbb, 0xb5, 0x50, 0x52,
      0xf9, 0xfc, 0x66, 0xa1, 0x67, 0x8e, 0xf2, 0xa3, 0x6d, 0xea, 0x78, 0x41,
      0x1e, 0x6c, 0x8d, 0x60, 0x50, 0x1b, 0x4e, 0x60, 0x59, 0x2d, 0x13, 0x69,
      0x8a, 0x94, 0x3b, 0x50, 0x91, 0x85, 0xdb, 0x91, 0x2e, 0x2e, 0xa1, 0x0b,
      0xe0, 0x61, 0x71, 0x23, 0x6b, 0x32, 0x7c, 0x71, 0x71, 0x60, 0x94, 0xc9,
      0x64, 0xa6, 0x8b, 0x03, 0x37, 0x7f, 0x51, 0x3a, 0x05, 0xbc, 0xd9, 0x9c,
      0x1f, 0x34, 0x65, 0x83, 0xbb, 0x05, 0x29, 0x77, 0xa1, 0x0a, 0x12, 0xad,
      0xfc, 0x75, 0x80, 0x34, 0xe5, 0x61, 0x7d, 0xa4, 0xc1, 0x27, 0x65, 0x85,
      0xe5, 0x77, 0x4e, 0x1f, 0x3b, 0x99, 0x78, 0xb0, 0x9d, 0x0e, 0x9c, 0x44,
      0xd3, 0xbc, 0x86, 0x15, 0x1c, 0x43, 0xaa, 0xd1, 0x85, 0x71, 0x27, 0x17,
      0x34, 0x02, 0x23, 0xac, 0x38, 0x1d, 0x21, 0x15, 0x0a, 0x04, 0x29, 0x4e,
      0x97, 0xbb, 0x13, 0xbb, 0xda, 0x21, 0xb5, 0xa1, 0x82, 0xb6, 0xda, 0x96,
      0x9e, 0x19, 0xa7, 0xfd, 0x07, 0x27, 0x37, 0xfa, 0x8e, 0x88, 0x0a, 0x53,
      0xc2, 0x42, 0x8e, 0x3d, 0x04, 0x9b, 0x7d, 0x21, 0x97, 0x40, 0x52, 0x96,
      0xdd, 0xb3, 0x61, 0x91, 0x2a, 0x7b, 0xcf, 0x48, 0x27, 0xce, 0xd6, 0x11,
      0xd0, 0xc7, 0xa7, 0xda, 0x10, 0x4d, 0xde, 0x43, 0x22, 0x09, 0x53, 0x39,
      0xf6, 0x4a, 0x61, 0xd5, 0xbb, 0x10, 0x8f, 0xf0, 0xbf, 0x4d, 0x78, 0x0c,
      0xae, 0x50, 0x9f, 0xb2, 0x2c, 0x25, 0x69, 0x14, 0x19, 0x3f, 0xf7, 0x34,
      0x90, 0x42, 0x58, 0x12, 0x37, 0xd5, 0x22, 0x82, 0x88, 0x24, 0xee, 0x3b,
      0xdf, 0xd0, 0x7f, 0xb0, 0x3f, 0x1f, 0x94, 0x2d, 0x2e, 0xa1, 0x79, 0xfe,
      0x72, 0x2f, 0x06, 0xcc, 0x03, 0xde, 0x5b, 0x69, 0x85, 0x9e, 0xdb, 0x06,
      0xef, 0xf3, 0x89, 0xb2, 0x7d, 0xce, 0x59, 0x84, 0x45, 0x70, 0x21, 0x62,
      0x23, 0x59, 0x3d, 0x4b, 0xa3, 0x2d, 0x9a, 0xba, 0xc8, 0xcd, 0x04, 0x90,
      0x40, 0xef, 0x65, 0x34,
  };
  const uint8_t kEphemeralSeed[64] = {
      0x3c, 0xb1, 0xee, 0xa9, 0x88, 0x00, 0x4b, 0x93, 0x10, 0x3c, 0xfb,
      0x0a, 0xee, 0xfd, 0x2a, 0x68, 0x6e, 0x01, 0xfa, 0x4a, 0x58, 0xe8,
      0xa3, 0x63, 0x9c, 0xa8, 0xa1, 0xe3, 0xf9, 0xae, 0x57, 0xe2, 0x35,
      0xb8, 0xcc, 0x87, 0x3c, 0x23, 0xdc, 0x62, 0xb8, 0xd2, 0x60, 0x16,
      0x9a, 0xfa, 0x2f, 0x75, 0xab, 0x91, 0x6a, 0x58, 0xd9, 0x74, 0x91,
      0x88, 0x35, 0xd2, 0x5e, 0x6a, 0x43, 0x50, 0x85, 0xb2,
  };
  const uint8_t kExpectedSharedSecret[XWING_SHARED_SECRET_BYTES] = {
      0xd2, 0xdf, 0x05, 0x22, 0x12, 0x8f, 0x09, 0xdd, 0x8e, 0x2c, 0x92,
      0xb1, 0xe9, 0x05, 0xc7, 0x93, 0xd8, 0xf5, 0x7a, 0x54, 0xc3, 0xda,
      0x25, 0x86, 0x1f, 0x10, 0xbf, 0x4c, 0xa6, 0x13, 0xe3, 0x84};
  const uint8_t kExpectedCiphertext[XWING_CIPHERTEXT_BYTES] = {
      0xb8, 0x3a, 0xa8, 0x28, 0xd4, 0xd6, 0x2b, 0x9a, 0x83, 0xce, 0xff, 0xe1,
      0xd3, 0xd3, 0xbb, 0x1e, 0xf3, 0x12, 0x64, 0x64, 0x3c, 0x07, 0x0c, 0x57,
      0x98, 0x92, 0x7e, 0x41, 0xfb, 0x07, 0x91, 0x4a, 0x27, 0x3f, 0x8f, 0x96,
      0xe7, 0x82, 0x6c, 0xd5, 0x37, 0x5a, 0x28, 0x3d, 0x7d, 0xa8, 0x85, 0x30,
      0x4c, 0x5d, 0xe0, 0x51, 0x6a, 0x0f, 0x06, 0x54, 0x24, 0x3d, 0xc5, 0xb9,
      0x7f, 0x8b, 0xfe, 0xb8, 0x31, 0xf6, 0x82, 0x51, 0x21, 0x9a, 0xab, 0xdd,
      0x72, 0x3b, 0xc6, 0x51, 0x20, 0x41, 0xac, 0xba, 0xef, 0x8a, 0xf4, 0x42,
      0x65, 0x52, 0x49, 0x42, 0xb9, 0x02, 0xe6, 0x8f, 0xfd, 0x23, 0x22, 0x1c,
      0xda, 0x70, 0xb1, 0xb5, 0x5d, 0x77, 0x6a, 0x92, 0xd1, 0x14, 0x3e, 0xa3,
      0xa0, 0xc4, 0x75, 0xf6, 0x3e, 0xe6, 0x89, 0x01, 0x57, 0xc7, 0x11, 0x6d,
      0xae, 0x3f, 0x62, 0xbf, 0x72, 0xf6, 0x0a, 0xcd, 0x2b, 0xb8, 0xcc, 0x31,
      0xce, 0x2b, 0xa0, 0xde, 0x36, 0x4f, 0x52, 0xb8, 0xed, 0x38, 0xc7, 0x9d,
      0x71, 0x97, 0x15, 0x96, 0x3a, 0x5d, 0xd3, 0x84, 0x2d, 0x8e, 0x8b, 0x43,
      0xab, 0x70, 0x4e, 0x47, 0x59, 0xb5, 0x32, 0x7b, 0xf0, 0x27, 0xc6, 0x3c,
      0x8f, 0xa8, 0x57, 0xc4, 0x90, 0x8d, 0x5a, 0x8a, 0x7b, 0x88, 0xac, 0x7f,
      0x2b, 0xe3, 0x94, 0xd9, 0x3c, 0x37, 0x06, 0xdd, 0xd4, 0xe6, 0x98, 0xcc,
      0x6c, 0xe3, 0x70, 0x10, 0x1f, 0x4d, 0x02, 0x13, 0x25, 0x42, 0x38, 0xb4,
      0xa2, 0xe8, 0x82, 0x1b, 0x6e, 0x41, 0x4a, 0x1c, 0xf2, 0x0f, 0x6c, 0x12,
      0x44, 0xb6, 0x99, 0x04, 0x6f, 0x5a, 0x01, 0xca, 0xa0, 0xa1, 0xa5, 0x55,
      0x16, 0x30, 0x0b, 0x40, 0xd2, 0x04, 0x8c, 0x77, 0xcc, 0x73, 0xaf, 0xba,
      0x79, 0xaf, 0xee, 0xa9, 0xd2, 0xc0, 0x11, 0x8b, 0xdf, 0x2a, 0xdb, 0x88,
      0x70, 0xdc, 0x32, 0x8c, 0x55, 0x16, 0xcc, 0x45, 0xb1, 0xa2, 0x05, 0x81,
      0x41, 0x03, 0x9e, 0x2c, 0x90, 0xa1, 0x10, 0xa9, 0xe1, 0x6b, 0x31, 0x8d,
      0xfb, 0x53, 0xbd, 0x49, 0xa1, 0x26, 0xd6, 0xb7, 0x3f, 0x21, 0x57, 0x87,
      0x51, 0x7b, 0x89, 0x17, 0xcc, 0x01, 0xca, 0xbd, 0x10, 0x7d, 0x06, 0x85,
      0x98, 0x54, 0xee, 0x8b, 0x4f, 0x98, 0x61, 0xc2, 0x26, 0xd3, 0x76, 0x4c,
      0x87, 0x33, 0x9a, 0xb1, 0x6c, 0x36, 0x67, 0xd2, 0xf4, 0x93, 0x84, 0xe5,
      0x54, 0x56, 0xdd, 0x40, 0x41, 0x4b, 0x70, 0xa6, 0xaf, 0x84, 0x15, 0x85,
      0xf4, 0xc9, 0x0c, 0x68, 0x72, 0x5d, 0x57, 0x70, 0x4e, 0xe8, 0xee, 0x7c,
      0xe6, 0xe2, 0xf9, 0xbe, 0x58, 0x2d, 0xbe, 0xe9, 0x85, 0xe0, 0x38, 0xff,
      0xc3, 0x46, 0xeb, 0xfb, 0x4e, 0x22, 0x15, 0x8b, 0x6c, 0x84, 0x37, 0x4a,
      0x9a, 0xb4, 0xa4, 0x4e, 0x1f, 0x91, 0xde, 0x5a, 0xac, 0x51, 0x97, 0xf8,
      0x9b, 0xc5, 0xe5, 0x44, 0x2f, 0x51, 0xf9, 0xa5, 0x93, 0x7b, 0x10, 0x2b,
      0xa3, 0xbe, 0xae, 0xbf, 0x6e, 0x1c, 0x58, 0x38, 0x0a, 0x4a, 0x5f, 0xed,
      0xce, 0x4a, 0x4e, 0x50, 0x26, 0xf8, 0x8f, 0x52, 0x8f, 0x59, 0xff, 0xd2,
      0xdb, 0x41, 0x75, 0x2b, 0x3a, 0x3d, 0x90, 0xef, 0xab, 0xe4, 0x63, 0x89,
      0x9b, 0x7d, 0x40, 0x87, 0x0c, 0x53, 0x0c, 0x88, 0x41, 0xe8, 0x71, 0x2b,
      0x73, 0x36, 0x68, 0xed, 0x03, 0x3a, 0xdb, 0xfa, 0xfb, 0x2d, 0x49, 0xd3,
      0x7a, 0x44, 0xd4, 0x06, 0x4e, 0x58, 0x63, 0xeb, 0x0a, 0xf0, 0xa0, 0x8d,
      0x47, 0xb3, 0xcc, 0x88, 0x83, 0x73, 0xbc, 0x05, 0xf7, 0xa3, 0x3b, 0x84,
      0x1b, 0xc2, 0x58, 0x7c, 0x57, 0xeb, 0x69, 0x55, 0x4e, 0x8a, 0x37, 0x67,
      0xb7, 0x50, 0x69, 0x17, 0xb6, 0xb7, 0x04, 0x98, 0x72, 0x7f, 0x16, 0xea,
      0xc1, 0xa3, 0x6e, 0xc8, 0xd8, 0xcf, 0xaf, 0x75, 0x15, 0x49, 0xf2, 0x27,
      0x7d, 0xb2, 0x77, 0xe8, 0xa5, 0x5a, 0x9a, 0x51, 0x06, 0xb2, 0x3a, 0x02,
      0x06, 0xb4, 0x72, 0x1f, 0xa9, 0xb3, 0x04, 0x85, 0x52, 0xc5, 0xbd, 0x5b,
      0x59, 0x4d, 0x6e, 0x24, 0x7f, 0x38, 0xc1, 0x8c, 0x59, 0x1a, 0xea, 0x7f,
      0x56, 0x24, 0x9c, 0x72, 0xce, 0x7b, 0x11, 0x7a, 0xfc, 0xc3, 0xa8, 0x62,
      0x15, 0x82, 0xf9, 0xcf, 0x71, 0x78, 0x7e, 0x18, 0x3d, 0xee, 0x09, 0x36,
      0x79, 0x76, 0xe9, 0x84, 0x09, 0xad, 0x92, 0x17, 0xa4, 0x97, 0xdf, 0x88,
      0x80, 0x42, 0x38, 0x4d, 0x77, 0x07, 0xa6, 0xb7, 0x8f, 0x5f, 0x7f, 0xb8,
      0x40, 0x9e, 0x3b, 0x53, 0x51, 0x75, 0x37, 0x34, 0x61, 0xb7, 0x76, 0x00,
      0x2d, 0x79, 0x9c, 0xba, 0xd6, 0x28, 0x60, 0xbe, 0x70, 0x57, 0x3e, 0xcb,
      0xe1, 0x3b, 0x24, 0x6e, 0x0d, 0xa7, 0xe9, 0x3a, 0x52, 0x16, 0x8e, 0x0f,
      0xb6, 0xa9, 0x75, 0x6b, 0x89, 0x5e, 0xf7, 0xf0, 0x14, 0x7a, 0x0d, 0xc8,
      0x1b, 0xfa, 0x64, 0x4b, 0x08, 0x8a, 0x92, 0x28, 0x16, 0x0c, 0x0f, 0x9a,
      0xcf, 0x13, 0x79, 0xa2, 0x94, 0x1c, 0xd2, 0x8c, 0x06, 0xeb, 0xc8, 0x0e,
      0x44, 0xe1, 0x7a, 0xa2, 0xf8, 0x17, 0x70, 0x10, 0xaf, 0xd7, 0x8a, 0x97,
      0xce, 0x08, 0x68, 0xd1, 0x62, 0x9e, 0xbb, 0x29, 0x4c, 0x51, 0x51, 0x81,
      0x2c, 0x58, 0x3d, 0xae, 0xb8, 0x86, 0x85, 0x22, 0x0f, 0x4d, 0xa9, 0x11,
      0x81, 0x12, 0xe0, 0x70, 0x41, 0xfc, 0xc2, 0x4d, 0x55, 0x64, 0xa9, 0x9f,
      0xdb, 0xde, 0x28, 0x86, 0x9f, 0xe0, 0x72, 0x23, 0x87, 0xd7, 0xa9, 0xa4,
      0xd1, 0x6e, 0x1c, 0xc8, 0x55, 0x59, 0x17, 0xe0, 0x99, 0x44, 0xaa, 0x5e,
      0xba, 0xaa, 0xec, 0x2c, 0xf6, 0x26, 0x93, 0xaf, 0xad, 0x42, 0xa3, 0xf5,
      0x18, 0xfc, 0xe6, 0x7d, 0x27, 0x3c, 0xc6, 0xc9, 0xfb, 0x54, 0x72, 0xb3,
      0x80, 0xe8, 0x57, 0x3e, 0xc7, 0xde, 0x06, 0xa3, 0xba, 0x2f, 0xd5, 0xf9,
      0x31, 0xd7, 0x25, 0xb4, 0x93, 0x02, 0x6c, 0xb0, 0xac, 0xbd, 0x3f, 0xe6,
      0x2d, 0x00, 0xe4, 0xc7, 0x90, 0xd9, 0x65, 0xd7, 0xa0, 0x3a, 0x3c, 0x0b,
      0x42, 0x22, 0xba, 0x8c, 0x2a, 0x9a, 0x16, 0xe2, 0xac, 0x65, 0x8f, 0x57,
      0x2a, 0xe0, 0xe7, 0x46, 0xea, 0xfc, 0x4f, 0xeb, 0xa0, 0x23, 0x57, 0x6f,
      0x08, 0x94, 0x22, 0x78, 0xa0, 0x41, 0xfb, 0x82, 0xa7, 0x0a, 0x59, 0x5d,
      0x5b, 0xac, 0xbf, 0x29, 0x7c, 0xe2, 0x02, 0x98, 0x98, 0xa7, 0x1e, 0x5c,
      0x3b, 0x0d, 0x1c, 0x62, 0x28, 0xb4, 0x85, 0xb1, 0xad, 0xe5, 0x09, 0xb3,
      0x5f, 0xbc, 0xa7, 0xec, 0xa9, 0x7b, 0x21, 0x32, 0xe7, 0xcb, 0x6b, 0xc4,
      0x65, 0x37, 0x51, 0x46, 0xb7, 0xdc, 0xea, 0xc9, 0x69, 0x30, 0x8a, 0xc0,
      0xc2, 0xac, 0x89, 0xe7, 0x86, 0x3e, 0xb8, 0x94, 0x30, 0x15, 0xb2, 0x43,
      0x14, 0xca, 0xfb, 0x9c, 0x7c, 0x0e, 0x85, 0xfe, 0x54, 0x3d, 0x56, 0x65,
      0x8c, 0x21, 0x36, 0x32, 0x59, 0x9e, 0xfa, 0xbf, 0xc1, 0xec, 0x49, 0xdd,
      0x8c, 0x88, 0x54, 0x7b, 0xb2, 0xcc, 0x40, 0xc9, 0xd3, 0x8c, 0xbd, 0x30,
      0x99, 0xb4, 0x54, 0x78, 0x40, 0x56, 0x05, 0x31, 0xd0, 0x18, 0x8c, 0xd1,
      0xe9, 0xc2, 0x3a, 0x0e, 0xbe, 0xe0, 0xa0, 0x3d, 0x55, 0x77, 0xd6, 0x6b,
      0x1d, 0x2b, 0xcb, 0x4b, 0xaa, 0xf2, 0x1c, 0xc7, 0xfe, 0xf1, 0xe0, 0x38,
      0x06, 0xca, 0x96, 0x29, 0x9d, 0xf0, 0xdf, 0xbc, 0x56, 0xe1, 0xb2, 0xb4,
      0x3e, 0x4f, 0xc2, 0x0c, 0x37, 0xf8, 0x34, 0xc4, 0xaf, 0x62, 0x12, 0x7e,
      0x7d, 0xae, 0x86, 0xc3, 0xc2, 0x5a, 0x2f, 0x69, 0x6a, 0xc8, 0xb5, 0x89,
      0xde, 0xc7, 0x1d, 0x59, 0x5b, 0xfb, 0xe9, 0x4b, 0x5e, 0xd4, 0xbc, 0x07,
      0xd8, 0x00, 0xb3, 0x30, 0x79, 0x6f, 0xda, 0x89, 0xed, 0xb7, 0x7b, 0xe0,
      0x29, 0x41, 0x36, 0x13, 0x93, 0x54, 0xeb, 0x8c, 0xd3, 0x75, 0x91, 0x57,
      0x8f, 0x9c, 0x60, 0x0d, 0xd9, 0xbe, 0x8e, 0xc6, 0x21, 0x9f, 0xdd, 0x50,
      0x7a, 0xdf, 0x33, 0x97, 0xed, 0x4d, 0x68, 0x70, 0x7b, 0x8d, 0x13, 0xb2,
      0x4c, 0xe4, 0xcd, 0x8f, 0xb2, 0x28, 0x51, 0xbf, 0xe9, 0xd6, 0x32, 0x40,
      0x7f, 0x31, 0xed, 0x6f, 0x7c, 0xb1, 0x60, 0x0d, 0xe5, 0x6f, 0x17, 0x57,
      0x67, 0x40, 0xce, 0x2a, 0x32, 0xfc, 0x51, 0x45, 0x03, 0x01, 0x45, 0xcf,
      0xb9, 0x7e, 0x63, 0xe0, 0xe4, 0x1d, 0x35, 0x42, 0x74, 0xa0, 0x79, 0xd3,
      0xe6, 0xfb, 0x2e, 0x15};

  CBS cbs;
  CBS_init(&cbs, kPrivateKey, sizeof(kPrivateKey));
  XWING_private_key private_key;
  ASSERT_TRUE(XWING_parse_private_key(&private_key, &cbs));

  uint8_t public_key[XWING_PUBLIC_KEY_BYTES];
  ASSERT_TRUE(XWING_public_from_private(public_key, &private_key));
  EXPECT_EQ(Bytes(public_key), Bytes(kExpectedPublicKey));

  uint8_t ciphertext[XWING_CIPHERTEXT_BYTES];
  uint8_t shared_secret[XWING_SHARED_SECRET_BYTES];
  ASSERT_TRUE(XWING_encap_external_entropy(ciphertext, shared_secret,
                                           public_key, kEphemeralSeed));
  EXPECT_EQ(Bytes(ciphertext), Bytes(kExpectedCiphertext));
  EXPECT_EQ(Bytes(shared_secret), Bytes(kExpectedSharedSecret));

  uint8_t decapsulated[XWING_SHARED_SECRET_BYTES];
  ASSERT_TRUE(XWING_decap(decapsulated, ciphertext, &private_key));
  EXPECT_EQ(Bytes(decapsulated), Bytes(shared_secret));
}

}  // namespace
