pub use flate2::Compression;
use Encoder;
use filter::Standard;
use std::io::{self, Write};

#[repr(u8)]
#[derive(Clone, Copy, Eq, PartialEq, Hash, Debug)]
/// A color format, which specifies how the bytes represent the pixels.
pub enum ColorFormat {
    /// A format encoding only the luminosity of each pixel.
    ///
    /// Valid bit depths are 1, 2, 4, 8 and 16 bits.
    Gray = 0,
    /// A format encoding the red, green, then blue components of each pixel.
    ///
    /// Valid bit depths are 8 and 16 bits.
    RGB = 2,
    /// A single-channel format encoding the index of each pixel in a palette.
    ///
    /// If you use this format, you probably want to also use `meta::palette`
    /// or `meta::palette_alpha` before writing any image data, because
    /// otherwise, the decoder probably won't know what each index represents.
    ///
    /// Valid bit depths are 1, 2, 4 and 8 bits.
    Palette = 3,
    /// The same as Gray, but with an alpha channel added to the end.
    ///
    /// Valid bit depths are 8 and 16 bits.
    GrayA = 4,
    /// The same as RGB, but with an alpha channel added to the end.
    ///
    /// Valid bit depths are 8 and 16 bits.
    RGBA = 6,
}

impl ColorFormat {
    /// The number of components in each pixel.
    pub fn channels(self) -> usize {
        use self::ColorFormat::*;

        match self {
            Gray    => 1,
            RGB     => 3,
            Palette => 1,
            GrayA   => 2,
            RGBA    => 4,
        }
    }
}

#[derive(Debug)]
/// Image properties and compression options.
pub struct Options {
    /// The compression level.
    pub level:  Compression,
    /// The width of the image, in pixels.
    pub width:  u32,
    /// The height of the image, in pixels.
    pub height: u32,
    /// The number of bits in each channel.
    pub depth:  u8,
    /// The format of the image.
    pub format: ColorFormat,
    /// The size of the compressed data buffer.
    ///
    /// This value determines the maximum size of an IDAT chunk.
    pub buffer: usize,
}

impl Options {
    /// Specifies an 8-bit RGBA image optimizing for image size.
    pub fn smallest(width: u32, height: u32) -> Self {
        Options {
            level: Compression::best(),
            width,
            height,
            depth: 8,
            format: ColorFormat::RGBA,
            buffer: 32 * 1024,
        }
    }

    /// Specifies an 8-bit RGBA image optimizing for encoding speed.
    pub fn fastest(width: u32, height: u32) -> Self {
        Options {
            level: Compression::fast(),
            width,
            height,
            depth: 8,
            format: ColorFormat::RGBA,
            buffer: 32 * 1024,
        }
    }

    /// Make a new encoder with these options and the default filter.
    ///
    /// To customize the filter as well, see `Encoder::new`.
    pub fn build<W>(&self, sink: W) -> io::Result<Encoder<W, Standard>>
    where
        W: Write,
    {
        Encoder::new(self, sink)
    }

    /// The number of bytes occupied by a single row of the uncompressed image.
    pub fn stride(&self) -> usize {
        (
            self.width as usize *
            self.depth as usize *
            self.format.channels() as usize
        + 7) / 8
    }
}
