package typeparams

import (
	"go/ast"
	"go/token"
	"go/types"
	"strings"
	"testing"

	"github.com/google/go-cmp/cmp"
	"golang.org/x/tools/go/ast/astutil"

	"github.com/gopherjs/gopherjs/internal/srctesting"
)

func TestVisitor(t *testing.T) {
	// This test verifies that instance collector is able to discover
	// instantiations of generic types and functions in all possible contexts.
	const src = `package testcase

	type A struct{}
	type B struct{}
	type C struct{}
	type D struct{}
	type E struct{}
	type F struct{}
	type G struct{}

	type typ[T any, V any] []T
	func (t *typ[T, V]) method(x T) {}
	func fun[U any, W any](x U, y W) {}

	func entry1(arg typ[int8, A]) (result typ[int16, A]) {
		fun(1, A{})
		fun[int8, A](1, A{})
		println(fun[int16, A])

		t := typ[int, A]{}
		t.method(0)
		(*typ[int32, A]).method(nil, 0)

		type x struct{ T []typ[int64, A] }
		type y[X any] struct{ T []typ[A, X] }
		_ = y[int8]{}
		_ = y[A]{}

		return
	}

	func entry2[T any](arg typ[int8, T]) (result typ[int16, T]) {
		var zeroT T
		fun(1, zeroT)
		fun[int8, T](1, zeroT)
		println(fun[int16, T])

		t := typ[int, T]{}
		t.method(0)
		(*typ[int32, T]).method(nil, 0)

		type x struct{ T []typ[int64, T] }
		type y[X any] struct{ T []typ[T, X] }
		_ = y[int8]{}
		_ = y[T]{}

		return
	}

	type entry3[T any] struct{
		typ[int, T]
		field1 struct { field2 typ[int8, T] }
	}
	func (e entry3[T]) method(arg typ[int8, T]) (result typ[int16, T]) {
		var zeroT T
		fun(1, zeroT)
		fun[int8, T](1, zeroT)
		println(fun[int16, T])

		t := typ[int, T]{}
		t.method(0)
		(*typ[int32, T]).method(nil, 0)

		type x struct{ T []typ[int64, T] }
		type y[X any] struct{ T []typ[T, X] }
		_ = y[int8]{}
		_ = y[T]{}

		return
	}

	type entry4 struct{
		typ[int, E]
		field1 struct { field2 typ[int8, E] }
	}

	type entry5 = typ[int, F]
	`
	f := srctesting.New(t)
	file := f.Parse("test.go", src)
	info, pkg := f.Check("pkg/test", file)

	lookupObj := func(name string) types.Object {
		return srctesting.LookupObj(pkg, name)
	}
	lookupType := func(name string) types.Type { return lookupObj(name).Type() }
	lookupDecl := func(name string) ast.Node {
		obj := lookupObj(name)
		path, _ := astutil.PathEnclosingInterval(file, obj.Pos(), obj.Pos())
		for _, n := range path {
			switch n.(type) {
			case *ast.FuncDecl, *ast.TypeSpec:
				return n
			}
		}
		t.Fatalf("Could not find AST node representing %v", obj)
		return nil
	}

	// Generates a list of instances we expect to discover from functions and
	// methods. Sentinel type is a type parameter we use uniquely within one
	// context, which allows us to make sure that collection is not being tested
	// against a wrong part of AST.
	instancesInFunc := func(sentinel types.Type) []Instance {
		return []Instance{
			{
				// Called with type arguments inferred.
				Object: lookupObj("fun"),
				TArgs:  []types.Type{types.Typ[types.Int], sentinel},
			}, {
				// Called with type arguments explicitly specified.
				Object: lookupObj("fun"),
				TArgs:  []types.Type{types.Typ[types.Int8], sentinel},
			}, {
				// Passed as an argument.
				Object: lookupObj("fun"),
				TArgs:  []types.Type{types.Typ[types.Int16], sentinel},
			}, {
				// Literal expression.
				Object: lookupObj("typ"),
				TArgs:  []types.Type{types.Typ[types.Int], sentinel},
			}, {
				Object: lookupObj("typ.method"),
				TArgs:  []types.Type{types.Typ[types.Int], sentinel},
			}, {
				// Function argument.
				Object: lookupObj("typ"),
				TArgs:  []types.Type{types.Typ[types.Int8], sentinel},
			}, {
				Object: lookupObj("typ.method"),
				TArgs:  []types.Type{types.Typ[types.Int8], sentinel},
			}, {
				// Function return type.
				Object: lookupObj("typ"),
				TArgs:  []types.Type{types.Typ[types.Int16], sentinel},
			}, {
				Object: lookupObj("typ.method"),
				TArgs:  []types.Type{types.Typ[types.Int16], sentinel},
			}, {
				// Method expression.
				Object: lookupObj("typ"),
				TArgs:  []types.Type{types.Typ[types.Int32], sentinel},
			}, {
				Object: lookupObj("typ.method"),
				TArgs:  []types.Type{types.Typ[types.Int32], sentinel},
			}, {
				// Type decl statement.
				Object: lookupObj("typ"),
				TArgs:  []types.Type{types.Typ[types.Int64], sentinel},
			}, {
				Object: lookupObj("typ.method"),
				TArgs:  []types.Type{types.Typ[types.Int64], sentinel},
			},
		}
	}

	// Generates a list of instances we expect to discover from type declarations.
	// Sentinel type is a type parameter we use uniquely within one context, which
	// allows us to make sure that collection is not being tested against a wrong
	// part of AST.
	instancesInType := func(sentinel types.Type) []Instance {
		return []Instance{
			{
				Object: lookupObj("typ"),
				TArgs:  []types.Type{types.Typ[types.Int], sentinel},
			}, {
				Object: lookupObj("typ.method"),
				TArgs:  []types.Type{types.Typ[types.Int], sentinel},
			}, {
				Object: lookupObj("typ"),
				TArgs:  []types.Type{types.Typ[types.Int8], sentinel},
			}, {
				Object: lookupObj("typ.method"),
				TArgs:  []types.Type{types.Typ[types.Int8], sentinel},
			},
		}
	}

	tests := []struct {
		descr    string
		resolver *Resolver
		node     ast.Node
		want     []Instance
	}{
		{
			descr:    "non-generic function",
			resolver: nil,
			node:     lookupDecl("entry1"),
			want: append(
				instancesInFunc(lookupType("A")),
				Instance{
					Object: lookupObj("entry1.y"),
					TArgs:  []types.Type{types.Typ[types.Int8]},
				},
				Instance{
					Object: lookupObj("entry1.y"),
					TArgs:  []types.Type{lookupType("A")},
				},
			),
		}, {
			descr: "generic function",
			resolver: NewResolver(
				types.NewContext(),
				Instance{
					Object: lookupObj("entry2"),
					TArgs:  []types.Type{lookupType("B")},
				}),
			node: lookupDecl("entry2"),
			want: append(
				instancesInFunc(lookupType("B")),
				Instance{
					Object: lookupObj("entry2.x"),
					TNest:  []types.Type{lookupType("B")},
				},
				Instance{
					Object: lookupObj("entry1.y"),
					TNest:  []types.Type{lookupType("B")},
					TArgs:  []types.Type{types.Typ[types.Int8]},
				},
				Instance{
					Object: lookupObj("entry2.y"),
					TNest:  []types.Type{lookupType("B")},
					TArgs:  []types.Type{lookupType("B")},
				},
			),
		}, {
			descr: "generic method",
			resolver: NewResolver(
				types.NewContext(),
				Instance{
					Object: lookupObj("entry3.method"),
					TArgs:  []types.Type{lookupType("C")},
				}),
			node: lookupDecl("entry3.method"),
			want: append(
				instancesInFunc(lookupType("C")),
				Instance{
					Object: lookupObj("entry3"),
					TArgs:  []types.Type{lookupType("C")},
				},
				Instance{
					Object: lookupObj("entry3.method"),
					TArgs:  []types.Type{lookupType("C")},
				},
				Instance{
					Object: lookupObj("entry3.method.x"),
					TNest:  []types.Type{lookupType("C")},
				},
				Instance{
					Object: lookupObj("entry3.method.y"),
					TNest:  []types.Type{lookupType("C")},
					TArgs:  []types.Type{types.Typ[types.Int8]},
				},
				Instance{
					Object: lookupObj("entry3.method.y"),
					TNest:  []types.Type{lookupType("C")},
					TArgs:  []types.Type{lookupType("C")},
				},
			),
		}, {
			descr: "generic type declaration",
			resolver: NewResolver(
				types.NewContext(),
				Instance{
					Object: lookupObj("entry3"),
					TArgs:  []types.Type{lookupType("D")},
				}),
			node: lookupDecl("entry3"),
			want: instancesInType(lookupType("D")),
		}, {
			descr:    "non-generic type declaration",
			resolver: nil,
			node:     lookupDecl("entry4"),
			want:     instancesInType(lookupType("E")),
		}, {
			descr:    "non-generic type alias",
			resolver: nil,
			node:     lookupDecl("entry5"),
			want: []Instance{
				{
					Object: lookupObj("typ"),
					TArgs:  []types.Type{types.Typ[types.Int], lookupType("F")},
				},
				{
					Object: lookupObj("typ.method"),
					TArgs:  []types.Type{types.Typ[types.Int], lookupType("F")},
				},
			},
		},
	}

	for _, test := range tests {
		t.Run(test.descr, func(t *testing.T) {
			v := visitor{
				instances: &PackageInstanceSets{},
				resolver:  test.resolver,
				info:      info,
			}
			if test.resolver != nil {
				// Since we know all the tests are for functions and methods,
				// set the nested type to the type parameter from the resolver.
				v.nestTParams = test.resolver.tParams
				v.nestTArgs = test.resolver.tArgs
			}
			ast.Walk(&v, test.node)
			got := v.instances.Pkg(pkg).Values()
			if diff := cmp.Diff(test.want, got, instanceOpts()); diff != "" {
				t.Errorf("Discovered instance diff (-want,+got):\n%s", diff)
			}
		})
	}
}

func TestSeedVisitor(t *testing.T) {
	src := `package test
	type typ[T any] int
	func (t typ[T]) method(arg T) { var x typ[string]; _ = x }
	func fun[T any](arg T) { var y typ[string]; _ = y }

	const a typ[int] = 1
	var b typ[int]
	type c struct { field typ[int8] }
	func (_ c) method() { var _ typ[int16] }
	type d = typ[int32]
	func e() { var _ typ[int64] }
	`

	f := srctesting.New(t)
	file := f.Parse("test.go", src)
	info, pkg := f.Check("pkg/test", file)

	sv := seedVisitor{
		visitor: visitor{
			instances: &PackageInstanceSets{},
			resolver:  nil,
			info:      info,
		},
		objMap: map[types.Object]ast.Node{},
	}
	ast.Walk(&sv, file)

	tInst := func(tArg types.Type) Instance {
		return Instance{
			Object: pkg.Scope().Lookup("typ"),
			TArgs:  []types.Type{tArg},
		}
	}
	mInst := func(tArg types.Type) Instance {
		return Instance{
			Object: srctesting.LookupObj(pkg, "typ.method"),
			TArgs:  []types.Type{tArg},
		}
	}
	want := []Instance{
		tInst(types.Typ[types.Int]),
		mInst(types.Typ[types.Int]),
		tInst(types.Typ[types.Int8]),
		mInst(types.Typ[types.Int8]),
		tInst(types.Typ[types.Int16]),
		mInst(types.Typ[types.Int16]),
		tInst(types.Typ[types.Int32]),
		mInst(types.Typ[types.Int32]),
		tInst(types.Typ[types.Int64]),
		mInst(types.Typ[types.Int64]),
	}
	got := sv.instances.Pkg(pkg).Values()
	if diff := cmp.Diff(want, got, instanceOpts()); diff != "" {
		t.Errorf("Instances from seedVisitor contain diff (-want,+got):\n%s", diff)
	}
}

func TestCollector(t *testing.T) {
	src := `package test
	type typ[T any] int
	func (t typ[T]) method(arg T) { var _ typ[int]; fun[int8](0) }
	func fun[T any](arg T) {
		var _ typ[int16]

		type nested[U any] struct{}
		_ = nested[T]{}
	}

	type ignore = int

	func a() {
		var _ typ[int32]
		fun[int64](0)
	}
	`

	f := srctesting.New(t)
	file := f.Parse("test.go", src)
	info, pkg := f.Check("pkg/test", file)

	c := Collector{
		TContext:  types.NewContext(),
		Instances: &PackageInstanceSets{},
	}
	c.Scan(info, pkg, file)
	c.Finish()

	inst := func(name, tNest, tArg string) Instance {
		return Instance{
			Object: srctesting.LookupObj(pkg, name),
			TNest:  evalTypeArgs(t, f.FileSet, pkg, tNest),
			TArgs:  evalTypeArgs(t, f.FileSet, pkg, tArg),
		}
	}
	want := []Instance{
		inst(`typ`, ``, `int`),
		inst(`typ.method`, ``, `int`),
		inst(`fun`, ``, `int8`),
		inst(`fun.nested`, `int8`, `int8`),
		inst(`typ`, ``, `int16`),
		inst(`typ.method`, ``, `int16`),
		inst(`typ`, ``, `int32`),
		inst(`typ.method`, ``, `int32`),
		inst(`fun`, ``, `int64`),
		inst(`fun.nested`, `int64`, `int64`),
	}
	got := c.Instances.Pkg(pkg).Values()
	if diff := cmp.Diff(want, got, instanceOpts()); diff != "" {
		t.Errorf("Instances from Collector contain diff (-want,+got):\n%s", diff)
	}
}

func TestCollector_MoreNesting(t *testing.T) {
	src := `package test

	func fun[T any]() {
		type nestedCon struct{ X T }
		_ = nestedCon{}

		type nestedGen[U any] struct{ Y T; Z U }
		_ = nestedGen[T]{}
		_ = nestedGen[int8]{}

		type nestedCover[T any] struct{ W T }
		_ = nestedCover[T]{}
		_ = nestedCover[int16]{}
	}

	func a() {
		fun[int32]()
		fun[int64]()
	}
	`

	f := srctesting.New(t)
	file := f.Parse(`test.go`, src)
	info, pkg := f.Check(`pkg/test`, file)

	c := Collector{
		TContext:  types.NewContext(),
		Instances: &PackageInstanceSets{},
	}
	c.Scan(info, pkg, file)
	c.Finish()

	inst := func(name, tNest, tArg string) Instance {
		return Instance{
			Object: srctesting.LookupObj(pkg, name),
			TNest:  evalTypeArgs(t, f.FileSet, pkg, tNest),
			TArgs:  evalTypeArgs(t, f.FileSet, pkg, tArg),
		}
	}
	want := []Instance{
		inst(`fun`, ``, `int32`),
		inst(`fun`, ``, `int64`),

		inst(`fun.nestedCon`, `int32`, ``),
		inst(`fun.nestedCon`, `int64`, ``),

		inst(`fun.nestedGen`, `int32`, `int32`),
		inst(`fun.nestedGen`, `int32`, `int8`),
		inst(`fun.nestedGen`, `int64`, `int64`),
		inst(`fun.nestedGen`, `int64`, `int8`),

		inst(`fun.nestedCover`, `int32`, `int32`),
		inst(`fun.nestedCover`, `int32`, `int16`),
		inst(`fun.nestedCover`, `int64`, `int64`),
		inst(`fun.nestedCover`, `int64`, `int16`),
	}
	got := c.Instances.Pkg(pkg).Values()
	if diff := cmp.Diff(want, got, instanceOpts()); diff != `` {
		t.Errorf("Instances from Collector contain diff (-want,+got):\n%s", diff)
	}
}

func TestCollector_NestingWithVars(t *testing.T) {
	// This is loosely based off of go1.19.13/test/typeparam/issue47740b.go
	// I was getting an error where `Q.print[int;]` was showing up when
	// `Q.print` is not in a nesting context with `int` and this helped debug
	// it. The problem was that `q` was being treated like a type not a var.
	src := `package test

	type Q struct{ v any }
	func (q Q) print() {
		println(q.v)
	}

	func newQ(v any) Q {
		return Q{v}
	}

	type S[T any] struct{ x T }
	func (s S[T]) echo() {
		q := newQ(s.x)
		q.print()
	}

	func a() {
		s := S[int]{x: 0}
		s.echo()
	}
	`

	f := srctesting.New(t)
	file := f.Parse(`test.go`, src)
	info, pkg := f.Check(`pkg/test`, file)
	inst := func(name, tArg string) Instance {
		return Instance{
			Object: srctesting.LookupObj(pkg, name),
			TArgs:  evalTypeArgs(t, f.FileSet, pkg, tArg),
		}
	}

	c := Collector{
		TContext:  types.NewContext(),
		Instances: &PackageInstanceSets{},
	}
	c.Scan(info, pkg, file)
	c.Finish()

	want := []Instance{
		inst(`S`, `int`),
		inst(`S.echo`, `int`),
	}
	got := c.Instances.Pkg(pkg).Values()
	if diff := cmp.Diff(want, got, instanceOpts()); diff != `` {
		t.Errorf("Instances from Collector contain diff (-want,+got):\n%s", diff)
	}
}

func TestCollector_RecursiveTypeParams(t *testing.T) {
	// This is based off of part of go1.19.13/test/typeparam/nested.go
	src := `package test
	func F[A any]() {}
	func main() {
		type U[_ any] int
		type X[A any] U[X[A]]
		F[X[int]]()
	}
	`

	f := srctesting.New(t)
	tc := types.NewContext()
	file := f.Parse(`test.go`, src)
	info, pkg := f.Check(`test`, file)
	inst := func(name, tNest, tArg string) Instance {
		return Instance{
			Object: srctesting.LookupObj(pkg, name),
			TNest:  evalTypeArgs(t, f.FileSet, pkg, tNest),
			TArgs:  evalTypeArgs(t, f.FileSet, pkg, tArg),
		}
	}

	c := Collector{
		TContext:  tc,
		Instances: &PackageInstanceSets{},
	}
	c.Scan(info, pkg, file)
	c.Finish()

	xInst := inst(`main.X`, ``, `int`)
	xInt := xInst.Resolve(tc)
	want := []Instance{
		xInst,
		{
			Object: srctesting.LookupObj(pkg, `F`),
			TArgs:  []types.Type{xInt},
		},
		{
			Object: srctesting.LookupObj(pkg, `main.U`),
			TArgs:  []types.Type{xInt},
		},
	}
	got := c.Instances.Pkg(pkg).Values()
	if diff := cmp.Diff(want, got, instanceOpts()); diff != `` {
		t.Errorf("Instances from Collector contain diff (-want,+got):\n%s", diff)
	}
}

func TestCollector_NestedRecursiveTypeParams(t *testing.T) {
	// This is based off of part of go1.19.13/test/typeparam/nested.go
	src := `package test
	func F[A any]() any {
		type U[_ any] struct{ x A }
		type X[B any] U[X[B]]
		return X[int]{}
	}
	func main() {
		print(F[string]())
	}
	`

	f := srctesting.New(t)
	tc := types.NewContext()
	file := f.Parse(`test.go`, src)
	info, pkg := f.Check(`test`, file)
	inst := func(name, tNest, tArg string) Instance {
		return Instance{
			Object: srctesting.LookupObj(pkg, name),
			TNest:  evalTypeArgs(t, f.FileSet, pkg, tNest),
			TArgs:  evalTypeArgs(t, f.FileSet, pkg, tArg),
		}
	}

	c := Collector{
		TContext:  tc,
		Instances: &PackageInstanceSets{},
	}
	c.Scan(info, pkg, file)
	c.Finish()

	xInst := inst(`F.X`, `string`, `int`)
	xInt := xInst.Resolve(tc)
	want := []Instance{
		inst(`F`, ``, `string`),
		xInst,
		{
			Object: srctesting.LookupObj(pkg, `F.U`),
			TNest:  []types.Type{types.Typ[types.String]},
			TArgs:  []types.Type{xInt},
		},
	}
	got := c.Instances.Pkg(pkg).Values()
	if diff := cmp.Diff(want, got, instanceOpts()); diff != `` {
		t.Errorf("Instances from Collector contain diff (-want,+got):\n%s", diff)
	}
}

func TestCollector_LooselyRecursiveTypeParams(t *testing.T) {
	// This is based off of part of go1.19.13/test/typeparam/nested.go
	src := `package test
	func main() {
		type U[B any] struct{ y *B }
		type X[C any] struct{ p U[X[C]] }
		print(X[int]{})
	}
	`

	f := srctesting.New(t)
	tc := types.NewContext()
	file := f.Parse(`test.go`, src)
	info, pkg := f.Check(`test`, file)

	c := Collector{
		TContext:  tc,
		Instances: &PackageInstanceSets{},
	}
	c.Scan(info, pkg, file)
	c.Finish()

	xInst := Instance{
		Object: srctesting.LookupObj(pkg, `main.X`),
		TArgs:  []types.Type{types.Typ[types.Int]},
	}
	xInt := xInst.Resolve(tc)
	want := []Instance{
		xInst,
		{
			Object: srctesting.LookupObj(pkg, `main.U`),
			TArgs:  []types.Type{xInt},
		},
	}
	got := c.Instances.Pkg(pkg).Values()
	if diff := cmp.Diff(want, got, instanceOpts()); diff != `` {
		t.Errorf("Instances from Collector contain diff (-want,+got):\n%s", diff)
	}
}

func TestCollector_NestedTypeParams(t *testing.T) {
	// This is based off of part of go1.19.13/test/typeparam/nested.go
	src := `package test
	func F[A any]() any {
		type T[B any] struct{}
		type U[_ any] struct{ X A }
		return T[U[bool]]{}
	}
	func main() {
		print(F[int]())
	}
	`

	f := srctesting.New(t)
	tc := types.NewContext()
	file := f.Parse(`test.go`, src)
	info, pkg := f.Check(`test`, file)
	inst := func(name, tNest, tArg string) Instance {
		return Instance{
			Object: srctesting.LookupObj(pkg, name),
			TNest:  evalTypeArgs(t, f.FileSet, pkg, tNest),
			TArgs:  evalTypeArgs(t, f.FileSet, pkg, tArg),
		}
	}

	c := Collector{
		TContext:  tc,
		Instances: &PackageInstanceSets{},
	}
	c.Scan(info, pkg, file)
	c.Finish()

	uInst := inst(`F.U`, `int`, `bool`)
	uIntBool := uInst.Resolve(tc)
	want := []Instance{
		inst(`F`, ``, `int`),
		inst(`F.U`, `int`, `bool`),
		{
			Object: srctesting.LookupObj(pkg, `F.T`),
			TNest:  []types.Type{types.Typ[types.Int]},
			TArgs:  []types.Type{uIntBool},
		},
	}
	got := c.Instances.Pkg(pkg).Values()
	if diff := cmp.Diff(want, got, instanceOpts()); diff != `` {
		t.Errorf("Instances from Collector contain diff (-want,+got):\n%s", diff)
	}
}

func TestCollector_ImplicitTypeInstance(t *testing.T) {
	f := srctesting.New(t)
	tc := types.NewContext()
	inst := func(pkg *types.Package, name, tArg string) Instance {
		return Instance{
			Object: srctesting.LookupObj(pkg, name),
			TArgs:  evalTypeArgs(t, f.FileSet, pkg, tArg),
		}
	}

	fooSrc := `package foo
	type Foo[T any] struct{}
	func NewFoo[T any]() *Foo[T] {
		return &Foo[T]{}
	}`
	fooFile := f.Parse(`foo/foo.go`, fooSrc)
	_, fooPkg := f.Check(`foo`, fooFile)

	mainSrc := `package test
	import "foo"
	func main() {
		print(foo.NewFoo[int]())
	}`
	mainFile := f.Parse(`main.go`, mainSrc)
	_, mainPkg := f.Check(`test`, mainFile)

	c := Collector{
		TContext:  tc,
		Instances: &PackageInstanceSets{},
	}
	// The issue which caused this test to be written only occurred when
	// fooFile was scanned before mainFile, otherwise it would work fine.
	// The problem was that `foo.Foo[int]` was not being collected in this
	// order because mainFile adds an instance into the instance set for "foo"
	// after "foo" was already propagated. This was fixed by performing
	// propagation after all the packages are scanned via `Finish`.
	c.Scan(f.Info, fooPkg, fooFile)
	c.Scan(f.Info, mainPkg, mainFile)
	c.Finish()

	want := []Instance{
		inst(fooPkg, `NewFoo`, `int`),
		inst(fooPkg, `Foo`, `int`),
	}
	got := c.Instances.Pkg(fooPkg).Values()
	if diff := cmp.Diff(want, got, instanceOpts()); diff != `` {
		t.Errorf("Instances from Collector contain diff (-want,+got):\n%s", diff)
	}
}

func TestCollector_MoreImplicitTypeInstance(t *testing.T) {
	f := srctesting.New(t)
	tc := types.NewContext()
	inst := func(pkg *types.Package, name, tArg string) Instance {
		return Instance{
			Object: srctesting.LookupObj(pkg, name),
			TArgs:  evalTypeArgs(t, f.FileSet, pkg, tArg),
		}
	}

	fooSrc := `package foo
	type Foo[T, U any] struct{}
	func NewFoo[T, U any]() *Foo[T, U] {
		return &Foo[T, U]{}
	}`
	fooFile := f.Parse(`foo/foo.go`, fooSrc)
	_, fooPkg := f.Check(`foo`, fooFile)

	barSrc := `package bar
	import "foo"
	func Bar[T, U any](f *foo.Foo[T, U]) *foo.Foo[U, T] {
		return foo.NewFoo[U, T]()
	}`
	barFile := f.Parse(`bar/bar.go`, barSrc)
	_, barPkg := f.Check(`bar`, barFile)

	mainSrc := `package test
	import "foo"
	import "bar"
	func main() {
		f := foo.NewFoo[int, string]()
		print(bar.Bar[int, string](f))
	}`
	mainFile := f.Parse(`main.go`, mainSrc)
	_, mainPkg := f.Check(`test`, mainFile)

	want := []Instance{
		inst(fooPkg, `NewFoo`, `int, string`),
		inst(fooPkg, `NewFoo`, `string, int`),
		inst(fooPkg, `Foo`, `int, string`),
		inst(fooPkg, `Foo`, `string, int`),
	}
	trial := func(order ...int) {
		c := Collector{
			TContext:  tc,
			Instances: &PackageInstanceSets{},
		}
		for _, i := range order {
			switch i {
			case 1:
				c.Scan(f.Info, fooPkg, fooFile)
			case 2:
				c.Scan(f.Info, barPkg, barFile)
			case 3:
				c.Scan(f.Info, mainPkg, mainFile)
			}
		}
		c.Finish()

		got := c.Instances.Pkg(fooPkg).Values()
		if diff := cmp.Diff(want, got, instanceOpts()); diff != `` {
			t.Errorf("Instances from Collector trial %v contain diff (-want,+got):\n%s", order, diff)
		}
	}

	trial(1, 2, 3)
	trial(1, 3, 2)
	trial(2, 1, 3)
	trial(2, 3, 1)
	trial(3, 1, 2)
	trial(3, 2, 1)
}

func evalTypeArgs(t *testing.T, fSet *token.FileSet, pkg *types.Package, expr string) []types.Type {
	if len(expr) == 0 {
		return nil
	}
	args := strings.Split(expr, ",")
	targs := make([]types.Type, 0, len(args))
	for _, astr := range args {
		tv, err := types.Eval(fSet, pkg, 0, astr)
		if err != nil {
			t.Fatalf("Eval(%s) failed: %v", astr, err)
		}
		targs = append(targs, tv.Type)
	}
	return targs
}

func TestCollector_CrossPackage(t *testing.T) {
	f := srctesting.New(t)
	const src = `package foo
	type X[T any] struct {Value T}

	func F[G any](g G) {
		x := X[G]{}
		println(x)
	}

	func DoFoo() {
		F(int8(8))
	}
	`
	fooFile := f.Parse("foo.go", src)
	_, fooPkg := f.Check("pkg/foo", fooFile)

	const src2 = `package bar
	import "pkg/foo"
	func FProxy[T any](t T) {
		foo.F[T](t)
	}
	func DoBar() {
		FProxy(int16(16))
	}
	`
	barFile := f.Parse("bar.go", src2)
	_, barPkg := f.Check("pkg/bar", barFile)

	c := Collector{
		TContext:  types.NewContext(),
		Instances: &PackageInstanceSets{},
	}
	c.Scan(f.Info, barPkg, barFile)
	c.Scan(f.Info, fooPkg, fooFile)
	c.Finish()

	inst := func(pkg *types.Package, name string, tArg types.BasicKind) Instance {
		return Instance{
			Object: srctesting.LookupObj(pkg, name),
			TArgs:  []types.Type{types.Typ[tArg]},
		}
	}

	wantFooInstances := []Instance{
		inst(fooPkg, "F", types.Int16), // Found in "pkg/foo".
		inst(fooPkg, "F", types.Int8),
		inst(fooPkg, "X", types.Int16), // Found due to F[int16] found in "pkg/foo".
		inst(fooPkg, "X", types.Int8),
	}
	gotFooInstances := c.Instances.Pkg(fooPkg).Values()
	if diff := cmp.Diff(wantFooInstances, gotFooInstances, instanceOpts()); diff != "" {
		t.Errorf("Instances from pkg/foo contain diff (-want,+got):\n%s", diff)
	}

	wantBarInstances := []Instance{
		inst(barPkg, "FProxy", types.Int16),
	}
	gotBarInstances := c.Instances.Pkg(barPkg).Values()
	if diff := cmp.Diff(wantBarInstances, gotBarInstances, instanceOpts()); diff != "" {
		t.Errorf("Instances from pkg/foo contain diff (-want,+got):\n%s", diff)
	}
}

func TestResolver_SubstituteSelection(t *testing.T) {
	tests := []struct {
		descr   string
		src     string
		wantObj string
		wantSig string
	}{{
		descr: "type parameter method",
		src: `package test
		type stringer interface{ String() string }

		type x struct{}
		func (_ x) String() string { return "" }

		type g[T stringer] struct{}
		func (_ g[T]) Method(t T) string {
			return t.String()
		}`,
		wantObj: "func (pkg/test.x).String() string",
		wantSig: "func() string",
	}, {
		descr: "generic receiver type with type parameter",
		src: `package test
			type x struct{}

			type g[T any] struct{}
			func (_ g[T]) Method(t T) string {
				return g[T]{}.Method(t)
			}`,
		wantObj: "func (pkg/test.g[pkg/test.x]).Method(t pkg/test.x) string",
		wantSig: "func(t pkg/test.x) string",
	}, {
		descr: "method expression",
		src: `package test
				type x struct{}

				type g[T any] struct{}
				func (recv g[T]) Method(t T) string {
					return g[T].Method(recv, t)
				}`,
		wantObj: "func (pkg/test.g[pkg/test.x]).Method(t pkg/test.x) string",
		wantSig: "func(recv pkg/test.g[pkg/test.x], t pkg/test.x) string",
	}}

	for _, test := range tests {
		t.Run(test.descr, func(t *testing.T) {
			f := srctesting.New(t)
			file := f.Parse("test.go", test.src)
			info, pkg := f.Check("pkg/test", file)

			resolver := NewResolver(nil, Instance{
				Object: srctesting.LookupObj(pkg, "g.Method"),
				TArgs:  []types.Type{srctesting.LookupObj(pkg, "x").Type()},
			})

			if l := len(info.Selections); l != 1 {
				t.Fatalf("Got: %d selections. Want: 1", l)
			}
			for _, sel := range info.Selections {
				gotObj := types.ObjectString(resolver.SubstituteSelection(sel).Obj(), nil)
				if gotObj != test.wantObj {
					t.Fatalf("Got: resolver.SubstituteSelection().Obj() = %q. Want: %q.", gotObj, test.wantObj)
				}
				gotSig := types.TypeString(resolver.SubstituteSelection(sel).Type(), nil)
				if gotSig != test.wantSig {
					t.Fatalf("Got: resolver.SubstituteSelection().Type() = %q. Want: %q.", gotSig, test.wantSig)
				}
			}
		})
	}
}
