/*
 * Decompiled with CFR 0.152.
 */
package com.google.googlejavaformat.java.javadoc;

import com.google.common.base.CharMatcher;
import com.google.common.base.Preconditions;
import com.google.common.base.Verify;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.Iterators;
import com.google.common.collect.PeekingIterator;
import com.google.googlejavaformat.java.javadoc.CharStream;
import com.google.googlejavaformat.java.javadoc.NestingCounter;
import com.google.googlejavaformat.java.javadoc.Token;
import java.util.ArrayDeque;
import java.util.ArrayList;
import java.util.List;
import java.util.regex.Pattern;

final class JavadocLexer {
    private static final Pattern NON_UNIX_LINE_ENDING = Pattern.compile("\r\n?");
    private final CharStream input;
    private final NestingCounter braceDepth = new NestingCounter();
    private final NestingCounter preDepth = new NestingCounter();
    private final NestingCounter codeDepth = new NestingCounter();
    private final NestingCounter tableDepth = new NestingCounter();
    private boolean outerInlineTagIsSnippet;
    private boolean somethingSinceNewline;
    private static final CharMatcher NEWLINE = CharMatcher.is('\n');
    private static final Pattern NEWLINE_PATTERN = Pattern.compile("^[ \t]*\n[ \t]*[*]?[ \t]?");
    private static final Pattern FOOTER_TAG_PATTERN = Pattern.compile("^@(param\\s+<\\w+>|[a-z]\\w*)");
    private static final Pattern MOE_BEGIN_STRIP_COMMENT_PATTERN = Pattern.compile("^<!--\\s*MOE:begin_intracomment_strip\\s*-->");
    private static final Pattern MOE_END_STRIP_COMMENT_PATTERN = Pattern.compile("^<!--\\s*MOE:end_intracomment_strip\\s*-->");
    private static final Pattern HTML_COMMENT_PATTERN = JavadocLexer.fullCommentPattern();
    private static final Pattern PRE_OPEN_PATTERN = JavadocLexer.openTagPattern("pre");
    private static final Pattern PRE_CLOSE_PATTERN = JavadocLexer.closeTagPattern("pre");
    private static final Pattern CODE_OPEN_PATTERN = JavadocLexer.openTagPattern("code");
    private static final Pattern CODE_CLOSE_PATTERN = JavadocLexer.closeTagPattern("code");
    private static final Pattern TABLE_OPEN_PATTERN = JavadocLexer.openTagPattern("table");
    private static final Pattern TABLE_CLOSE_PATTERN = JavadocLexer.closeTagPattern("table");
    private static final Pattern LIST_OPEN_PATTERN = JavadocLexer.openTagPattern("ul|ol|dl");
    private static final Pattern LIST_CLOSE_PATTERN = JavadocLexer.closeTagPattern("ul|ol|dl");
    private static final Pattern LIST_ITEM_OPEN_PATTERN = JavadocLexer.openTagPattern("li|dt|dd");
    private static final Pattern LIST_ITEM_CLOSE_PATTERN = JavadocLexer.closeTagPattern("li|dt|dd");
    private static final Pattern HEADER_OPEN_PATTERN = JavadocLexer.openTagPattern("h[1-6]");
    private static final Pattern HEADER_CLOSE_PATTERN = JavadocLexer.closeTagPattern("h[1-6]");
    private static final Pattern PARAGRAPH_OPEN_PATTERN = JavadocLexer.openTagPattern("p");
    private static final Pattern PARAGRAPH_CLOSE_PATTERN = JavadocLexer.closeTagPattern("p");
    private static final Pattern BLOCKQUOTE_OPEN_PATTERN = JavadocLexer.openTagPattern("blockquote");
    private static final Pattern BLOCKQUOTE_CLOSE_PATTERN = JavadocLexer.closeTagPattern("blockquote");
    private static final Pattern BR_PATTERN = JavadocLexer.openTagPattern("br");
    private static final Pattern SNIPPET_TAG_OPEN_PATTERN = Pattern.compile("^[{]@snippet\\b");
    private static final Pattern INLINE_TAG_OPEN_PATTERN = Pattern.compile("^[{]@\\w*");
    private static final Pattern LITERAL_PATTERN = Pattern.compile("^.[^ \t\n@<{}*]*", 32);

    static ImmutableList<Token> lex(String input) throws LexException {
        input = JavadocLexer.stripJavadocBeginAndEnd(input);
        input = JavadocLexer.normalizeLineEndings(input);
        return new JavadocLexer(new CharStream(input)).generateTokens();
    }

    private static String normalizeLineEndings(String input) {
        return NON_UNIX_LINE_ENDING.matcher(input).replaceAll("\n");
    }

    private static String stripJavadocBeginAndEnd(String input) {
        Preconditions.checkArgument(input.startsWith("/**"), "Missing /**: %s", (Object)input);
        Preconditions.checkArgument(input.endsWith("*/") && input.length() > 4, "Missing */: %s", (Object)input);
        return input.substring("/**".length(), input.length() - "*/".length());
    }

    private JavadocLexer(CharStream input) {
        this.input = Preconditions.checkNotNull(input);
    }

    private ImmutableList<Token> generateTokens() throws LexException {
        ImmutableList.Builder tokens = ImmutableList.builder();
        Token token = new Token(Token.Type.BEGIN_JAVADOC, "/**");
        tokens.add(token);
        while (!this.input.isExhausted()) {
            token = this.readToken();
            tokens.add(token);
        }
        this.checkMatchingTags();
        token = new Token(Token.Type.END_JAVADOC, "*/");
        tokens.add(token);
        ImmutableList<Token> result = tokens.build();
        result = JavadocLexer.joinAdjacentLiteralsAndAdjacentWhitespace((List<Token>)result);
        result = JavadocLexer.inferParagraphTags(result);
        result = JavadocLexer.optionalizeSpacesAfterLinks(result);
        result = JavadocLexer.deindentPreCodeBlocks(result);
        return result;
    }

    private Token readToken() throws LexException {
        Token.Type type = this.consumeToken();
        String value = this.input.readAndResetRecorded();
        return new Token(type, value);
    }

    private Token.Type consumeToken() throws LexException {
        boolean preserveExistingFormatting = this.preserveExistingFormatting();
        if (this.input.tryConsumeRegex(NEWLINE_PATTERN)) {
            this.somethingSinceNewline = false;
            return preserveExistingFormatting ? Token.Type.FORCED_NEWLINE : Token.Type.WHITESPACE;
        }
        if (this.input.tryConsume(" ") || this.input.tryConsume("\t")) {
            return preserveExistingFormatting ? Token.Type.LITERAL : Token.Type.WHITESPACE;
        }
        if (!this.somethingSinceNewline && this.input.tryConsumeRegex(FOOTER_TAG_PATTERN)) {
            this.checkMatchingTags();
            this.somethingSinceNewline = true;
            return Token.Type.FOOTER_JAVADOC_TAG_START;
        }
        this.somethingSinceNewline = true;
        if (this.input.tryConsumeRegex(SNIPPET_TAG_OPEN_PATTERN)) {
            if (this.braceDepth.value() == 0) {
                this.braceDepth.increment();
                this.outerInlineTagIsSnippet = true;
                return Token.Type.SNIPPET_BEGIN;
            }
            this.braceDepth.increment();
            return Token.Type.LITERAL;
        }
        if (this.input.tryConsumeRegex(INLINE_TAG_OPEN_PATTERN)) {
            this.braceDepth.increment();
            return Token.Type.LITERAL;
        }
        if (this.input.tryConsume("{")) {
            this.braceDepth.incrementIfPositive();
            return Token.Type.LITERAL;
        }
        if (this.input.tryConsume("}")) {
            if (this.outerInlineTagIsSnippet && this.braceDepth.value() == 1) {
                this.braceDepth.decrementIfPositive();
                this.outerInlineTagIsSnippet = false;
                return Token.Type.SNIPPET_END;
            }
            this.braceDepth.decrementIfPositive();
            return Token.Type.LITERAL;
        }
        if (this.braceDepth.isPositive()) {
            Verify.verify(this.input.tryConsumeRegex(LITERAL_PATTERN));
            return Token.Type.LITERAL;
        }
        if (this.input.tryConsumeRegex(PRE_OPEN_PATTERN)) {
            this.preDepth.increment();
            return preserveExistingFormatting ? Token.Type.LITERAL : Token.Type.PRE_OPEN_TAG;
        }
        if (this.input.tryConsumeRegex(PRE_CLOSE_PATTERN)) {
            this.preDepth.decrementIfPositive();
            return this.preserveExistingFormatting() ? Token.Type.LITERAL : Token.Type.PRE_CLOSE_TAG;
        }
        if (this.input.tryConsumeRegex(CODE_OPEN_PATTERN)) {
            this.codeDepth.increment();
            return preserveExistingFormatting ? Token.Type.LITERAL : Token.Type.CODE_OPEN_TAG;
        }
        if (this.input.tryConsumeRegex(CODE_CLOSE_PATTERN)) {
            this.codeDepth.decrementIfPositive();
            return this.preserveExistingFormatting() ? Token.Type.LITERAL : Token.Type.CODE_CLOSE_TAG;
        }
        if (this.input.tryConsumeRegex(TABLE_OPEN_PATTERN)) {
            this.tableDepth.increment();
            return preserveExistingFormatting ? Token.Type.LITERAL : Token.Type.TABLE_OPEN_TAG;
        }
        if (this.input.tryConsumeRegex(TABLE_CLOSE_PATTERN)) {
            this.tableDepth.decrementIfPositive();
            return this.preserveExistingFormatting() ? Token.Type.LITERAL : Token.Type.TABLE_CLOSE_TAG;
        }
        if (preserveExistingFormatting) {
            Verify.verify(this.input.tryConsumeRegex(LITERAL_PATTERN));
            return Token.Type.LITERAL;
        }
        if (this.input.tryConsumeRegex(PARAGRAPH_OPEN_PATTERN)) {
            return Token.Type.PARAGRAPH_OPEN_TAG;
        }
        if (this.input.tryConsumeRegex(PARAGRAPH_CLOSE_PATTERN)) {
            return Token.Type.PARAGRAPH_CLOSE_TAG;
        }
        if (this.input.tryConsumeRegex(LIST_OPEN_PATTERN)) {
            return Token.Type.LIST_OPEN_TAG;
        }
        if (this.input.tryConsumeRegex(LIST_CLOSE_PATTERN)) {
            return Token.Type.LIST_CLOSE_TAG;
        }
        if (this.input.tryConsumeRegex(LIST_ITEM_OPEN_PATTERN)) {
            return Token.Type.LIST_ITEM_OPEN_TAG;
        }
        if (this.input.tryConsumeRegex(LIST_ITEM_CLOSE_PATTERN)) {
            return Token.Type.LIST_ITEM_CLOSE_TAG;
        }
        if (this.input.tryConsumeRegex(BLOCKQUOTE_OPEN_PATTERN)) {
            return Token.Type.BLOCKQUOTE_OPEN_TAG;
        }
        if (this.input.tryConsumeRegex(BLOCKQUOTE_CLOSE_PATTERN)) {
            return Token.Type.BLOCKQUOTE_CLOSE_TAG;
        }
        if (this.input.tryConsumeRegex(HEADER_OPEN_PATTERN)) {
            return Token.Type.HEADER_OPEN_TAG;
        }
        if (this.input.tryConsumeRegex(HEADER_CLOSE_PATTERN)) {
            return Token.Type.HEADER_CLOSE_TAG;
        }
        if (this.input.tryConsumeRegex(BR_PATTERN)) {
            return Token.Type.BR_TAG;
        }
        if (this.input.tryConsumeRegex(MOE_BEGIN_STRIP_COMMENT_PATTERN)) {
            return Token.Type.MOE_BEGIN_STRIP_COMMENT;
        }
        if (this.input.tryConsumeRegex(MOE_END_STRIP_COMMENT_PATTERN)) {
            return Token.Type.MOE_END_STRIP_COMMENT;
        }
        if (this.input.tryConsumeRegex(HTML_COMMENT_PATTERN)) {
            return Token.Type.HTML_COMMENT;
        }
        if (this.input.tryConsumeRegex(LITERAL_PATTERN)) {
            return Token.Type.LITERAL;
        }
        throw new AssertionError();
    }

    private boolean preserveExistingFormatting() {
        return this.preDepth.isPositive() || this.tableDepth.isPositive() || this.codeDepth.isPositive() || this.outerInlineTagIsSnippet;
    }

    private void checkMatchingTags() throws LexException {
        if (this.braceDepth.isPositive() || this.preDepth.isPositive() || this.tableDepth.isPositive() || this.codeDepth.isPositive()) {
            throw new LexException();
        }
    }

    private static ImmutableList<Token> joinAdjacentLiteralsAndAdjacentWhitespace(List<Token> input) {
        ImmutableList.Builder output = ImmutableList.builder();
        StringBuilder accumulated = new StringBuilder();
        PeekingIterator<Token> tokens = Iterators.peekingIterator(input.iterator());
        while (tokens.hasNext()) {
            if (tokens.peek().getType() == Token.Type.LITERAL) {
                accumulated.append(tokens.peek().getValue());
                tokens.next();
                continue;
            }
            if (accumulated.length() == 0) {
                output.add(tokens.peek());
                tokens.next();
                continue;
            }
            StringBuilder seenWhitespace = new StringBuilder();
            while (tokens.peek().getType() == Token.Type.WHITESPACE) {
                seenWhitespace.append(tokens.next().getValue());
            }
            if (tokens.peek().getType() == Token.Type.LITERAL && tokens.peek().getValue().startsWith("@")) {
                accumulated.append(" ");
                accumulated.append(tokens.peek().getValue());
                tokens.next();
                continue;
            }
            output.add(new Token(Token.Type.LITERAL, accumulated.toString()));
            accumulated.setLength(0);
            if (seenWhitespace.length() <= 0) continue;
            output.add(new Token(Token.Type.WHITESPACE, seenWhitespace.toString()));
        }
        return output.build();
    }

    private static ImmutableList<Token> inferParagraphTags(List<Token> input) {
        ImmutableList.Builder output = ImmutableList.builder();
        PeekingIterator<Token> tokens = Iterators.peekingIterator(input.iterator());
        while (tokens.hasNext()) {
            if (tokens.peek().getType() == Token.Type.LITERAL) {
                output.add(tokens.next());
                if (tokens.peek().getType() != Token.Type.WHITESPACE || !JavadocLexer.hasMultipleNewlines(tokens.peek().getValue())) continue;
                output.add(tokens.next());
                if (tokens.peek().getType() != Token.Type.LITERAL) continue;
                output.add(new Token(Token.Type.PARAGRAPH_OPEN_TAG, "<p>"));
                continue;
            }
            output.add(tokens.next());
        }
        return output.build();
    }

    private static ImmutableList<Token> optionalizeSpacesAfterLinks(List<Token> input) {
        ImmutableList.Builder output = ImmutableList.builder();
        PeekingIterator<Token> tokens = Iterators.peekingIterator(input.iterator());
        while (tokens.hasNext()) {
            if (tokens.peek().getType() == Token.Type.LITERAL && tokens.peek().getValue().matches("^href=[^>]*>")) {
                output.add(tokens.next());
                if (tokens.peek().getType() != Token.Type.WHITESPACE) continue;
                output.add(new Token(Token.Type.OPTIONAL_LINE_BREAK, tokens.next().getValue()));
                continue;
            }
            output.add(tokens.next());
        }
        return output.build();
    }

    private static ImmutableList<Token> deindentPreCodeBlocks(List<Token> input) {
        ImmutableList.Builder<Token> output = ImmutableList.builder();
        PeekingIterator<Token> tokens = Iterators.peekingIterator(input.iterator());
        while (tokens.hasNext()) {
            if (tokens.peek().getType() != Token.Type.PRE_OPEN_TAG) {
                output.add((Object)tokens.next());
                continue;
            }
            output.add((Object)tokens.next());
            ArrayList<Token> initialNewlines = new ArrayList<Token>();
            while (tokens.hasNext() && tokens.peek().getType() == Token.Type.FORCED_NEWLINE) {
                initialNewlines.add(tokens.next());
            }
            if (tokens.peek().getType() != Token.Type.LITERAL || !tokens.peek().getValue().matches("[ \t]*[{]@code")) {
                output.addAll(initialNewlines);
                output.add((Object)tokens.next());
                continue;
            }
            JavadocLexer.deindentPreCodeBlock(output, tokens);
        }
        return output.build();
    }

    private static void deindentPreCodeBlock(ImmutableList.Builder<Token> output, PeekingIterator<Token> tokens) {
        ArrayDeque<Token> saved = new ArrayDeque<Token>();
        output.add((Object)new Token(Token.Type.LITERAL, tokens.next().getValue().trim()));
        while (tokens.hasNext() && tokens.peek().getType() != Token.Type.PRE_CLOSE_TAG) {
            Token token = tokens.next();
            saved.addLast(token);
        }
        while (!saved.isEmpty() && ((Token)saved.peekFirst()).getType() == Token.Type.FORCED_NEWLINE) {
            saved.removeFirst();
        }
        while (!saved.isEmpty() && ((Token)saved.peekLast()).getType() == Token.Type.FORCED_NEWLINE) {
            saved.removeLast();
        }
        if (saved.isEmpty()) {
            return;
        }
        Token last = (Token)saved.peekLast();
        boolean trailingBrace = false;
        if (last.getType() == Token.Type.LITERAL && last.getValue().endsWith("}")) {
            saved.removeLast();
            if (last.length() > 1) {
                saved.addLast(new Token(Token.Type.LITERAL, last.getValue().substring(0, last.getValue().length() - 1)));
                saved.addLast(new Token(Token.Type.FORCED_NEWLINE, null));
            }
            trailingBrace = true;
        }
        int trim = -1;
        for (Token token : saved) {
            int idx;
            if (token.getType() != Token.Type.LITERAL || (idx = CharMatcher.isNot(' ').indexIn(token.getValue())) == -1 || trim != -1 && idx >= trim) continue;
            trim = idx;
        }
        output.add((Object)new Token(Token.Type.FORCED_NEWLINE, "\n"));
        for (Token token : saved) {
            if (token.getType() == Token.Type.LITERAL) {
                output.add((Object)new Token(Token.Type.LITERAL, trim > 0 && token.length() > trim ? token.getValue().substring(trim) : token.getValue()));
                continue;
            }
            output.add((Object)token);
        }
        if (trailingBrace) {
            output.add((Object)new Token(Token.Type.LITERAL, "}"));
        } else {
            output.add((Object)new Token(Token.Type.FORCED_NEWLINE, "\n"));
        }
    }

    private static boolean hasMultipleNewlines(String s2) {
        return NEWLINE.countIn(s2) > 1;
    }

    private static Pattern fullCommentPattern() {
        return Pattern.compile("^<!--.*?-->", 32);
    }

    private static Pattern openTagPattern(String namePattern) {
        return Pattern.compile(String.format("^<(?:%s)\\b[^>]*>", namePattern), 2);
    }

    private static Pattern closeTagPattern(String namePattern) {
        return Pattern.compile(String.format("^</(?:%s)\\b[^>]*>", namePattern), 2);
    }

    static class LexException
    extends Exception {
        LexException() {
        }
    }
}

