// Copyright (c) 2021 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

// This header contains interfaces that abstract away different backing
// protocols for WebTransport.

#ifndef QUICHE_QUIC_CORE_WEB_TRANSPORT_INTERFACE_H_
#define QUICHE_QUIC_CORE_WEB_TRANSPORT_INTERFACE_H_

#include "absl/strings/str_cat.h"
#include "quiche/quic/core/quic_types.h"
#include "quiche/web_transport/web_transport.h"

namespace quic {

using WebTransportSessionError = webtransport::SessionErrorCode;
using WebTransportStreamError = webtransport::StreamErrorCode;

using WebTransportStreamVisitor = webtransport::StreamVisitor;
using WebTransportStream = webtransport::Stream;
using WebTransportVisitor = webtransport::SessionVisitor;
using WebTransportSession = webtransport::Session;

inline webtransport::DatagramStatus DatagramStatusToWebTransportStatus(
    DatagramStatus status) {
  switch (status) {
    case DATAGRAM_STATUS_SUCCESS:
      return webtransport::DatagramStatus(
          webtransport::DatagramStatusCode::kSuccess, "");
    case DATAGRAM_STATUS_BLOCKED:
      return webtransport::DatagramStatus(
          webtransport::DatagramStatusCode::kBlocked,
          "QUIC connection write-blocked");
    case DATAGRAM_STATUS_TOO_LARGE:
      return webtransport::DatagramStatus(
          webtransport::DatagramStatusCode::kTooBig,
          "Datagram payload exceeded maximum allowed size");
    case DATAGRAM_STATUS_ENCRYPTION_NOT_ESTABLISHED:
    case DATAGRAM_STATUS_INTERNAL_ERROR:
    case DATAGRAM_STATUS_UNSUPPORTED:
    case DATAGRAM_STATUS_SETTINGS_NOT_RECEIVED:
      return webtransport::DatagramStatus(
          webtransport::DatagramStatusCode::kInternalError,
          absl::StrCat("Internal error: ", DatagramStatusToString(status)));
    default:
      return webtransport::DatagramStatus(
          webtransport::DatagramStatusCode::kInternalError,
          absl::StrCat("Unknown status: ", DatagramStatusToString(status)));
  }
}

}  // namespace quic

#endif  // QUICHE_QUIC_CORE_WEB_TRANSPORT_INTERFACE_H_
