// Copyright 2019 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include <optional>
#include <string>
#include <utility>

#include "quiche/common/platform/api/quiche_test.h"
#include "quiche/common/structured_headers.h"

// This file contains tests cases for the Structured Header parser and
// serializer, taken from the public test case repository at
// https://github.com/httpwg/structured-field-tests. All of the tests are named,
// so a given test case can be found in the JSON files in that repository by
// searching for the test name. This file is generated, with the test cases
// being automatically translated from the JSON source to C++ unit tests. Please
// do not modify, as the contents will be overwritten when this is re-generated.

// Generated on 2022-03-15 from structured-field-tests.git @
// faed1f92942abd4fb5d61b1f9f0dc359f499f1d7.

namespace quiche {
namespace structured_headers {
namespace {

// Helpers to make test cases clearer

Item Integer(int64_t value) { return Item(value); }

std::pair<std::string, Item> BooleanParam(std::string key, bool value) {
  return std::make_pair(key, Item(value));
}

std::pair<std::string, Item> DoubleParam(std::string key, double value) {
  return std::make_pair(key, Item(value));
}

std::pair<std::string, Item> Param(std::string key, int64_t value) {
  return std::make_pair(key, Item(value));
}

std::pair<std::string, Item> Param(std::string key, std::string value) {
  return std::make_pair(key, Item(value));
}

std::pair<std::string, Item> TokenParam(std::string key, std::string value) {
  return std::make_pair(key, Item(value, Item::kTokenType));
}

const struct ParameterizedItemTestCase {
  const char* name;
  const char* raw;
  size_t raw_len;
  const std::optional<ParameterizedItem>
      expected;           // nullopt if parse error is expected.
  const char* canonical;  // nullptr if parse error is expected, or if canonical
                          // format is identical to raw.
} parameterized_item_test_cases[] = {
    // binary.json
    {"basic binary",
     ":aGVsbG8=:",
     10,
     {{Item("hello", Item::kByteSequenceType), {}}},
     nullptr},
    {"empty binary",
     "::",
     2,
     {{Item("", Item::kByteSequenceType), {}}},
     nullptr},
    {"bad paddding",
     ":aGVsbG8:",
     9,
     {{Item("hello", Item::kByteSequenceType), {}}},
     ":aGVsbG8=:"},
    {"bad end delimiter", ":aGVsbG8=", 9, std::nullopt, nullptr},
    {"extra whitespace", ":aGVsb G8=:", 11, std::nullopt, nullptr},
    {"extra chars", ":aGVsbG!8=:", 11, std::nullopt, nullptr},
    {"suffix chars", ":aGVsbG8=!:", 11, std::nullopt, nullptr},
    {"non-zero pad bits",
     ":iZ==:",
     6,
     {{Item("\211", Item::kByteSequenceType), {}}},
     ":iQ==:"},
    {"non-ASCII binary",
     ":/+Ah:",
     6,
     {{Item("\377\340!", Item::kByteSequenceType), {}}},
     nullptr},
    {"base64url binary", ":_-Ah:", 6, std::nullopt, nullptr},
    // boolean.json
    {"basic true boolean", "?1", 2, {{Item(true), {}}}, nullptr},
    {"basic false boolean", "?0", 2, {{Item(false), {}}}, nullptr},
    {"unknown boolean", "?Q", 2, std::nullopt, nullptr},
    {"whitespace boolean", "? 1", 3, std::nullopt, nullptr},
    {"negative zero boolean", "?-0", 3, std::nullopt, nullptr},
    {"T boolean", "?T", 2, std::nullopt, nullptr},
    {"F boolean", "?F", 2, std::nullopt, nullptr},
    {"t boolean", "?t", 2, std::nullopt, nullptr},
    {"f boolean", "?f", 2, std::nullopt, nullptr},
    {"spelled-out True boolean", "?True", 5, std::nullopt, nullptr},
    {"spelled-out False boolean", "?False", 6, std::nullopt, nullptr},
    // examples.json
    {"Foo-Example",
     "2; foourl=\"https://foo.example.com/\"",
     36,
     {{Integer(2), {Param("foourl", "https://foo.example.com/")}}},
     "2;foourl=\"https://foo.example.com/\""},
    {"Example-IntHeader",
     "1; a; b=?0",
     10,
     {{Integer(1), {BooleanParam("a", true), BooleanParam("b", false)}}},
     "1;a;b=?0"},
    {"Example-IntItemHeader", "5", 1, {{Integer(5), {}}}, nullptr},
    {"Example-IntItemHeader (params)",
     "5; foo=bar",
     10,
     {{Integer(5), {TokenParam("foo", "bar")}}},
     "5;foo=bar"},
    {"Example-IntegerHeader", "42", 2, {{Integer(42), {}}}, nullptr},
    {"Example-FloatHeader", "4.5", 3, {{Item(4.500000), {}}}, nullptr},
    {"Example-StringHeader",
     "\"hello world\"",
     13,
     {{Item("hello world"), {}}},
     nullptr},
    {"Example-BinaryHdr",
     ":cHJldGVuZCB0aGlzIGlzIGJpbmFyeSBjb250ZW50Lg==:",
     46,
     {{Item("pretend this is binary content.", Item::kByteSequenceType), {}}},
     nullptr},
    {"Example-BoolHdr", "?1", 2, {{Item(true), {}}}, nullptr},
    // item.json
    {"empty item", "", 0, std::nullopt, nullptr},
    {"leading space", " \t 1", 4, std::nullopt, nullptr},
    {"trailing space", "1 \t ", 4, std::nullopt, nullptr},
    {"leading and trailing space", "  1  ", 5, {{Integer(1), {}}}, "1"},
    {"leading and trailing whitespace", "     1  ", 8, {{Integer(1), {}}}, "1"},
    // number-generated.json
    {"1 digits of zero", "0", 1, {{Integer(0), {}}}, "0"},
    {"1 digit small integer", "1", 1, {{Integer(1), {}}}, nullptr},
    {"1 digit large integer", "9", 1, {{Integer(9), {}}}, nullptr},
    {"2 digits of zero", "00", 2, {{Integer(0), {}}}, "0"},
    {"2 digit small integer", "11", 2, {{Integer(11), {}}}, nullptr},
    {"2 digit large integer", "99", 2, {{Integer(99), {}}}, nullptr},
    {"3 digits of zero", "000", 3, {{Integer(0), {}}}, "0"},
    {"3 digit small integer", "111", 3, {{Integer(111), {}}}, nullptr},
    {"3 digit large integer", "999", 3, {{Integer(999), {}}}, nullptr},
    {"4 digits of zero", "0000", 4, {{Integer(0), {}}}, "0"},
    {"4 digit small integer", "1111", 4, {{Integer(1111), {}}}, nullptr},
    {"4 digit large integer", "9999", 4, {{Integer(9999), {}}}, nullptr},
    {"5 digits of zero", "00000", 5, {{Integer(0), {}}}, "0"},
    {"5 digit small integer", "11111", 5, {{Integer(11111), {}}}, nullptr},
    {"5 digit large integer", "99999", 5, {{Integer(99999), {}}}, nullptr},
    {"6 digits of zero", "000000", 6, {{Integer(0), {}}}, "0"},
    {"6 digit small integer", "111111", 6, {{Integer(111111), {}}}, nullptr},
    {"6 digit large integer", "999999", 6, {{Integer(999999), {}}}, nullptr},
    {"7 digits of zero", "0000000", 7, {{Integer(0), {}}}, "0"},
    {"7 digit small integer", "1111111", 7, {{Integer(1111111), {}}}, nullptr},
    {"7 digit large integer", "9999999", 7, {{Integer(9999999), {}}}, nullptr},
    {"8 digits of zero", "00000000", 8, {{Integer(0), {}}}, "0"},
    {"8 digit small integer",
     "11111111",
     8,
     {{Integer(11111111), {}}},
     nullptr},
    {"8 digit large integer",
     "99999999",
     8,
     {{Integer(99999999), {}}},
     nullptr},
    {"9 digits of zero", "000000000", 9, {{Integer(0), {}}}, "0"},
    {"9 digit small integer",
     "111111111",
     9,
     {{Integer(111111111), {}}},
     nullptr},
    {"9 digit large integer",
     "999999999",
     9,
     {{Integer(999999999), {}}},
     nullptr},
    {"10 digits of zero", "0000000000", 10, {{Integer(0), {}}}, "0"},
    {"10 digit small integer",
     "1111111111",
     10,
     {{Integer(1111111111), {}}},
     nullptr},
    {"10 digit large integer",
     "9999999999",
     10,
     {{Integer(9999999999), {}}},
     nullptr},
    {"11 digits of zero", "00000000000", 11, {{Integer(0), {}}}, "0"},
    {"11 digit small integer",
     "11111111111",
     11,
     {{Integer(11111111111), {}}},
     nullptr},
    {"11 digit large integer",
     "99999999999",
     11,
     {{Integer(99999999999), {}}},
     nullptr},
    {"12 digits of zero", "000000000000", 12, {{Integer(0), {}}}, "0"},
    {"12 digit small integer",
     "111111111111",
     12,
     {{Integer(111111111111), {}}},
     nullptr},
    {"12 digit large integer",
     "999999999999",
     12,
     {{Integer(999999999999), {}}},
     nullptr},
    {"13 digits of zero", "0000000000000", 13, {{Integer(0), {}}}, "0"},
    {"13 digit small integer",
     "1111111111111",
     13,
     {{Integer(1111111111111), {}}},
     nullptr},
    {"13 digit large integer",
     "9999999999999",
     13,
     {{Integer(9999999999999), {}}},
     nullptr},
    {"14 digits of zero", "00000000000000", 14, {{Integer(0), {}}}, "0"},
    {"14 digit small integer",
     "11111111111111",
     14,
     {{Integer(11111111111111), {}}},
     nullptr},
    {"14 digit large integer",
     "99999999999999",
     14,
     {{Integer(99999999999999), {}}},
     nullptr},
    {"15 digits of zero", "000000000000000", 15, {{Integer(0), {}}}, "0"},
    {"15 digit small integer",
     "111111111111111",
     15,
     {{Integer(111111111111111), {}}},
     nullptr},
    {"15 digit large integer",
     "999999999999999",
     15,
     {{Integer(999999999999999), {}}},
     nullptr},
    {"2 digit 0, 1 fractional small decimal",
     "0.1",
     3,
     {{Item(0.100000), {}}},
     "0.1"},
    {"2 digit, 1 fractional 0 decimal",
     "1.0",
     3,
     {{Item(1.000000), {}}},
     "1.0"},
    {"2 digit, 1 fractional small decimal",
     "1.1",
     3,
     {{Item(1.100000), {}}},
     nullptr},
    {"2 digit, 1 fractional large decimal",
     "9.9",
     3,
     {{Item(9.900000), {}}},
     nullptr},
    {"3 digit 0, 2 fractional small decimal",
     "0.11",
     4,
     {{Item(0.110000), {}}},
     "0.11"},
    {"3 digit, 2 fractional 0 decimal",
     "1.00",
     4,
     {{Item(1.000000), {}}},
     "1.0"},
    {"3 digit, 2 fractional small decimal",
     "1.11",
     4,
     {{Item(1.110000), {}}},
     nullptr},
    {"3 digit, 2 fractional large decimal",
     "9.99",
     4,
     {{Item(9.990000), {}}},
     nullptr},
    {"4 digit 0, 3 fractional small decimal",
     "0.111",
     5,
     {{Item(0.111000), {}}},
     "0.111"},
    {"4 digit, 3 fractional 0 decimal",
     "1.000",
     5,
     {{Item(1.000000), {}}},
     "1.0"},
    {"4 digit, 3 fractional small decimal",
     "1.111",
     5,
     {{Item(1.111000), {}}},
     nullptr},
    {"4 digit, 3 fractional large decimal",
     "9.999",
     5,
     {{Item(9.999000), {}}},
     nullptr},
    {"3 digit 0, 1 fractional small decimal",
     "00.1",
     4,
     {{Item(0.100000), {}}},
     "0.1"},
    {"3 digit, 1 fractional 0 decimal",
     "11.0",
     4,
     {{Item(11.000000), {}}},
     "11.0"},
    {"3 digit, 1 fractional small decimal",
     "11.1",
     4,
     {{Item(11.100000), {}}},
     nullptr},
    {"3 digit, 1 fractional large decimal",
     "99.9",
     4,
     {{Item(99.900000), {}}},
     nullptr},
    {"4 digit 0, 2 fractional small decimal",
     "00.11",
     5,
     {{Item(0.110000), {}}},
     "0.11"},
    {"4 digit, 2 fractional 0 decimal",
     "11.00",
     5,
     {{Item(11.000000), {}}},
     "11.0"},
    {"4 digit, 2 fractional small decimal",
     "11.11",
     5,
     {{Item(11.110000), {}}},
     nullptr},
    {"4 digit, 2 fractional large decimal",
     "99.99",
     5,
     {{Item(99.990000), {}}},
     nullptr},
    {"5 digit 0, 3 fractional small decimal",
     "00.111",
     6,
     {{Item(0.111000), {}}},
     "0.111"},
    {"5 digit, 3 fractional 0 decimal",
     "11.000",
     6,
     {{Item(11.000000), {}}},
     "11.0"},
    {"5 digit, 3 fractional small decimal",
     "11.111",
     6,
     {{Item(11.111000), {}}},
     nullptr},
    {"5 digit, 3 fractional large decimal",
     "99.999",
     6,
     {{Item(99.999000), {}}},
     nullptr},
    {"4 digit 0, 1 fractional small decimal",
     "000.1",
     5,
     {{Item(0.100000), {}}},
     "0.1"},
    {"4 digit, 1 fractional 0 decimal",
     "111.0",
     5,
     {{Item(111.000000), {}}},
     "111.0"},
    {"4 digit, 1 fractional small decimal",
     "111.1",
     5,
     {{Item(111.100000), {}}},
     nullptr},
    {"4 digit, 1 fractional large decimal",
     "999.9",
     5,
     {{Item(999.900000), {}}},
     nullptr},
    {"5 digit 0, 2 fractional small decimal",
     "000.11",
     6,
     {{Item(0.110000), {}}},
     "0.11"},
    {"5 digit, 2 fractional 0 decimal",
     "111.00",
     6,
     {{Item(111.000000), {}}},
     "111.0"},
    {"5 digit, 2 fractional small decimal",
     "111.11",
     6,
     {{Item(111.110000), {}}},
     nullptr},
    {"5 digit, 2 fractional large decimal",
     "999.99",
     6,
     {{Item(999.990000), {}}},
     nullptr},
    {"6 digit 0, 3 fractional small decimal",
     "000.111",
     7,
     {{Item(0.111000), {}}},
     "0.111"},
    {"6 digit, 3 fractional 0 decimal",
     "111.000",
     7,
     {{Item(111.000000), {}}},
     "111.0"},
    {"6 digit, 3 fractional small decimal",
     "111.111",
     7,
     {{Item(111.111000), {}}},
     nullptr},
    {"6 digit, 3 fractional large decimal",
     "999.999",
     7,
     {{Item(999.999000), {}}},
     nullptr},
    {"5 digit 0, 1 fractional small decimal",
     "0000.1",
     6,
     {{Item(0.100000), {}}},
     "0.1"},
    {"5 digit, 1 fractional 0 decimal",
     "1111.0",
     6,
     {{Item(1111.000000), {}}},
     "1111.0"},
    {"5 digit, 1 fractional small decimal",
     "1111.1",
     6,
     {{Item(1111.100000), {}}},
     nullptr},
    {"5 digit, 1 fractional large decimal",
     "9999.9",
     6,
     {{Item(9999.900000), {}}},
     nullptr},
    {"6 digit 0, 2 fractional small decimal",
     "0000.11",
     7,
     {{Item(0.110000), {}}},
     "0.11"},
    {"6 digit, 2 fractional 0 decimal",
     "1111.00",
     7,
     {{Item(1111.000000), {}}},
     "1111.0"},
    {"6 digit, 2 fractional small decimal",
     "1111.11",
     7,
     {{Item(1111.110000), {}}},
     nullptr},
    {"6 digit, 2 fractional large decimal",
     "9999.99",
     7,
     {{Item(9999.990000), {}}},
     nullptr},
    {"7 digit 0, 3 fractional small decimal",
     "0000.111",
     8,
     {{Item(0.111000), {}}},
     "0.111"},
    {"7 digit, 3 fractional 0 decimal",
     "1111.000",
     8,
     {{Item(1111.000000), {}}},
     "1111.0"},
    {"7 digit, 3 fractional small decimal",
     "1111.111",
     8,
     {{Item(1111.111000), {}}},
     nullptr},
    {"7 digit, 3 fractional large decimal",
     "9999.999",
     8,
     {{Item(9999.999000), {}}},
     nullptr},
    {"6 digit 0, 1 fractional small decimal",
     "00000.1",
     7,
     {{Item(0.100000), {}}},
     "0.1"},
    {"6 digit, 1 fractional 0 decimal",
     "11111.0",
     7,
     {{Item(11111.000000), {}}},
     "11111.0"},
    {"6 digit, 1 fractional small decimal",
     "11111.1",
     7,
     {{Item(11111.100000), {}}},
     nullptr},
    {"6 digit, 1 fractional large decimal",
     "99999.9",
     7,
     {{Item(99999.900000), {}}},
     nullptr},
    {"7 digit 0, 2 fractional small decimal",
     "00000.11",
     8,
     {{Item(0.110000), {}}},
     "0.11"},
    {"7 digit, 2 fractional 0 decimal",
     "11111.00",
     8,
     {{Item(11111.000000), {}}},
     "11111.0"},
    {"7 digit, 2 fractional small decimal",
     "11111.11",
     8,
     {{Item(11111.110000), {}}},
     nullptr},
    {"7 digit, 2 fractional large decimal",
     "99999.99",
     8,
     {{Item(99999.990000), {}}},
     nullptr},
    {"8 digit 0, 3 fractional small decimal",
     "00000.111",
     9,
     {{Item(0.111000), {}}},
     "0.111"},
    {"8 digit, 3 fractional 0 decimal",
     "11111.000",
     9,
     {{Item(11111.000000), {}}},
     "11111.0"},
    {"8 digit, 3 fractional small decimal",
     "11111.111",
     9,
     {{Item(11111.111000), {}}},
     nullptr},
    {"8 digit, 3 fractional large decimal",
     "99999.999",
     9,
     {{Item(99999.999000), {}}},
     nullptr},
    {"7 digit 0, 1 fractional small decimal",
     "000000.1",
     8,
     {{Item(0.100000), {}}},
     "0.1"},
    {"7 digit, 1 fractional 0 decimal",
     "111111.0",
     8,
     {{Item(111111.000000), {}}},
     "111111.0"},
    {"7 digit, 1 fractional small decimal",
     "111111.1",
     8,
     {{Item(111111.100000), {}}},
     nullptr},
    {"7 digit, 1 fractional large decimal",
     "999999.9",
     8,
     {{Item(999999.900000), {}}},
     nullptr},
    {"8 digit 0, 2 fractional small decimal",
     "000000.11",
     9,
     {{Item(0.110000), {}}},
     "0.11"},
    {"8 digit, 2 fractional 0 decimal",
     "111111.00",
     9,
     {{Item(111111.000000), {}}},
     "111111.0"},
    {"8 digit, 2 fractional small decimal",
     "111111.11",
     9,
     {{Item(111111.110000), {}}},
     nullptr},
    {"8 digit, 2 fractional large decimal",
     "999999.99",
     9,
     {{Item(999999.990000), {}}},
     nullptr},
    {"9 digit 0, 3 fractional small decimal",
     "000000.111",
     10,
     {{Item(0.111000), {}}},
     "0.111"},
    {"9 digit, 3 fractional 0 decimal",
     "111111.000",
     10,
     {{Item(111111.000000), {}}},
     "111111.0"},
    {"9 digit, 3 fractional small decimal",
     "111111.111",
     10,
     {{Item(111111.111000), {}}},
     nullptr},
    {"9 digit, 3 fractional large decimal",
     "999999.999",
     10,
     {{Item(999999.999000), {}}},
     nullptr},
    {"8 digit 0, 1 fractional small decimal",
     "0000000.1",
     9,
     {{Item(0.100000), {}}},
     "0.1"},
    {"8 digit, 1 fractional 0 decimal",
     "1111111.0",
     9,
     {{Item(1111111.000000), {}}},
     "1111111.0"},
    {"8 digit, 1 fractional small decimal",
     "1111111.1",
     9,
     {{Item(1111111.100000), {}}},
     nullptr},
    {"8 digit, 1 fractional large decimal",
     "9999999.9",
     9,
     {{Item(9999999.900000), {}}},
     nullptr},
    {"9 digit 0, 2 fractional small decimal",
     "0000000.11",
     10,
     {{Item(0.110000), {}}},
     "0.11"},
    {"9 digit, 2 fractional 0 decimal",
     "1111111.00",
     10,
     {{Item(1111111.000000), {}}},
     "1111111.0"},
    {"9 digit, 2 fractional small decimal",
     "1111111.11",
     10,
     {{Item(1111111.110000), {}}},
     nullptr},
    {"9 digit, 2 fractional large decimal",
     "9999999.99",
     10,
     {{Item(9999999.990000), {}}},
     nullptr},
    {"10 digit 0, 3 fractional small decimal",
     "0000000.111",
     11,
     {{Item(0.111000), {}}},
     "0.111"},
    {"10 digit, 3 fractional 0 decimal",
     "1111111.000",
     11,
     {{Item(1111111.000000), {}}},
     "1111111.0"},
    {"10 digit, 3 fractional small decimal",
     "1111111.111",
     11,
     {{Item(1111111.111000), {}}},
     nullptr},
    {"10 digit, 3 fractional large decimal",
     "9999999.999",
     11,
     {{Item(9999999.999000), {}}},
     nullptr},
    {"9 digit 0, 1 fractional small decimal",
     "00000000.1",
     10,
     {{Item(0.100000), {}}},
     "0.1"},
    {"9 digit, 1 fractional 0 decimal",
     "11111111.0",
     10,
     {{Item(11111111.000000), {}}},
     "11111111.0"},
    {"9 digit, 1 fractional small decimal",
     "11111111.1",
     10,
     {{Item(11111111.100000), {}}},
     nullptr},
    {"9 digit, 1 fractional large decimal",
     "99999999.9",
     10,
     {{Item(99999999.900000), {}}},
     nullptr},
    {"10 digit 0, 2 fractional small decimal",
     "00000000.11",
     11,
     {{Item(0.110000), {}}},
     "0.11"},
    {"10 digit, 2 fractional 0 decimal",
     "11111111.00",
     11,
     {{Item(11111111.000000), {}}},
     "11111111.0"},
    {"10 digit, 2 fractional small decimal",
     "11111111.11",
     11,
     {{Item(11111111.110000), {}}},
     nullptr},
    {"10 digit, 2 fractional large decimal",
     "99999999.99",
     11,
     {{Item(99999999.990000), {}}},
     nullptr},
    {"11 digit 0, 3 fractional small decimal",
     "00000000.111",
     12,
     {{Item(0.111000), {}}},
     "0.111"},
    {"11 digit, 3 fractional 0 decimal",
     "11111111.000",
     12,
     {{Item(11111111.000000), {}}},
     "11111111.0"},
    {"11 digit, 3 fractional small decimal",
     "11111111.111",
     12,
     {{Item(11111111.111000), {}}},
     nullptr},
    {"11 digit, 3 fractional large decimal",
     "99999999.999",
     12,
     {{Item(99999999.999000), {}}},
     nullptr},
    {"10 digit 0, 1 fractional small decimal",
     "000000000.1",
     11,
     {{Item(0.100000), {}}},
     "0.1"},
    {"10 digit, 1 fractional 0 decimal",
     "111111111.0",
     11,
     {{Item(111111111.000000), {}}},
     "111111111.0"},
    {"10 digit, 1 fractional small decimal",
     "111111111.1",
     11,
     {{Item(111111111.100000), {}}},
     nullptr},
    {"10 digit, 1 fractional large decimal",
     "999999999.9",
     11,
     {{Item(999999999.900000), {}}},
     nullptr},
    {"11 digit 0, 2 fractional small decimal",
     "000000000.11",
     12,
     {{Item(0.110000), {}}},
     "0.11"},
    {"11 digit, 2 fractional 0 decimal",
     "111111111.00",
     12,
     {{Item(111111111.000000), {}}},
     "111111111.0"},
    {"11 digit, 2 fractional small decimal",
     "111111111.11",
     12,
     {{Item(111111111.110000), {}}},
     nullptr},
    {"11 digit, 2 fractional large decimal",
     "999999999.99",
     12,
     {{Item(999999999.990000), {}}},
     nullptr},
    {"12 digit 0, 3 fractional small decimal",
     "000000000.111",
     13,
     {{Item(0.111000), {}}},
     "0.111"},
    {"12 digit, 3 fractional 0 decimal",
     "111111111.000",
     13,
     {{Item(111111111.000000), {}}},
     "111111111.0"},
    {"12 digit, 3 fractional small decimal",
     "111111111.111",
     13,
     {{Item(111111111.111000), {}}},
     nullptr},
    {"12 digit, 3 fractional large decimal",
     "999999999.999",
     13,
     {{Item(999999999.999000), {}}},
     nullptr},
    {"11 digit 0, 1 fractional small decimal",
     "0000000000.1",
     12,
     {{Item(0.100000), {}}},
     "0.1"},
    {"11 digit, 1 fractional 0 decimal",
     "1111111111.0",
     12,
     {{Item(1111111111.000000), {}}},
     "1111111111.0"},
    {"11 digit, 1 fractional small decimal",
     "1111111111.1",
     12,
     {{Item(1111111111.100000), {}}},
     nullptr},
    {"11 digit, 1 fractional large decimal",
     "9999999999.9",
     12,
     {{Item(9999999999.900000), {}}},
     nullptr},
    {"12 digit 0, 2 fractional small decimal",
     "0000000000.11",
     13,
     {{Item(0.110000), {}}},
     "0.11"},
    {"12 digit, 2 fractional 0 decimal",
     "1111111111.00",
     13,
     {{Item(1111111111.000000), {}}},
     "1111111111.0"},
    {"12 digit, 2 fractional small decimal",
     "1111111111.11",
     13,
     {{Item(1111111111.110000), {}}},
     nullptr},
    {"12 digit, 2 fractional large decimal",
     "9999999999.99",
     13,
     {{Item(9999999999.990000), {}}},
     nullptr},
    {"13 digit 0, 3 fractional small decimal",
     "0000000000.111",
     14,
     {{Item(0.111000), {}}},
     "0.111"},
    {"13 digit, 3 fractional 0 decimal",
     "1111111111.000",
     14,
     {{Item(1111111111.000000), {}}},
     "1111111111.0"},
    {"13 digit, 3 fractional small decimal",
     "1111111111.111",
     14,
     {{Item(1111111111.111000), {}}},
     nullptr},
    {"13 digit, 3 fractional large decimal",
     "9999999999.999",
     14,
     {{Item(9999999999.999001), {}}},
     nullptr},
    {"12 digit 0, 1 fractional small decimal",
     "00000000000.1",
     13,
     {{Item(0.100000), {}}},
     "0.1"},
    {"12 digit, 1 fractional 0 decimal",
     "11111111111.0",
     13,
     {{Item(11111111111.000000), {}}},
     "11111111111.0"},
    {"12 digit, 1 fractional small decimal",
     "11111111111.1",
     13,
     {{Item(11111111111.100000), {}}},
     nullptr},
    {"12 digit, 1 fractional large decimal",
     "99999999999.9",
     13,
     {{Item(99999999999.899994), {}}},
     nullptr},
    {"13 digit 0, 2 fractional small decimal",
     "00000000000.11",
     14,
     {{Item(0.110000), {}}},
     "0.11"},
    {"13 digit, 2 fractional 0 decimal",
     "11111111111.00",
     14,
     {{Item(11111111111.000000), {}}},
     "11111111111.0"},
    {"13 digit, 2 fractional small decimal",
     "11111111111.11",
     14,
     {{Item(11111111111.110001), {}}},
     nullptr},
    {"13 digit, 2 fractional large decimal",
     "99999999999.99",
     14,
     {{Item(99999999999.990005), {}}},
     nullptr},
    {"14 digit 0, 3 fractional small decimal",
     "00000000000.111",
     15,
     {{Item(0.111000), {}}},
     "0.111"},
    {"14 digit, 3 fractional 0 decimal",
     "11111111111.000",
     15,
     {{Item(11111111111.000000), {}}},
     "11111111111.0"},
    {"14 digit, 3 fractional small decimal",
     "11111111111.111",
     15,
     {{Item(11111111111.111000), {}}},
     nullptr},
    {"14 digit, 3 fractional large decimal",
     "99999999999.999",
     15,
     {{Item(99999999999.998993), {}}},
     nullptr},
    {"13 digit 0, 1 fractional small decimal",
     "000000000000.1",
     14,
     {{Item(0.100000), {}}},
     "0.1"},
    {"13 digit, 1 fractional 0 decimal",
     "111111111111.0",
     14,
     {{Item(111111111111.000000), {}}},
     "111111111111.0"},
    {"13 digit, 1 fractional small decimal",
     "111111111111.1",
     14,
     {{Item(111111111111.100006), {}}},
     nullptr},
    {"13 digit, 1 fractional large decimal",
     "999999999999.9",
     14,
     {{Item(999999999999.900024), {}}},
     nullptr},
    {"14 digit 0, 2 fractional small decimal",
     "000000000000.11",
     15,
     {{Item(0.110000), {}}},
     "0.11"},
    {"14 digit, 2 fractional 0 decimal",
     "111111111111.00",
     15,
     {{Item(111111111111.000000), {}}},
     "111111111111.0"},
    {"14 digit, 2 fractional small decimal",
     "111111111111.11",
     15,
     {{Item(111111111111.110001), {}}},
     nullptr},
    {"14 digit, 2 fractional large decimal",
     "999999999999.99",
     15,
     {{Item(999999999999.989990), {}}},
     nullptr},
    {"15 digit 0, 3 fractional small decimal",
     "000000000000.111",
     16,
     {{Item(0.111000), {}}},
     "0.111"},
    {"15 digit, 3 fractional 0 decimal",
     "111111111111.000",
     16,
     {{Item(111111111111.000000), {}}},
     "111111111111.0"},
    {"15 digit, 3 fractional small decimal",
     "111111111111.111",
     16,
     {{Item(111111111111.110992), {}}},
     nullptr},
    {"15 digit, 3 fractional large decimal",
     "999999999999.999",
     16,
     {{Item(999999999999.999023), {}}},
     nullptr},
    {"too many digit 0 decimal", "000000000000000.0", 17, std::nullopt,
     nullptr},
    {"too many fractional digits 0 decimal", "000000000000.0000", 17,
     std::nullopt, nullptr},
    {"too many digit 9 decimal", "999999999999999.9", 17, std::nullopt,
     nullptr},
    {"too many fractional digits 9 decimal", "999999999999.9999", 17,
     std::nullopt, nullptr},
    // number.json
    {"basic integer", "42", 2, {{Integer(42), {}}}, nullptr},
    {"zero integer", "0", 1, {{Integer(0), {}}}, nullptr},
    {"negative zero", "-0", 2, {{Integer(0), {}}}, "0"},
    {"double negative zero", "--0", 3, std::nullopt, nullptr},
    {"negative integer", "-42", 3, {{Integer(-42), {}}}, nullptr},
    {"leading 0 integer", "042", 3, {{Integer(42), {}}}, "42"},
    {"leading 0 negative integer", "-042", 4, {{Integer(-42), {}}}, "-42"},
    {"leading 0 zero", "00", 2, {{Integer(0), {}}}, "0"},
    {"comma", "2,3", 3, std::nullopt, nullptr},
    {"negative non-DIGIT first character", "-a23", 4, std::nullopt, nullptr},
    {"sign out of place", "4-2", 3, std::nullopt, nullptr},
    {"whitespace after sign", "- 42", 4, std::nullopt, nullptr},
    {"long integer",
     "123456789012345",
     15,
     {{Integer(123456789012345), {}}},
     nullptr},
    {"long negative integer",
     "-123456789012345",
     16,
     {{Integer(-123456789012345), {}}},
     nullptr},
    {"too long integer", "1234567890123456", 16, std::nullopt, nullptr},
    {"negative too long integer", "-1234567890123456", 17, std::nullopt,
     nullptr},
    {"simple decimal", "1.23", 4, {{Item(1.230000), {}}}, nullptr},
    {"negative decimal", "-1.23", 5, {{Item(-1.230000), {}}}, nullptr},
    {"decimal, whitespace after decimal", "1. 23", 5, std::nullopt, nullptr},
    {"decimal, whitespace before decimal", "1 .23", 5, std::nullopt, nullptr},
    {"negative decimal, whitespace after sign", "- 1.23", 6, std::nullopt,
     nullptr},
    {"tricky precision decimal",
     "123456789012.1",
     14,
     {{Item(123456789012.100006), {}}},
     nullptr},
    {"double decimal decimal", "1.5.4", 5, std::nullopt, nullptr},
    {"adjacent double decimal decimal", "1..4", 4, std::nullopt, nullptr},
    {"decimal with three fractional digits",
     "1.123",
     5,
     {{Item(1.123000), {}}},
     nullptr},
    {"negative decimal with three fractional digits",
     "-1.123",
     6,
     {{Item(-1.123000), {}}},
     nullptr},
    {"decimal with four fractional digits", "1.1234", 6, std::nullopt, nullptr},
    {"negative decimal with four fractional digits", "-1.1234", 7, std::nullopt,
     nullptr},
    {"decimal with thirteen integer digits", "1234567890123.0", 15,
     std::nullopt, nullptr},
    {"negative decimal with thirteen integer digits", "-1234567890123.0", 16,
     std::nullopt, nullptr},
    // string-generated.json
    {"0x00 in string", "\" \000 \"", 5, std::nullopt, nullptr},
    {"0x01 in string", "\" \001 \"", 5, std::nullopt, nullptr},
    {"0x02 in string", "\" \002 \"", 5, std::nullopt, nullptr},
    {"0x03 in string", "\" \003 \"", 5, std::nullopt, nullptr},
    {"0x04 in string", "\" \004 \"", 5, std::nullopt, nullptr},
    {"0x05 in string", "\" \005 \"", 5, std::nullopt, nullptr},
    {"0x06 in string", "\" \006 \"", 5, std::nullopt, nullptr},
    {"0x07 in string", "\" \a \"", 5, std::nullopt, nullptr},
    {"0x08 in string", "\" \b \"", 5, std::nullopt, nullptr},
    {"0x09 in string", "\" \t \"", 5, std::nullopt, nullptr},
    {"0x0a in string", "\" \n \"", 5, std::nullopt, nullptr},
    {"0x0b in string", "\" \v \"", 5, std::nullopt, nullptr},
    {"0x0c in string", "\" \f \"", 5, std::nullopt, nullptr},
    {"0x0d in string", "\" \r \"", 5, std::nullopt, nullptr},
    {"0x0e in string", "\" \016 \"", 5, std::nullopt, nullptr},
    {"0x0f in string", "\" \017 \"", 5, std::nullopt, nullptr},
    {"0x10 in string", "\" \020 \"", 5, std::nullopt, nullptr},
    {"0x11 in string", "\" \021 \"", 5, std::nullopt, nullptr},
    {"0x12 in string", "\" \022 \"", 5, std::nullopt, nullptr},
    {"0x13 in string", "\" \023 \"", 5, std::nullopt, nullptr},
    {"0x14 in string", "\" \024 \"", 5, std::nullopt, nullptr},
    {"0x15 in string", "\" \025 \"", 5, std::nullopt, nullptr},
    {"0x16 in string", "\" \026 \"", 5, std::nullopt, nullptr},
    {"0x17 in string", "\" \027 \"", 5, std::nullopt, nullptr},
    {"0x18 in string", "\" \030 \"", 5, std::nullopt, nullptr},
    {"0x19 in string", "\" \031 \"", 5, std::nullopt, nullptr},
    {"0x1a in string", "\" \032 \"", 5, std::nullopt, nullptr},
    {"0x1b in string", "\" \033 \"", 5, std::nullopt, nullptr},
    {"0x1c in string", "\" \034 \"", 5, std::nullopt, nullptr},
    {"0x1d in string", "\" \035 \"", 5, std::nullopt, nullptr},
    {"0x1e in string", "\" \036 \"", 5, std::nullopt, nullptr},
    {"0x1f in string", "\" \037 \"", 5, std::nullopt, nullptr},
    {"0x20 in string", "\"   \"", 5, {{Item("   "), {}}}, nullptr},
    {"0x21 in string", "\" ! \"", 5, {{Item(" ! "), {}}}, nullptr},
    {"0x22 in string", "\" \" \"", 5, std::nullopt, nullptr},
    {"0x23 in string", "\" # \"", 5, {{Item(" # "), {}}}, nullptr},
    {"0x24 in string", "\" $ \"", 5, {{Item(" $ "), {}}}, nullptr},
    {"0x25 in string", "\" % \"", 5, {{Item(" % "), {}}}, nullptr},
    {"0x26 in string", "\" & \"", 5, {{Item(" & "), {}}}, nullptr},
    {"0x27 in string", "\" ' \"", 5, {{Item(" ' "), {}}}, nullptr},
    {"0x28 in string", "\" ( \"", 5, {{Item(" ( "), {}}}, nullptr},
    {"0x29 in string", "\" ) \"", 5, {{Item(" ) "), {}}}, nullptr},
    {"0x2a in string", "\" * \"", 5, {{Item(" * "), {}}}, nullptr},
    {"0x2b in string", "\" + \"", 5, {{Item(" + "), {}}}, nullptr},
    {"0x2c in string", "\" , \"", 5, {{Item(" , "), {}}}, nullptr},
    {"0x2d in string", "\" - \"", 5, {{Item(" - "), {}}}, nullptr},
    {"0x2e in string", "\" . \"", 5, {{Item(" . "), {}}}, nullptr},
    {"0x2f in string", "\" / \"", 5, {{Item(" / "), {}}}, nullptr},
    {"0x30 in string", "\" 0 \"", 5, {{Item(" 0 "), {}}}, nullptr},
    {"0x31 in string", "\" 1 \"", 5, {{Item(" 1 "), {}}}, nullptr},
    {"0x32 in string", "\" 2 \"", 5, {{Item(" 2 "), {}}}, nullptr},
    {"0x33 in string", "\" 3 \"", 5, {{Item(" 3 "), {}}}, nullptr},
    {"0x34 in string", "\" 4 \"", 5, {{Item(" 4 "), {}}}, nullptr},
    {"0x35 in string", "\" 5 \"", 5, {{Item(" 5 "), {}}}, nullptr},
    {"0x36 in string", "\" 6 \"", 5, {{Item(" 6 "), {}}}, nullptr},
    {"0x37 in string", "\" 7 \"", 5, {{Item(" 7 "), {}}}, nullptr},
    {"0x38 in string", "\" 8 \"", 5, {{Item(" 8 "), {}}}, nullptr},
    {"0x39 in string", "\" 9 \"", 5, {{Item(" 9 "), {}}}, nullptr},
    {"0x3a in string", "\" : \"", 5, {{Item(" : "), {}}}, nullptr},
    {"0x3b in string", "\" ; \"", 5, {{Item(" ; "), {}}}, nullptr},
    {"0x3c in string", "\" < \"", 5, {{Item(" < "), {}}}, nullptr},
    {"0x3d in string", "\" = \"", 5, {{Item(" = "), {}}}, nullptr},
    {"0x3e in string", "\" > \"", 5, {{Item(" > "), {}}}, nullptr},
    {"0x3f in string", "\" ? \"", 5, {{Item(" ? "), {}}}, nullptr},
    {"0x40 in string", "\" @ \"", 5, {{Item(" @ "), {}}}, nullptr},
    {"0x41 in string", "\" A \"", 5, {{Item(" A "), {}}}, nullptr},
    {"0x42 in string", "\" B \"", 5, {{Item(" B "), {}}}, nullptr},
    {"0x43 in string", "\" C \"", 5, {{Item(" C "), {}}}, nullptr},
    {"0x44 in string", "\" D \"", 5, {{Item(" D "), {}}}, nullptr},
    {"0x45 in string", "\" E \"", 5, {{Item(" E "), {}}}, nullptr},
    {"0x46 in string", "\" F \"", 5, {{Item(" F "), {}}}, nullptr},
    {"0x47 in string", "\" G \"", 5, {{Item(" G "), {}}}, nullptr},
    {"0x48 in string", "\" H \"", 5, {{Item(" H "), {}}}, nullptr},
    {"0x49 in string", "\" I \"", 5, {{Item(" I "), {}}}, nullptr},
    {"0x4a in string", "\" J \"", 5, {{Item(" J "), {}}}, nullptr},
    {"0x4b in string", "\" K \"", 5, {{Item(" K "), {}}}, nullptr},
    {"0x4c in string", "\" L \"", 5, {{Item(" L "), {}}}, nullptr},
    {"0x4d in string", "\" M \"", 5, {{Item(" M "), {}}}, nullptr},
    {"0x4e in string", "\" N \"", 5, {{Item(" N "), {}}}, nullptr},
    {"0x4f in string", "\" O \"", 5, {{Item(" O "), {}}}, nullptr},
    {"0x50 in string", "\" P \"", 5, {{Item(" P "), {}}}, nullptr},
    {"0x51 in string", "\" Q \"", 5, {{Item(" Q "), {}}}, nullptr},
    {"0x52 in string", "\" R \"", 5, {{Item(" R "), {}}}, nullptr},
    {"0x53 in string", "\" S \"", 5, {{Item(" S "), {}}}, nullptr},
    {"0x54 in string", "\" T \"", 5, {{Item(" T "), {}}}, nullptr},
    {"0x55 in string", "\" U \"", 5, {{Item(" U "), {}}}, nullptr},
    {"0x56 in string", "\" V \"", 5, {{Item(" V "), {}}}, nullptr},
    {"0x57 in string", "\" W \"", 5, {{Item(" W "), {}}}, nullptr},
    {"0x58 in string", "\" X \"", 5, {{Item(" X "), {}}}, nullptr},
    {"0x59 in string", "\" Y \"", 5, {{Item(" Y "), {}}}, nullptr},
    {"0x5a in string", "\" Z \"", 5, {{Item(" Z "), {}}}, nullptr},
    {"0x5b in string", "\" [ \"", 5, {{Item(" [ "), {}}}, nullptr},
    {"0x5c in string", "\" \\ \"", 5, std::nullopt, nullptr},
    {"0x5d in string", "\" ] \"", 5, {{Item(" ] "), {}}}, nullptr},
    {"0x5e in string", "\" ^ \"", 5, {{Item(" ^ "), {}}}, nullptr},
    {"0x5f in string", "\" _ \"", 5, {{Item(" _ "), {}}}, nullptr},
    {"0x60 in string", "\" ` \"", 5, {{Item(" ` "), {}}}, nullptr},
    {"0x61 in string", "\" a \"", 5, {{Item(" a "), {}}}, nullptr},
    {"0x62 in string", "\" b \"", 5, {{Item(" b "), {}}}, nullptr},
    {"0x63 in string", "\" c \"", 5, {{Item(" c "), {}}}, nullptr},
    {"0x64 in string", "\" d \"", 5, {{Item(" d "), {}}}, nullptr},
    {"0x65 in string", "\" e \"", 5, {{Item(" e "), {}}}, nullptr},
    {"0x66 in string", "\" f \"", 5, {{Item(" f "), {}}}, nullptr},
    {"0x67 in string", "\" g \"", 5, {{Item(" g "), {}}}, nullptr},
    {"0x68 in string", "\" h \"", 5, {{Item(" h "), {}}}, nullptr},
    {"0x69 in string", "\" i \"", 5, {{Item(" i "), {}}}, nullptr},
    {"0x6a in string", "\" j \"", 5, {{Item(" j "), {}}}, nullptr},
    {"0x6b in string", "\" k \"", 5, {{Item(" k "), {}}}, nullptr},
    {"0x6c in string", "\" l \"", 5, {{Item(" l "), {}}}, nullptr},
    {"0x6d in string", "\" m \"", 5, {{Item(" m "), {}}}, nullptr},
    {"0x6e in string", "\" n \"", 5, {{Item(" n "), {}}}, nullptr},
    {"0x6f in string", "\" o \"", 5, {{Item(" o "), {}}}, nullptr},
    {"0x70 in string", "\" p \"", 5, {{Item(" p "), {}}}, nullptr},
    {"0x71 in string", "\" q \"", 5, {{Item(" q "), {}}}, nullptr},
    {"0x72 in string", "\" r \"", 5, {{Item(" r "), {}}}, nullptr},
    {"0x73 in string", "\" s \"", 5, {{Item(" s "), {}}}, nullptr},
    {"0x74 in string", "\" t \"", 5, {{Item(" t "), {}}}, nullptr},
    {"0x75 in string", "\" u \"", 5, {{Item(" u "), {}}}, nullptr},
    {"0x76 in string", "\" v \"", 5, {{Item(" v "), {}}}, nullptr},
    {"0x77 in string", "\" w \"", 5, {{Item(" w "), {}}}, nullptr},
    {"0x78 in string", "\" x \"", 5, {{Item(" x "), {}}}, nullptr},
    {"0x79 in string", "\" y \"", 5, {{Item(" y "), {}}}, nullptr},
    {"0x7a in string", "\" z \"", 5, {{Item(" z "), {}}}, nullptr},
    {"0x7b in string", "\" { \"", 5, {{Item(" { "), {}}}, nullptr},
    {"0x7c in string", "\" | \"", 5, {{Item(" | "), {}}}, nullptr},
    {"0x7d in string", "\" } \"", 5, {{Item(" } "), {}}}, nullptr},
    {"0x7e in string", "\" ~ \"", 5, {{Item(" ~ "), {}}}, nullptr},
    {"0x7f in string", "\" \177 \"", 5, std::nullopt, nullptr},
    {"Escaped 0x00 in string", "\"\\\000\"", 4, std::nullopt, nullptr},
    {"Escaped 0x01 in string", "\"\\\001\"", 4, std::nullopt, nullptr},
    {"Escaped 0x02 in string", "\"\\\002\"", 4, std::nullopt, nullptr},
    {"Escaped 0x03 in string", "\"\\\003\"", 4, std::nullopt, nullptr},
    {"Escaped 0x04 in string", "\"\\\004\"", 4, std::nullopt, nullptr},
    {"Escaped 0x05 in string", "\"\\\005\"", 4, std::nullopt, nullptr},
    {"Escaped 0x06 in string", "\"\\\006\"", 4, std::nullopt, nullptr},
    {"Escaped 0x07 in string", "\"\\\a\"", 4, std::nullopt, nullptr},
    {"Escaped 0x08 in string", "\"\\\b\"", 4, std::nullopt, nullptr},
    {"Escaped 0x09 in string", "\"\\\t\"", 4, std::nullopt, nullptr},
    {"Escaped 0x0a in string", "\"\\\n\"", 4, std::nullopt, nullptr},
    {"Escaped 0x0b in string", "\"\\\v\"", 4, std::nullopt, nullptr},
    {"Escaped 0x0c in string", "\"\\\f\"", 4, std::nullopt, nullptr},
    {"Escaped 0x0d in string", "\"\\\r\"", 4, std::nullopt, nullptr},
    {"Escaped 0x0e in string", "\"\\\016\"", 4, std::nullopt, nullptr},
    {"Escaped 0x0f in string", "\"\\\017\"", 4, std::nullopt, nullptr},
    {"Escaped 0x10 in string", "\"\\\020\"", 4, std::nullopt, nullptr},
    {"Escaped 0x11 in string", "\"\\\021\"", 4, std::nullopt, nullptr},
    {"Escaped 0x12 in string", "\"\\\022\"", 4, std::nullopt, nullptr},
    {"Escaped 0x13 in string", "\"\\\023\"", 4, std::nullopt, nullptr},
    {"Escaped 0x14 in string", "\"\\\024\"", 4, std::nullopt, nullptr},
    {"Escaped 0x15 in string", "\"\\\025\"", 4, std::nullopt, nullptr},
    {"Escaped 0x16 in string", "\"\\\026\"", 4, std::nullopt, nullptr},
    {"Escaped 0x17 in string", "\"\\\027\"", 4, std::nullopt, nullptr},
    {"Escaped 0x18 in string", "\"\\\030\"", 4, std::nullopt, nullptr},
    {"Escaped 0x19 in string", "\"\\\031\"", 4, std::nullopt, nullptr},
    {"Escaped 0x1a in string", "\"\\\032\"", 4, std::nullopt, nullptr},
    {"Escaped 0x1b in string", "\"\\\033\"", 4, std::nullopt, nullptr},
    {"Escaped 0x1c in string", "\"\\\034\"", 4, std::nullopt, nullptr},
    {"Escaped 0x1d in string", "\"\\\035\"", 4, std::nullopt, nullptr},
    {"Escaped 0x1e in string", "\"\\\036\"", 4, std::nullopt, nullptr},
    {"Escaped 0x1f in string", "\"\\\037\"", 4, std::nullopt, nullptr},
    {"Escaped 0x20 in string", "\"\\ \"", 4, std::nullopt, nullptr},
    {"Escaped 0x21 in string", "\"\\!\"", 4, std::nullopt, nullptr},
    {"Escaped 0x22 in string", "\"\\\"\"", 4, {{Item("\""), {}}}, nullptr},
    {"Escaped 0x23 in string", "\"\\#\"", 4, std::nullopt, nullptr},
    {"Escaped 0x24 in string", "\"\\$\"", 4, std::nullopt, nullptr},
    {"Escaped 0x25 in string", "\"\\%\"", 4, std::nullopt, nullptr},
    {"Escaped 0x26 in string", "\"\\&\"", 4, std::nullopt, nullptr},
    {"Escaped 0x27 in string", "\"\\'\"", 4, std::nullopt, nullptr},
    {"Escaped 0x28 in string", "\"\\(\"", 4, std::nullopt, nullptr},
    {"Escaped 0x29 in string", "\"\\)\"", 4, std::nullopt, nullptr},
    {"Escaped 0x2a in string", "\"\\*\"", 4, std::nullopt, nullptr},
    {"Escaped 0x2b in string", "\"\\+\"", 4, std::nullopt, nullptr},
    {"Escaped 0x2c in string", "\"\\,\"", 4, std::nullopt, nullptr},
    {"Escaped 0x2d in string", "\"\\-\"", 4, std::nullopt, nullptr},
    {"Escaped 0x2e in string", "\"\\.\"", 4, std::nullopt, nullptr},
    {"Escaped 0x2f in string", "\"\\/\"", 4, std::nullopt, nullptr},
    {"Escaped 0x30 in string", "\"\\0\"", 4, std::nullopt, nullptr},
    {"Escaped 0x31 in string", "\"\\1\"", 4, std::nullopt, nullptr},
    {"Escaped 0x32 in string", "\"\\2\"", 4, std::nullopt, nullptr},
    {"Escaped 0x33 in string", "\"\\3\"", 4, std::nullopt, nullptr},
    {"Escaped 0x34 in string", "\"\\4\"", 4, std::nullopt, nullptr},
    {"Escaped 0x35 in string", "\"\\5\"", 4, std::nullopt, nullptr},
    {"Escaped 0x36 in string", "\"\\6\"", 4, std::nullopt, nullptr},
    {"Escaped 0x37 in string", "\"\\7\"", 4, std::nullopt, nullptr},
    {"Escaped 0x38 in string", "\"\\8\"", 4, std::nullopt, nullptr},
    {"Escaped 0x39 in string", "\"\\9\"", 4, std::nullopt, nullptr},
    {"Escaped 0x3a in string", "\"\\:\"", 4, std::nullopt, nullptr},
    {"Escaped 0x3b in string", "\"\\;\"", 4, std::nullopt, nullptr},
    {"Escaped 0x3c in string", "\"\\<\"", 4, std::nullopt, nullptr},
    {"Escaped 0x3d in string", "\"\\=\"", 4, std::nullopt, nullptr},
    {"Escaped 0x3e in string", "\"\\>\"", 4, std::nullopt, nullptr},
    {"Escaped 0x3f in string", "\"\\?\"", 4, std::nullopt, nullptr},
    {"Escaped 0x40 in string", "\"\\@\"", 4, std::nullopt, nullptr},
    {"Escaped 0x41 in string", "\"\\A\"", 4, std::nullopt, nullptr},
    {"Escaped 0x42 in string", "\"\\B\"", 4, std::nullopt, nullptr},
    {"Escaped 0x43 in string", "\"\\C\"", 4, std::nullopt, nullptr},
    {"Escaped 0x44 in string", "\"\\D\"", 4, std::nullopt, nullptr},
    {"Escaped 0x45 in string", "\"\\E\"", 4, std::nullopt, nullptr},
    {"Escaped 0x46 in string", "\"\\F\"", 4, std::nullopt, nullptr},
    {"Escaped 0x47 in string", "\"\\G\"", 4, std::nullopt, nullptr},
    {"Escaped 0x48 in string", "\"\\H\"", 4, std::nullopt, nullptr},
    {"Escaped 0x49 in string", "\"\\I\"", 4, std::nullopt, nullptr},
    {"Escaped 0x4a in string", "\"\\J\"", 4, std::nullopt, nullptr},
    {"Escaped 0x4b in string", "\"\\K\"", 4, std::nullopt, nullptr},
    {"Escaped 0x4c in string", "\"\\L\"", 4, std::nullopt, nullptr},
    {"Escaped 0x4d in string", "\"\\M\"", 4, std::nullopt, nullptr},
    {"Escaped 0x4e in string", "\"\\N\"", 4, std::nullopt, nullptr},
    {"Escaped 0x4f in string", "\"\\O\"", 4, std::nullopt, nullptr},
    {"Escaped 0x50 in string", "\"\\P\"", 4, std::nullopt, nullptr},
    {"Escaped 0x51 in string", "\"\\Q\"", 4, std::nullopt, nullptr},
    {"Escaped 0x52 in string", "\"\\R\"", 4, std::nullopt, nullptr},
    {"Escaped 0x53 in string", "\"\\S\"", 4, std::nullopt, nullptr},
    {"Escaped 0x54 in string", "\"\\T\"", 4, std::nullopt, nullptr},
    {"Escaped 0x55 in string", "\"\\U\"", 4, std::nullopt, nullptr},
    {"Escaped 0x56 in string", "\"\\V\"", 4, std::nullopt, nullptr},
    {"Escaped 0x57 in string", "\"\\W\"", 4, std::nullopt, nullptr},
    {"Escaped 0x58 in string", "\"\\X\"", 4, std::nullopt, nullptr},
    {"Escaped 0x59 in string", "\"\\Y\"", 4, std::nullopt, nullptr},
    {"Escaped 0x5a in string", "\"\\Z\"", 4, std::nullopt, nullptr},
    {"Escaped 0x5b in string", "\"\\[\"", 4, std::nullopt, nullptr},
    {"Escaped 0x5c in string", "\"\\\\\"", 4, {{Item("\\"), {}}}, nullptr},
    {"Escaped 0x5d in string", "\"\\]\"", 4, std::nullopt, nullptr},
    {"Escaped 0x5e in string", "\"\\^\"", 4, std::nullopt, nullptr},
    {"Escaped 0x5f in string", "\"\\_\"", 4, std::nullopt, nullptr},
    {"Escaped 0x60 in string", "\"\\`\"", 4, std::nullopt, nullptr},
    {"Escaped 0x61 in string", "\"\\a\"", 4, std::nullopt, nullptr},
    {"Escaped 0x62 in string", "\"\\b\"", 4, std::nullopt, nullptr},
    {"Escaped 0x63 in string", "\"\\c\"", 4, std::nullopt, nullptr},
    {"Escaped 0x64 in string", "\"\\d\"", 4, std::nullopt, nullptr},
    {"Escaped 0x65 in string", "\"\\e\"", 4, std::nullopt, nullptr},
    {"Escaped 0x66 in string", "\"\\f\"", 4, std::nullopt, nullptr},
    {"Escaped 0x67 in string", "\"\\g\"", 4, std::nullopt, nullptr},
    {"Escaped 0x68 in string", "\"\\h\"", 4, std::nullopt, nullptr},
    {"Escaped 0x69 in string", "\"\\i\"", 4, std::nullopt, nullptr},
    {"Escaped 0x6a in string", "\"\\j\"", 4, std::nullopt, nullptr},
    {"Escaped 0x6b in string", "\"\\k\"", 4, std::nullopt, nullptr},
    {"Escaped 0x6c in string", "\"\\l\"", 4, std::nullopt, nullptr},
    {"Escaped 0x6d in string", "\"\\m\"", 4, std::nullopt, nullptr},
    {"Escaped 0x6e in string", "\"\\n\"", 4, std::nullopt, nullptr},
    {"Escaped 0x6f in string", "\"\\o\"", 4, std::nullopt, nullptr},
    {"Escaped 0x70 in string", "\"\\p\"", 4, std::nullopt, nullptr},
    {"Escaped 0x71 in string", "\"\\q\"", 4, std::nullopt, nullptr},
    {"Escaped 0x72 in string", "\"\\r\"", 4, std::nullopt, nullptr},
    {"Escaped 0x73 in string", "\"\\s\"", 4, std::nullopt, nullptr},
    {"Escaped 0x74 in string", "\"\\t\"", 4, std::nullopt, nullptr},
    {"Escaped 0x75 in string", "\"\\u\"", 4, std::nullopt, nullptr},
    {"Escaped 0x76 in string", "\"\\v\"", 4, std::nullopt, nullptr},
    {"Escaped 0x77 in string", "\"\\w\"", 4, std::nullopt, nullptr},
    {"Escaped 0x78 in string", "\"\\x\"", 4, std::nullopt, nullptr},
    {"Escaped 0x79 in string", "\"\\y\"", 4, std::nullopt, nullptr},
    {"Escaped 0x7a in string", "\"\\z\"", 4, std::nullopt, nullptr},
    {"Escaped 0x7b in string", "\"\\{\"", 4, std::nullopt, nullptr},
    {"Escaped 0x7c in string", "\"\\|\"", 4, std::nullopt, nullptr},
    {"Escaped 0x7d in string", "\"\\}\"", 4, std::nullopt, nullptr},
    {"Escaped 0x7e in string", "\"\\~\"", 4, std::nullopt, nullptr},
    {"Escaped 0x7f in string", "\"\\\177\"", 4, std::nullopt, nullptr},
    // string.json
    {"basic string", "\"foo bar\"", 9, {{Item("foo bar"), {}}}, nullptr},
    {"empty string", "\"\"", 2, {{Item(""), {}}}, nullptr},
    {"long string",
     "\"foo foo foo foo foo foo foo foo foo foo foo foo foo foo foo foo foo "
     "foo foo foo foo foo foo foo foo foo foo foo foo foo foo foo foo foo foo "
     "foo foo foo foo foo foo foo foo foo foo foo foo foo foo foo foo foo foo "
     "foo foo foo foo foo foo foo foo foo foo foo foo \"",
     262,
     {{Item("foo foo foo foo foo foo foo foo foo foo foo foo foo foo foo foo "
            "foo foo foo foo foo foo foo foo foo foo foo foo foo foo foo foo "
            "foo foo foo foo foo foo foo foo foo foo foo foo foo foo foo foo "
            "foo foo foo foo foo foo foo foo foo foo foo foo foo foo foo foo "
            "foo "),
       {}}},
     nullptr},
    {"whitespace string", "\"   \"", 5, {{Item("   "), {}}}, nullptr},
    {"non-ascii string", "\"f\374\374\"", 5, std::nullopt, nullptr},
    {"tab in string", "\"\\t\"", 4, std::nullopt, nullptr},
    {"newline in string", "\" \\n \"", 6, std::nullopt, nullptr},
    {"single quoted string", "'foo'", 5, std::nullopt, nullptr},
    {"unbalanced string", "\"foo", 4, std::nullopt, nullptr},
    {"string quoting",
     "\"foo \\\"bar\\\" \\\\ baz\"",
     20,
     {{Item("foo \"bar\" \\ baz"), {}}},
     nullptr},
    {"bad string quoting", "\"foo \\,\"", 8, std::nullopt, nullptr},
    {"ending string quote", "\"foo \\\"", 7, std::nullopt, nullptr},
    {"abruptly ending string quote", "\"foo \\", 6, std::nullopt, nullptr},
    // token-generated.json
    {"0x00 in token", "a\000a", 3, std::nullopt, nullptr},
    {"0x01 in token", "a\001a", 3, std::nullopt, nullptr},
    {"0x02 in token", "a\002a", 3, std::nullopt, nullptr},
    {"0x03 in token", "a\003a", 3, std::nullopt, nullptr},
    {"0x04 in token", "a\004a", 3, std::nullopt, nullptr},
    {"0x05 in token", "a\005a", 3, std::nullopt, nullptr},
    {"0x06 in token", "a\006a", 3, std::nullopt, nullptr},
    {"0x07 in token", "a\aa", 3, std::nullopt, nullptr},
    {"0x08 in token", "a\ba", 3, std::nullopt, nullptr},
    {"0x09 in token", "a\ta", 3, std::nullopt, nullptr},
    {"0x0a in token", "a\na", 3, std::nullopt, nullptr},
    {"0x0b in token", "a\va", 3, std::nullopt, nullptr},
    {"0x0c in token", "a\fa", 3, std::nullopt, nullptr},
    {"0x0d in token", "a\ra", 3, std::nullopt, nullptr},
    {"0x0e in token", "a\016a", 3, std::nullopt, nullptr},
    {"0x0f in token", "a\017a", 3, std::nullopt, nullptr},
    {"0x10 in token", "a\020a", 3, std::nullopt, nullptr},
    {"0x11 in token", "a\021a", 3, std::nullopt, nullptr},
    {"0x12 in token", "a\022a", 3, std::nullopt, nullptr},
    {"0x13 in token", "a\023a", 3, std::nullopt, nullptr},
    {"0x14 in token", "a\024a", 3, std::nullopt, nullptr},
    {"0x15 in token", "a\025a", 3, std::nullopt, nullptr},
    {"0x16 in token", "a\026a", 3, std::nullopt, nullptr},
    {"0x17 in token", "a\027a", 3, std::nullopt, nullptr},
    {"0x18 in token", "a\030a", 3, std::nullopt, nullptr},
    {"0x19 in token", "a\031a", 3, std::nullopt, nullptr},
    {"0x1a in token", "a\032a", 3, std::nullopt, nullptr},
    {"0x1b in token", "a\033a", 3, std::nullopt, nullptr},
    {"0x1c in token", "a\034a", 3, std::nullopt, nullptr},
    {"0x1d in token", "a\035a", 3, std::nullopt, nullptr},
    {"0x1e in token", "a\036a", 3, std::nullopt, nullptr},
    {"0x1f in token", "a\037a", 3, std::nullopt, nullptr},
    {"0x20 in token", "a a", 3, std::nullopt, nullptr},
    {"0x21 in token", "a!a", 3, {{Item("a!a", Item::kTokenType), {}}}, nullptr},
    {"0x22 in token", "a\"a", 3, std::nullopt, nullptr},
    {"0x23 in token", "a#a", 3, {{Item("a#a", Item::kTokenType), {}}}, nullptr},
    {"0x24 in token", "a$a", 3, {{Item("a$a", Item::kTokenType), {}}}, nullptr},
    {"0x25 in token", "a%a", 3, {{Item("a%a", Item::kTokenType), {}}}, nullptr},
    {"0x26 in token", "a&a", 3, {{Item("a&a", Item::kTokenType), {}}}, nullptr},
    {"0x27 in token", "a'a", 3, {{Item("a'a", Item::kTokenType), {}}}, nullptr},
    {"0x28 in token", "a(a", 3, std::nullopt, nullptr},
    {"0x29 in token", "a)a", 3, std::nullopt, nullptr},
    {"0x2a in token", "a*a", 3, {{Item("a*a", Item::kTokenType), {}}}, nullptr},
    {"0x2b in token", "a+a", 3, {{Item("a+a", Item::kTokenType), {}}}, nullptr},
    {"0x2c in token", "a,a", 3, std::nullopt, nullptr},
    {"0x2d in token", "a-a", 3, {{Item("a-a", Item::kTokenType), {}}}, nullptr},
    {"0x2e in token", "a.a", 3, {{Item("a.a", Item::kTokenType), {}}}, nullptr},
    {"0x2f in token", "a/a", 3, {{Item("a/a", Item::kTokenType), {}}}, nullptr},
    {"0x30 in token", "a0a", 3, {{Item("a0a", Item::kTokenType), {}}}, nullptr},
    {"0x31 in token", "a1a", 3, {{Item("a1a", Item::kTokenType), {}}}, nullptr},
    {"0x32 in token", "a2a", 3, {{Item("a2a", Item::kTokenType), {}}}, nullptr},
    {"0x33 in token", "a3a", 3, {{Item("a3a", Item::kTokenType), {}}}, nullptr},
    {"0x34 in token", "a4a", 3, {{Item("a4a", Item::kTokenType), {}}}, nullptr},
    {"0x35 in token", "a5a", 3, {{Item("a5a", Item::kTokenType), {}}}, nullptr},
    {"0x36 in token", "a6a", 3, {{Item("a6a", Item::kTokenType), {}}}, nullptr},
    {"0x37 in token", "a7a", 3, {{Item("a7a", Item::kTokenType), {}}}, nullptr},
    {"0x38 in token", "a8a", 3, {{Item("a8a", Item::kTokenType), {}}}, nullptr},
    {"0x39 in token", "a9a", 3, {{Item("a9a", Item::kTokenType), {}}}, nullptr},
    {"0x3a in token", "a:a", 3, {{Item("a:a", Item::kTokenType), {}}}, nullptr},
    {"0x3b in token",
     "a;a",
     3,
     {{Item("a", Item::kTokenType), {BooleanParam("a", true)}}},
     nullptr},
    {"0x3c in token", "a<a", 3, std::nullopt, nullptr},
    {"0x3d in token", "a=a", 3, std::nullopt, nullptr},
    {"0x3e in token", "a>a", 3, std::nullopt, nullptr},
    {"0x3f in token", "a?a", 3, std::nullopt, nullptr},
    {"0x40 in token", "a@a", 3, std::nullopt, nullptr},
    {"0x41 in token", "aAa", 3, {{Item("aAa", Item::kTokenType), {}}}, nullptr},
    {"0x42 in token", "aBa", 3, {{Item("aBa", Item::kTokenType), {}}}, nullptr},
    {"0x43 in token", "aCa", 3, {{Item("aCa", Item::kTokenType), {}}}, nullptr},
    {"0x44 in token", "aDa", 3, {{Item("aDa", Item::kTokenType), {}}}, nullptr},
    {"0x45 in token", "aEa", 3, {{Item("aEa", Item::kTokenType), {}}}, nullptr},
    {"0x46 in token", "aFa", 3, {{Item("aFa", Item::kTokenType), {}}}, nullptr},
    {"0x47 in token", "aGa", 3, {{Item("aGa", Item::kTokenType), {}}}, nullptr},
    {"0x48 in token", "aHa", 3, {{Item("aHa", Item::kTokenType), {}}}, nullptr},
    {"0x49 in token", "aIa", 3, {{Item("aIa", Item::kTokenType), {}}}, nullptr},
    {"0x4a in token", "aJa", 3, {{Item("aJa", Item::kTokenType), {}}}, nullptr},
    {"0x4b in token", "aKa", 3, {{Item("aKa", Item::kTokenType), {}}}, nullptr},
    {"0x4c in token", "aLa", 3, {{Item("aLa", Item::kTokenType), {}}}, nullptr},
    {"0x4d in token", "aMa", 3, {{Item("aMa", Item::kTokenType), {}}}, nullptr},
    {"0x4e in token", "aNa", 3, {{Item("aNa", Item::kTokenType), {}}}, nullptr},
    {"0x4f in token", "aOa", 3, {{Item("aOa", Item::kTokenType), {}}}, nullptr},
    {"0x50 in token", "aPa", 3, {{Item("aPa", Item::kTokenType), {}}}, nullptr},
    {"0x51 in token", "aQa", 3, {{Item("aQa", Item::kTokenType), {}}}, nullptr},
    {"0x52 in token", "aRa", 3, {{Item("aRa", Item::kTokenType), {}}}, nullptr},
    {"0x53 in token", "aSa", 3, {{Item("aSa", Item::kTokenType), {}}}, nullptr},
    {"0x54 in token", "aTa", 3, {{Item("aTa", Item::kTokenType), {}}}, nullptr},
    {"0x55 in token", "aUa", 3, {{Item("aUa", Item::kTokenType), {}}}, nullptr},
    {"0x56 in token", "aVa", 3, {{Item("aVa", Item::kTokenType), {}}}, nullptr},
    {"0x57 in token", "aWa", 3, {{Item("aWa", Item::kTokenType), {}}}, nullptr},
    {"0x58 in token", "aXa", 3, {{Item("aXa", Item::kTokenType), {}}}, nullptr},
    {"0x59 in token", "aYa", 3, {{Item("aYa", Item::kTokenType), {}}}, nullptr},
    {"0x5a in token", "aZa", 3, {{Item("aZa", Item::kTokenType), {}}}, nullptr},
    {"0x5b in token", "a[a", 3, std::nullopt, nullptr},
    {"0x5c in token", "a\\a", 3, std::nullopt, nullptr},
    {"0x5d in token", "a]a", 3, std::nullopt, nullptr},
    {"0x5e in token", "a^a", 3, {{Item("a^a", Item::kTokenType), {}}}, nullptr},
    {"0x5f in token", "a_a", 3, {{Item("a_a", Item::kTokenType), {}}}, nullptr},
    {"0x60 in token", "a`a", 3, {{Item("a`a", Item::kTokenType), {}}}, nullptr},
    {"0x61 in token", "aaa", 3, {{Item("aaa", Item::kTokenType), {}}}, nullptr},
    {"0x62 in token", "aba", 3, {{Item("aba", Item::kTokenType), {}}}, nullptr},
    {"0x63 in token", "aca", 3, {{Item("aca", Item::kTokenType), {}}}, nullptr},
    {"0x64 in token", "ada", 3, {{Item("ada", Item::kTokenType), {}}}, nullptr},
    {"0x65 in token", "aea", 3, {{Item("aea", Item::kTokenType), {}}}, nullptr},
    {"0x66 in token", "afa", 3, {{Item("afa", Item::kTokenType), {}}}, nullptr},
    {"0x67 in token", "aga", 3, {{Item("aga", Item::kTokenType), {}}}, nullptr},
    {"0x68 in token", "aha", 3, {{Item("aha", Item::kTokenType), {}}}, nullptr},
    {"0x69 in token", "aia", 3, {{Item("aia", Item::kTokenType), {}}}, nullptr},
    {"0x6a in token", "aja", 3, {{Item("aja", Item::kTokenType), {}}}, nullptr},
    {"0x6b in token", "aka", 3, {{Item("aka", Item::kTokenType), {}}}, nullptr},
    {"0x6c in token", "ala", 3, {{Item("ala", Item::kTokenType), {}}}, nullptr},
    {"0x6d in token", "ama", 3, {{Item("ama", Item::kTokenType), {}}}, nullptr},
    {"0x6e in token", "ana", 3, {{Item("ana", Item::kTokenType), {}}}, nullptr},
    {"0x6f in token", "aoa", 3, {{Item("aoa", Item::kTokenType), {}}}, nullptr},
    {"0x70 in token", "apa", 3, {{Item("apa", Item::kTokenType), {}}}, nullptr},
    {"0x71 in token", "aqa", 3, {{Item("aqa", Item::kTokenType), {}}}, nullptr},
    {"0x72 in token", "ara", 3, {{Item("ara", Item::kTokenType), {}}}, nullptr},
    {"0x73 in token", "asa", 3, {{Item("asa", Item::kTokenType), {}}}, nullptr},
    {"0x74 in token", "ata", 3, {{Item("ata", Item::kTokenType), {}}}, nullptr},
    {"0x75 in token", "aua", 3, {{Item("aua", Item::kTokenType), {}}}, nullptr},
    {"0x76 in token", "ava", 3, {{Item("ava", Item::kTokenType), {}}}, nullptr},
    {"0x77 in token", "awa", 3, {{Item("awa", Item::kTokenType), {}}}, nullptr},
    {"0x78 in token", "axa", 3, {{Item("axa", Item::kTokenType), {}}}, nullptr},
    {"0x79 in token", "aya", 3, {{Item("aya", Item::kTokenType), {}}}, nullptr},
    {"0x7a in token", "aza", 3, {{Item("aza", Item::kTokenType), {}}}, nullptr},
    {"0x7b in token", "a{a", 3, std::nullopt, nullptr},
    {"0x7c in token", "a|a", 3, {{Item("a|a", Item::kTokenType), {}}}, nullptr},
    {"0x7d in token", "a}a", 3, std::nullopt, nullptr},
    {"0x7e in token", "a~a", 3, {{Item("a~a", Item::kTokenType), {}}}, nullptr},
    {"0x7f in token", "a\177a", 3, std::nullopt, nullptr},
    {"0x00 starting an token", "\000a", 2, std::nullopt, nullptr},
    {"0x01 starting an token", "\001a", 2, std::nullopt, nullptr},
    {"0x02 starting an token", "\002a", 2, std::nullopt, nullptr},
    {"0x03 starting an token", "\003a", 2, std::nullopt, nullptr},
    {"0x04 starting an token", "\004a", 2, std::nullopt, nullptr},
    {"0x05 starting an token", "\005a", 2, std::nullopt, nullptr},
    {"0x06 starting an token", "\006a", 2, std::nullopt, nullptr},
    {"0x07 starting an token", "\aa", 2, std::nullopt, nullptr},
    {"0x08 starting an token", "\ba", 2, std::nullopt, nullptr},
    {"0x09 starting an token", "\ta", 2, std::nullopt, nullptr},
    {"0x0a starting an token", "\na", 2, std::nullopt, nullptr},
    {"0x0b starting an token", "\va", 2, std::nullopt, nullptr},
    {"0x0c starting an token", "\fa", 2, std::nullopt, nullptr},
    {"0x0d starting an token", "\ra", 2, std::nullopt, nullptr},
    {"0x0e starting an token", "\016a", 2, std::nullopt, nullptr},
    {"0x0f starting an token", "\017a", 2, std::nullopt, nullptr},
    {"0x10 starting an token", "\020a", 2, std::nullopt, nullptr},
    {"0x11 starting an token", "\021a", 2, std::nullopt, nullptr},
    {"0x12 starting an token", "\022a", 2, std::nullopt, nullptr},
    {"0x13 starting an token", "\023a", 2, std::nullopt, nullptr},
    {"0x14 starting an token", "\024a", 2, std::nullopt, nullptr},
    {"0x15 starting an token", "\025a", 2, std::nullopt, nullptr},
    {"0x16 starting an token", "\026a", 2, std::nullopt, nullptr},
    {"0x17 starting an token", "\027a", 2, std::nullopt, nullptr},
    {"0x18 starting an token", "\030a", 2, std::nullopt, nullptr},
    {"0x19 starting an token", "\031a", 2, std::nullopt, nullptr},
    {"0x1a starting an token", "\032a", 2, std::nullopt, nullptr},
    {"0x1b starting an token", "\033a", 2, std::nullopt, nullptr},
    {"0x1c starting an token", "\034a", 2, std::nullopt, nullptr},
    {"0x1d starting an token", "\035a", 2, std::nullopt, nullptr},
    {"0x1e starting an token", "\036a", 2, std::nullopt, nullptr},
    {"0x1f starting an token", "\037a", 2, std::nullopt, nullptr},
    {"0x20 starting an token",
     " a",
     2,
     {{Item("a", Item::kTokenType), {}}},
     "a"},
    {"0x21 starting an token", "!a", 2, std::nullopt, nullptr},
    {"0x22 starting an token", "\"a", 2, std::nullopt, nullptr},
    {"0x23 starting an token", "#a", 2, std::nullopt, nullptr},
    {"0x24 starting an token", "$a", 2, std::nullopt, nullptr},
    {"0x25 starting an token", "%a", 2, std::nullopt, nullptr},
    {"0x26 starting an token", "&a", 2, std::nullopt, nullptr},
    {"0x27 starting an token", "'a", 2, std::nullopt, nullptr},
    {"0x28 starting an token", "(a", 2, std::nullopt, nullptr},
    {"0x29 starting an token", ")a", 2, std::nullopt, nullptr},
    {"0x2a starting an token",
     "*a",
     2,
     {{Item("*a", Item::kTokenType), {}}},
     nullptr},
    {"0x2b starting an token", "+a", 2, std::nullopt, nullptr},
    {"0x2c starting an token", ",a", 2, std::nullopt, nullptr},
    {"0x2d starting an token", "-a", 2, std::nullopt, nullptr},
    {"0x2e starting an token", ".a", 2, std::nullopt, nullptr},
    {"0x2f starting an token", "/a", 2, std::nullopt, nullptr},
    {"0x30 starting an token", "0a", 2, std::nullopt, nullptr},
    {"0x31 starting an token", "1a", 2, std::nullopt, nullptr},
    {"0x32 starting an token", "2a", 2, std::nullopt, nullptr},
    {"0x33 starting an token", "3a", 2, std::nullopt, nullptr},
    {"0x34 starting an token", "4a", 2, std::nullopt, nullptr},
    {"0x35 starting an token", "5a", 2, std::nullopt, nullptr},
    {"0x36 starting an token", "6a", 2, std::nullopt, nullptr},
    {"0x37 starting an token", "7a", 2, std::nullopt, nullptr},
    {"0x38 starting an token", "8a", 2, std::nullopt, nullptr},
    {"0x39 starting an token", "9a", 2, std::nullopt, nullptr},
    {"0x3a starting an token", ":a", 2, std::nullopt, nullptr},
    {"0x3b starting an token", ";a", 2, std::nullopt, nullptr},
    {"0x3c starting an token", "<a", 2, std::nullopt, nullptr},
    {"0x3d starting an token", "=a", 2, std::nullopt, nullptr},
    {"0x3e starting an token", ">a", 2, std::nullopt, nullptr},
    {"0x3f starting an token", "?a", 2, std::nullopt, nullptr},
    {"0x40 starting an token", "@a", 2, std::nullopt, nullptr},
    {"0x41 starting an token",
     "Aa",
     2,
     {{Item("Aa", Item::kTokenType), {}}},
     nullptr},
    {"0x42 starting an token",
     "Ba",
     2,
     {{Item("Ba", Item::kTokenType), {}}},
     nullptr},
    {"0x43 starting an token",
     "Ca",
     2,
     {{Item("Ca", Item::kTokenType), {}}},
     nullptr},
    {"0x44 starting an token",
     "Da",
     2,
     {{Item("Da", Item::kTokenType), {}}},
     nullptr},
    {"0x45 starting an token",
     "Ea",
     2,
     {{Item("Ea", Item::kTokenType), {}}},
     nullptr},
    {"0x46 starting an token",
     "Fa",
     2,
     {{Item("Fa", Item::kTokenType), {}}},
     nullptr},
    {"0x47 starting an token",
     "Ga",
     2,
     {{Item("Ga", Item::kTokenType), {}}},
     nullptr},
    {"0x48 starting an token",
     "Ha",
     2,
     {{Item("Ha", Item::kTokenType), {}}},
     nullptr},
    {"0x49 starting an token",
     "Ia",
     2,
     {{Item("Ia", Item::kTokenType), {}}},
     nullptr},
    {"0x4a starting an token",
     "Ja",
     2,
     {{Item("Ja", Item::kTokenType), {}}},
     nullptr},
    {"0x4b starting an token",
     "Ka",
     2,
     {{Item("Ka", Item::kTokenType), {}}},
     nullptr},
    {"0x4c starting an token",
     "La",
     2,
     {{Item("La", Item::kTokenType), {}}},
     nullptr},
    {"0x4d starting an token",
     "Ma",
     2,
     {{Item("Ma", Item::kTokenType), {}}},
     nullptr},
    {"0x4e starting an token",
     "Na",
     2,
     {{Item("Na", Item::kTokenType), {}}},
     nullptr},
    {"0x4f starting an token",
     "Oa",
     2,
     {{Item("Oa", Item::kTokenType), {}}},
     nullptr},
    {"0x50 starting an token",
     "Pa",
     2,
     {{Item("Pa", Item::kTokenType), {}}},
     nullptr},
    {"0x51 starting an token",
     "Qa",
     2,
     {{Item("Qa", Item::kTokenType), {}}},
     nullptr},
    {"0x52 starting an token",
     "Ra",
     2,
     {{Item("Ra", Item::kTokenType), {}}},
     nullptr},
    {"0x53 starting an token",
     "Sa",
     2,
     {{Item("Sa", Item::kTokenType), {}}},
     nullptr},
    {"0x54 starting an token",
     "Ta",
     2,
     {{Item("Ta", Item::kTokenType), {}}},
     nullptr},
    {"0x55 starting an token",
     "Ua",
     2,
     {{Item("Ua", Item::kTokenType), {}}},
     nullptr},
    {"0x56 starting an token",
     "Va",
     2,
     {{Item("Va", Item::kTokenType), {}}},
     nullptr},
    {"0x57 starting an token",
     "Wa",
     2,
     {{Item("Wa", Item::kTokenType), {}}},
     nullptr},
    {"0x58 starting an token",
     "Xa",
     2,
     {{Item("Xa", Item::kTokenType), {}}},
     nullptr},
    {"0x59 starting an token",
     "Ya",
     2,
     {{Item("Ya", Item::kTokenType), {}}},
     nullptr},
    {"0x5a starting an token",
     "Za",
     2,
     {{Item("Za", Item::kTokenType), {}}},
     nullptr},
    {"0x5b starting an token", "[a", 2, std::nullopt, nullptr},
    {"0x5c starting an token", "\\a", 2, std::nullopt, nullptr},
    {"0x5d starting an token", "]a", 2, std::nullopt, nullptr},
    {"0x5e starting an token", "^a", 2, std::nullopt, nullptr},
    {"0x5f starting an token", "_a", 2, std::nullopt, nullptr},
    {"0x60 starting an token", "`a", 2, std::nullopt, nullptr},
    {"0x61 starting an token",
     "aa",
     2,
     {{Item("aa", Item::kTokenType), {}}},
     nullptr},
    {"0x62 starting an token",
     "ba",
     2,
     {{Item("ba", Item::kTokenType), {}}},
     nullptr},
    {"0x63 starting an token",
     "ca",
     2,
     {{Item("ca", Item::kTokenType), {}}},
     nullptr},
    {"0x64 starting an token",
     "da",
     2,
     {{Item("da", Item::kTokenType), {}}},
     nullptr},
    {"0x65 starting an token",
     "ea",
     2,
     {{Item("ea", Item::kTokenType), {}}},
     nullptr},
    {"0x66 starting an token",
     "fa",
     2,
     {{Item("fa", Item::kTokenType), {}}},
     nullptr},
    {"0x67 starting an token",
     "ga",
     2,
     {{Item("ga", Item::kTokenType), {}}},
     nullptr},
    {"0x68 starting an token",
     "ha",
     2,
     {{Item("ha", Item::kTokenType), {}}},
     nullptr},
    {"0x69 starting an token",
     "ia",
     2,
     {{Item("ia", Item::kTokenType), {}}},
     nullptr},
    {"0x6a starting an token",
     "ja",
     2,
     {{Item("ja", Item::kTokenType), {}}},
     nullptr},
    {"0x6b starting an token",
     "ka",
     2,
     {{Item("ka", Item::kTokenType), {}}},
     nullptr},
    {"0x6c starting an token",
     "la",
     2,
     {{Item("la", Item::kTokenType), {}}},
     nullptr},
    {"0x6d starting an token",
     "ma",
     2,
     {{Item("ma", Item::kTokenType), {}}},
     nullptr},
    {"0x6e starting an token",
     "na",
     2,
     {{Item("na", Item::kTokenType), {}}},
     nullptr},
    {"0x6f starting an token",
     "oa",
     2,
     {{Item("oa", Item::kTokenType), {}}},
     nullptr},
    {"0x70 starting an token",
     "pa",
     2,
     {{Item("pa", Item::kTokenType), {}}},
     nullptr},
    {"0x71 starting an token",
     "qa",
     2,
     {{Item("qa", Item::kTokenType), {}}},
     nullptr},
    {"0x72 starting an token",
     "ra",
     2,
     {{Item("ra", Item::kTokenType), {}}},
     nullptr},
    {"0x73 starting an token",
     "sa",
     2,
     {{Item("sa", Item::kTokenType), {}}},
     nullptr},
    {"0x74 starting an token",
     "ta",
     2,
     {{Item("ta", Item::kTokenType), {}}},
     nullptr},
    {"0x75 starting an token",
     "ua",
     2,
     {{Item("ua", Item::kTokenType), {}}},
     nullptr},
    {"0x76 starting an token",
     "va",
     2,
     {{Item("va", Item::kTokenType), {}}},
     nullptr},
    {"0x77 starting an token",
     "wa",
     2,
     {{Item("wa", Item::kTokenType), {}}},
     nullptr},
    {"0x78 starting an token",
     "xa",
     2,
     {{Item("xa", Item::kTokenType), {}}},
     nullptr},
    {"0x79 starting an token",
     "ya",
     2,
     {{Item("ya", Item::kTokenType), {}}},
     nullptr},
    {"0x7a starting an token",
     "za",
     2,
     {{Item("za", Item::kTokenType), {}}},
     nullptr},
    {"0x7b starting an token", "{a", 2, std::nullopt, nullptr},
    {"0x7c starting an token", "|a", 2, std::nullopt, nullptr},
    {"0x7d starting an token", "}a", 2, std::nullopt, nullptr},
    {"0x7e starting an token", "~a", 2, std::nullopt, nullptr},
    {"0x7f starting an token", "\177a", 2, std::nullopt, nullptr},
    // token.json
    {"basic token - item",
     "a_b-c.d3:f%00/*",
     15,
     {{Item("a_b-c.d3:f%00/*", Item::kTokenType), {}}},
     nullptr},
    {"token with capitals - item",
     "fooBar",
     6,
     {{Item("fooBar", Item::kTokenType), {}}},
     nullptr},
    {"token starting with capitals - item",
     "FooBar",
     6,
     {{Item("FooBar", Item::kTokenType), {}}},
     nullptr},
};

const struct ListTestCase {
  const char* name;
  const char* raw;
  size_t raw_len;
  const std::optional<List> expected;  // nullopt if parse error is expected.
  const char* canonical;  // nullptr if parse error is expected, or if canonical
                          // format is identical to raw.
} list_test_cases[] = {
    // examples.json
    {"Example-StrListHeader",
     "\"foo\", \"bar\", \"It was the best of times.\"",
     41,
     {{{Item("foo"), {}},
       {Item("bar"), {}},
       {Item("It was the best of times."), {}}}},
     nullptr},
    {"Example-Hdr (list on one line)",
     "foo, bar",
     8,
     {{{Item("foo", Item::kTokenType), {}},
       {Item("bar", Item::kTokenType), {}}}},
     nullptr},
    {"Example-Hdr (list on two lines)",
     "foo, bar",
     8,
     {{{Item("foo", Item::kTokenType), {}},
       {Item("bar", Item::kTokenType), {}}}},
     "foo, bar"},
    {"Example-StrListListHeader",
     "(\"foo\" \"bar\"), (\"baz\"), (\"bat\" \"one\"), ()",
     41,
     {{{{{Item("foo"), {}}, {Item("bar"), {}}}, {}},
       {{{Item("baz"), {}}}, {}},
       {{{Item("bat"), {}}, {Item("one"), {}}}, {}},
       {std::vector<ParameterizedItem>(), {}}}},
     nullptr},
    {"Example-ListListParam",
     "(\"foo\"; a=1;b=2);lvl=5, (\"bar\" \"baz\");lvl=1",
     43,
     {{{{{Item("foo"), {Param("a", 1), Param("b", 2)}}}, {Param("lvl", 5)}},
       {{{Item("bar"), {}}, {Item("baz"), {}}}, {Param("lvl", 1)}}}},
     "(\"foo\";a=1;b=2);lvl=5, (\"bar\" \"baz\");lvl=1"},
    {"Example-ParamListHeader",
     "abc;a=1;b=2; cde_456, (ghi;jk=4 l);q=\"9\";r=w",
     44,
     {{{Item("abc", Item::kTokenType),
        {Param("a", 1), Param("b", 2), BooleanParam("cde_456", true)}},
       {{{Item("ghi", Item::kTokenType), {Param("jk", 4)}},
         {Item("l", Item::kTokenType), {}}},
        {Param("q", "9"), TokenParam("r", "w")}}}},
     "abc;a=1;b=2;cde_456, (ghi;jk=4 l);q=\"9\";r=w"},
    // key-generated.json
    {"0x00 in parameterised list key", "foo; a\000a=1", 10, std::nullopt,
     nullptr},
    {"0x01 in parameterised list key", "foo; a\001a=1", 10, std::nullopt,
     nullptr},
    {"0x02 in parameterised list key", "foo; a\002a=1", 10, std::nullopt,
     nullptr},
    {"0x03 in parameterised list key", "foo; a\003a=1", 10, std::nullopt,
     nullptr},
    {"0x04 in parameterised list key", "foo; a\004a=1", 10, std::nullopt,
     nullptr},
    {"0x05 in parameterised list key", "foo; a\005a=1", 10, std::nullopt,
     nullptr},
    {"0x06 in parameterised list key", "foo; a\006a=1", 10, std::nullopt,
     nullptr},
    {"0x07 in parameterised list key", "foo; a\aa=1", 10, std::nullopt,
     nullptr},
    {"0x08 in parameterised list key", "foo; a\ba=1", 10, std::nullopt,
     nullptr},
    {"0x09 in parameterised list key", "foo; a\ta=1", 10, std::nullopt,
     nullptr},
    {"0x0a in parameterised list key", "foo; a\na=1", 10, std::nullopt,
     nullptr},
    {"0x0b in parameterised list key", "foo; a\va=1", 10, std::nullopt,
     nullptr},
    {"0x0c in parameterised list key", "foo; a\fa=1", 10, std::nullopt,
     nullptr},
    {"0x0d in parameterised list key", "foo; a\ra=1", 10, std::nullopt,
     nullptr},
    {"0x0e in parameterised list key", "foo; a\016a=1", 10, std::nullopt,
     nullptr},
    {"0x0f in parameterised list key", "foo; a\017a=1", 10, std::nullopt,
     nullptr},
    {"0x10 in parameterised list key", "foo; a\020a=1", 10, std::nullopt,
     nullptr},
    {"0x11 in parameterised list key", "foo; a\021a=1", 10, std::nullopt,
     nullptr},
    {"0x12 in parameterised list key", "foo; a\022a=1", 10, std::nullopt,
     nullptr},
    {"0x13 in parameterised list key", "foo; a\023a=1", 10, std::nullopt,
     nullptr},
    {"0x14 in parameterised list key", "foo; a\024a=1", 10, std::nullopt,
     nullptr},
    {"0x15 in parameterised list key", "foo; a\025a=1", 10, std::nullopt,
     nullptr},
    {"0x16 in parameterised list key", "foo; a\026a=1", 10, std::nullopt,
     nullptr},
    {"0x17 in parameterised list key", "foo; a\027a=1", 10, std::nullopt,
     nullptr},
    {"0x18 in parameterised list key", "foo; a\030a=1", 10, std::nullopt,
     nullptr},
    {"0x19 in parameterised list key", "foo; a\031a=1", 10, std::nullopt,
     nullptr},
    {"0x1a in parameterised list key", "foo; a\032a=1", 10, std::nullopt,
     nullptr},
    {"0x1b in parameterised list key", "foo; a\033a=1", 10, std::nullopt,
     nullptr},
    {"0x1c in parameterised list key", "foo; a\034a=1", 10, std::nullopt,
     nullptr},
    {"0x1d in parameterised list key", "foo; a\035a=1", 10, std::nullopt,
     nullptr},
    {"0x1e in parameterised list key", "foo; a\036a=1", 10, std::nullopt,
     nullptr},
    {"0x1f in parameterised list key", "foo; a\037a=1", 10, std::nullopt,
     nullptr},
    {"0x20 in parameterised list key", "foo; a a=1", 10, std::nullopt, nullptr},
    {"0x21 in parameterised list key", "foo; a!a=1", 10, std::nullopt, nullptr},
    {"0x22 in parameterised list key", "foo; a\"a=1", 10, std::nullopt,
     nullptr},
    {"0x23 in parameterised list key", "foo; a#a=1", 10, std::nullopt, nullptr},
    {"0x24 in parameterised list key", "foo; a$a=1", 10, std::nullopt, nullptr},
    {"0x25 in parameterised list key", "foo; a%a=1", 10, std::nullopt, nullptr},
    {"0x26 in parameterised list key", "foo; a&a=1", 10, std::nullopt, nullptr},
    {"0x27 in parameterised list key", "foo; a'a=1", 10, std::nullopt, nullptr},
    {"0x28 in parameterised list key", "foo; a(a=1", 10, std::nullopt, nullptr},
    {"0x29 in parameterised list key", "foo; a)a=1", 10, std::nullopt, nullptr},
    {"0x2a in parameterised list key",
     "foo; a*a=1",
     10,
     {{{Item("foo", Item::kTokenType), {Param("a*a", 1)}}}},
     "foo;a*a=1"},
    {"0x2b in parameterised list key", "foo; a+a=1", 10, std::nullopt, nullptr},
    {"0x2c in parameterised list key", "foo; a,a=1", 10, std::nullopt, nullptr},
    {"0x2d in parameterised list key",
     "foo; a-a=1",
     10,
     {{{Item("foo", Item::kTokenType), {Param("a-a", 1)}}}},
     "foo;a-a=1"},
    {"0x2e in parameterised list key",
     "foo; a.a=1",
     10,
     {{{Item("foo", Item::kTokenType), {Param("a.a", 1)}}}},
     "foo;a.a=1"},
    {"0x2f in parameterised list key", "foo; a/a=1", 10, std::nullopt, nullptr},
    {"0x30 in parameterised list key",
     "foo; a0a=1",
     10,
     {{{Item("foo", Item::kTokenType), {Param("a0a", 1)}}}},
     "foo;a0a=1"},
    {"0x31 in parameterised list key",
     "foo; a1a=1",
     10,
     {{{Item("foo", Item::kTokenType), {Param("a1a", 1)}}}},
     "foo;a1a=1"},
    {"0x32 in parameterised list key",
     "foo; a2a=1",
     10,
     {{{Item("foo", Item::kTokenType), {Param("a2a", 1)}}}},
     "foo;a2a=1"},
    {"0x33 in parameterised list key",
     "foo; a3a=1",
     10,
     {{{Item("foo", Item::kTokenType), {Param("a3a", 1)}}}},
     "foo;a3a=1"},
    {"0x34 in parameterised list key",
     "foo; a4a=1",
     10,
     {{{Item("foo", Item::kTokenType), {Param("a4a", 1)}}}},
     "foo;a4a=1"},
    {"0x35 in parameterised list key",
     "foo; a5a=1",
     10,
     {{{Item("foo", Item::kTokenType), {Param("a5a", 1)}}}},
     "foo;a5a=1"},
    {"0x36 in parameterised list key",
     "foo; a6a=1",
     10,
     {{{Item("foo", Item::kTokenType), {Param("a6a", 1)}}}},
     "foo;a6a=1"},
    {"0x37 in parameterised list key",
     "foo; a7a=1",
     10,
     {{{Item("foo", Item::kTokenType), {Param("a7a", 1)}}}},
     "foo;a7a=1"},
    {"0x38 in parameterised list key",
     "foo; a8a=1",
     10,
     {{{Item("foo", Item::kTokenType), {Param("a8a", 1)}}}},
     "foo;a8a=1"},
    {"0x39 in parameterised list key",
     "foo; a9a=1",
     10,
     {{{Item("foo", Item::kTokenType), {Param("a9a", 1)}}}},
     "foo;a9a=1"},
    {"0x3a in parameterised list key", "foo; a:a=1", 10, std::nullopt, nullptr},
    {"0x3b in parameterised list key",
     "foo; a;a=1",
     10,
     {{{Item("foo", Item::kTokenType), {Param("a", 1)}}}},
     "foo;a=1"},
    {"0x3c in parameterised list key", "foo; a<a=1", 10, std::nullopt, nullptr},
    {"0x3d in parameterised list key", "foo; a=a=1", 10, std::nullopt, nullptr},
    {"0x3e in parameterised list key", "foo; a>a=1", 10, std::nullopt, nullptr},
    {"0x3f in parameterised list key", "foo; a?a=1", 10, std::nullopt, nullptr},
    {"0x40 in parameterised list key", "foo; a@a=1", 10, std::nullopt, nullptr},
    {"0x41 in parameterised list key", "foo; aAa=1", 10, std::nullopt, nullptr},
    {"0x42 in parameterised list key", "foo; aBa=1", 10, std::nullopt, nullptr},
    {"0x43 in parameterised list key", "foo; aCa=1", 10, std::nullopt, nullptr},
    {"0x44 in parameterised list key", "foo; aDa=1", 10, std::nullopt, nullptr},
    {"0x45 in parameterised list key", "foo; aEa=1", 10, std::nullopt, nullptr},
    {"0x46 in parameterised list key", "foo; aFa=1", 10, std::nullopt, nullptr},
    {"0x47 in parameterised list key", "foo; aGa=1", 10, std::nullopt, nullptr},
    {"0x48 in parameterised list key", "foo; aHa=1", 10, std::nullopt, nullptr},
    {"0x49 in parameterised list key", "foo; aIa=1", 10, std::nullopt, nullptr},
    {"0x4a in parameterised list key", "foo; aJa=1", 10, std::nullopt, nullptr},
    {"0x4b in parameterised list key", "foo; aKa=1", 10, std::nullopt, nullptr},
    {"0x4c in parameterised list key", "foo; aLa=1", 10, std::nullopt, nullptr},
    {"0x4d in parameterised list key", "foo; aMa=1", 10, std::nullopt, nullptr},
    {"0x4e in parameterised list key", "foo; aNa=1", 10, std::nullopt, nullptr},
    {"0x4f in parameterised list key", "foo; aOa=1", 10, std::nullopt, nullptr},
    {"0x50 in parameterised list key", "foo; aPa=1", 10, std::nullopt, nullptr},
    {"0x51 in parameterised list key", "foo; aQa=1", 10, std::nullopt, nullptr},
    {"0x52 in parameterised list key", "foo; aRa=1", 10, std::nullopt, nullptr},
    {"0x53 in parameterised list key", "foo; aSa=1", 10, std::nullopt, nullptr},
    {"0x54 in parameterised list key", "foo; aTa=1", 10, std::nullopt, nullptr},
    {"0x55 in parameterised list key", "foo; aUa=1", 10, std::nullopt, nullptr},
    {"0x56 in parameterised list key", "foo; aVa=1", 10, std::nullopt, nullptr},
    {"0x57 in parameterised list key", "foo; aWa=1", 10, std::nullopt, nullptr},
    {"0x58 in parameterised list key", "foo; aXa=1", 10, std::nullopt, nullptr},
    {"0x59 in parameterised list key", "foo; aYa=1", 10, std::nullopt, nullptr},
    {"0x5a in parameterised list key", "foo; aZa=1", 10, std::nullopt, nullptr},
    {"0x5b in parameterised list key", "foo; a[a=1", 10, std::nullopt, nullptr},
    {"0x5c in parameterised list key", "foo; a\\a=1", 10, std::nullopt,
     nullptr},
    {"0x5d in parameterised list key", "foo; a]a=1", 10, std::nullopt, nullptr},
    {"0x5e in parameterised list key", "foo; a^a=1", 10, std::nullopt, nullptr},
    {"0x5f in parameterised list key",
     "foo; a_a=1",
     10,
     {{{Item("foo", Item::kTokenType), {Param("a_a", 1)}}}},
     "foo;a_a=1"},
    {"0x60 in parameterised list key", "foo; a`a=1", 10, std::nullopt, nullptr},
    {"0x61 in parameterised list key",
     "foo; aaa=1",
     10,
     {{{Item("foo", Item::kTokenType), {Param("aaa", 1)}}}},
     "foo;aaa=1"},
    {"0x62 in parameterised list key",
     "foo; aba=1",
     10,
     {{{Item("foo", Item::kTokenType), {Param("aba", 1)}}}},
     "foo;aba=1"},
    {"0x63 in parameterised list key",
     "foo; aca=1",
     10,
     {{{Item("foo", Item::kTokenType), {Param("aca", 1)}}}},
     "foo;aca=1"},
    {"0x64 in parameterised list key",
     "foo; ada=1",
     10,
     {{{Item("foo", Item::kTokenType), {Param("ada", 1)}}}},
     "foo;ada=1"},
    {"0x65 in parameterised list key",
     "foo; aea=1",
     10,
     {{{Item("foo", Item::kTokenType), {Param("aea", 1)}}}},
     "foo;aea=1"},
    {"0x66 in parameterised list key",
     "foo; afa=1",
     10,
     {{{Item("foo", Item::kTokenType), {Param("afa", 1)}}}},
     "foo;afa=1"},
    {"0x67 in parameterised list key",
     "foo; aga=1",
     10,
     {{{Item("foo", Item::kTokenType), {Param("aga", 1)}}}},
     "foo;aga=1"},
    {"0x68 in parameterised list key",
     "foo; aha=1",
     10,
     {{{Item("foo", Item::kTokenType), {Param("aha", 1)}}}},
     "foo;aha=1"},
    {"0x69 in parameterised list key",
     "foo; aia=1",
     10,
     {{{Item("foo", Item::kTokenType), {Param("aia", 1)}}}},
     "foo;aia=1"},
    {"0x6a in parameterised list key",
     "foo; aja=1",
     10,
     {{{Item("foo", Item::kTokenType), {Param("aja", 1)}}}},
     "foo;aja=1"},
    {"0x6b in parameterised list key",
     "foo; aka=1",
     10,
     {{{Item("foo", Item::kTokenType), {Param("aka", 1)}}}},
     "foo;aka=1"},
    {"0x6c in parameterised list key",
     "foo; ala=1",
     10,
     {{{Item("foo", Item::kTokenType), {Param("ala", 1)}}}},
     "foo;ala=1"},
    {"0x6d in parameterised list key",
     "foo; ama=1",
     10,
     {{{Item("foo", Item::kTokenType), {Param("ama", 1)}}}},
     "foo;ama=1"},
    {"0x6e in parameterised list key",
     "foo; ana=1",
     10,
     {{{Item("foo", Item::kTokenType), {Param("ana", 1)}}}},
     "foo;ana=1"},
    {"0x6f in parameterised list key",
     "foo; aoa=1",
     10,
     {{{Item("foo", Item::kTokenType), {Param("aoa", 1)}}}},
     "foo;aoa=1"},
    {"0x70 in parameterised list key",
     "foo; apa=1",
     10,
     {{{Item("foo", Item::kTokenType), {Param("apa", 1)}}}},
     "foo;apa=1"},
    {"0x71 in parameterised list key",
     "foo; aqa=1",
     10,
     {{{Item("foo", Item::kTokenType), {Param("aqa", 1)}}}},
     "foo;aqa=1"},
    {"0x72 in parameterised list key",
     "foo; ara=1",
     10,
     {{{Item("foo", Item::kTokenType), {Param("ara", 1)}}}},
     "foo;ara=1"},
    {"0x73 in parameterised list key",
     "foo; asa=1",
     10,
     {{{Item("foo", Item::kTokenType), {Param("asa", 1)}}}},
     "foo;asa=1"},
    {"0x74 in parameterised list key",
     "foo; ata=1",
     10,
     {{{Item("foo", Item::kTokenType), {Param("ata", 1)}}}},
     "foo;ata=1"},
    {"0x75 in parameterised list key",
     "foo; aua=1",
     10,
     {{{Item("foo", Item::kTokenType), {Param("aua", 1)}}}},
     "foo;aua=1"},
    {"0x76 in parameterised list key",
     "foo; ava=1",
     10,
     {{{Item("foo", Item::kTokenType), {Param("ava", 1)}}}},
     "foo;ava=1"},
    {"0x77 in parameterised list key",
     "foo; awa=1",
     10,
     {{{Item("foo", Item::kTokenType), {Param("awa", 1)}}}},
     "foo;awa=1"},
    {"0x78 in parameterised list key",
     "foo; axa=1",
     10,
     {{{Item("foo", Item::kTokenType), {Param("axa", 1)}}}},
     "foo;axa=1"},
    {"0x79 in parameterised list key",
     "foo; aya=1",
     10,
     {{{Item("foo", Item::kTokenType), {Param("aya", 1)}}}},
     "foo;aya=1"},
    {"0x7a in parameterised list key",
     "foo; aza=1",
     10,
     {{{Item("foo", Item::kTokenType), {Param("aza", 1)}}}},
     "foo;aza=1"},
    {"0x7b in parameterised list key", "foo; a{a=1", 10, std::nullopt, nullptr},
    {"0x7c in parameterised list key", "foo; a|a=1", 10, std::nullopt, nullptr},
    {"0x7d in parameterised list key", "foo; a}a=1", 10, std::nullopt, nullptr},
    {"0x7e in parameterised list key", "foo; a~a=1", 10, std::nullopt, nullptr},
    {"0x7f in parameterised list key", "foo; a\177a=1", 10, std::nullopt,
     nullptr},
    {"0x00 starting a parameterised list key", "foo; \000a=1", 9, std::nullopt,
     nullptr},
    {"0x01 starting a parameterised list key", "foo; \001a=1", 9, std::nullopt,
     nullptr},
    {"0x02 starting a parameterised list key", "foo; \002a=1", 9, std::nullopt,
     nullptr},
    {"0x03 starting a parameterised list key", "foo; \003a=1", 9, std::nullopt,
     nullptr},
    {"0x04 starting a parameterised list key", "foo; \004a=1", 9, std::nullopt,
     nullptr},
    {"0x05 starting a parameterised list key", "foo; \005a=1", 9, std::nullopt,
     nullptr},
    {"0x06 starting a parameterised list key", "foo; \006a=1", 9, std::nullopt,
     nullptr},
    {"0x07 starting a parameterised list key", "foo; \aa=1", 9, std::nullopt,
     nullptr},
    {"0x08 starting a parameterised list key", "foo; \ba=1", 9, std::nullopt,
     nullptr},
    {"0x09 starting a parameterised list key", "foo; \ta=1", 9, std::nullopt,
     nullptr},
    {"0x0a starting a parameterised list key", "foo; \na=1", 9, std::nullopt,
     nullptr},
    {"0x0b starting a parameterised list key", "foo; \va=1", 9, std::nullopt,
     nullptr},
    {"0x0c starting a parameterised list key", "foo; \fa=1", 9, std::nullopt,
     nullptr},
    {"0x0d starting a parameterised list key", "foo; \ra=1", 9, std::nullopt,
     nullptr},
    {"0x0e starting a parameterised list key", "foo; \016a=1", 9, std::nullopt,
     nullptr},
    {"0x0f starting a parameterised list key", "foo; \017a=1", 9, std::nullopt,
     nullptr},
    {"0x10 starting a parameterised list key", "foo; \020a=1", 9, std::nullopt,
     nullptr},
    {"0x11 starting a parameterised list key", "foo; \021a=1", 9, std::nullopt,
     nullptr},
    {"0x12 starting a parameterised list key", "foo; \022a=1", 9, std::nullopt,
     nullptr},
    {"0x13 starting a parameterised list key", "foo; \023a=1", 9, std::nullopt,
     nullptr},
    {"0x14 starting a parameterised list key", "foo; \024a=1", 9, std::nullopt,
     nullptr},
    {"0x15 starting a parameterised list key", "foo; \025a=1", 9, std::nullopt,
     nullptr},
    {"0x16 starting a parameterised list key", "foo; \026a=1", 9, std::nullopt,
     nullptr},
    {"0x17 starting a parameterised list key", "foo; \027a=1", 9, std::nullopt,
     nullptr},
    {"0x18 starting a parameterised list key", "foo; \030a=1", 9, std::nullopt,
     nullptr},
    {"0x19 starting a parameterised list key", "foo; \031a=1", 9, std::nullopt,
     nullptr},
    {"0x1a starting a parameterised list key", "foo; \032a=1", 9, std::nullopt,
     nullptr},
    {"0x1b starting a parameterised list key", "foo; \033a=1", 9, std::nullopt,
     nullptr},
    {"0x1c starting a parameterised list key", "foo; \034a=1", 9, std::nullopt,
     nullptr},
    {"0x1d starting a parameterised list key", "foo; \035a=1", 9, std::nullopt,
     nullptr},
    {"0x1e starting a parameterised list key", "foo; \036a=1", 9, std::nullopt,
     nullptr},
    {"0x1f starting a parameterised list key", "foo; \037a=1", 9, std::nullopt,
     nullptr},
    {"0x20 starting a parameterised list key",
     "foo;  a=1",
     9,
     {{{Item("foo", Item::kTokenType), {Param("a", 1)}}}},
     "foo;a=1"},
    {"0x21 starting a parameterised list key", "foo; !a=1", 9, std::nullopt,
     nullptr},
    {"0x22 starting a parameterised list key", "foo; \"a=1", 9, std::nullopt,
     nullptr},
    {"0x23 starting a parameterised list key", "foo; #a=1", 9, std::nullopt,
     nullptr},
    {"0x24 starting a parameterised list key", "foo; $a=1", 9, std::nullopt,
     nullptr},
    {"0x25 starting a parameterised list key", "foo; %a=1", 9, std::nullopt,
     nullptr},
    {"0x26 starting a parameterised list key", "foo; &a=1", 9, std::nullopt,
     nullptr},
    {"0x27 starting a parameterised list key", "foo; 'a=1", 9, std::nullopt,
     nullptr},
    {"0x28 starting a parameterised list key", "foo; (a=1", 9, std::nullopt,
     nullptr},
    {"0x29 starting a parameterised list key", "foo; )a=1", 9, std::nullopt,
     nullptr},
    {"0x2a starting a parameterised list key",
     "foo; *a=1",
     9,
     {{{Item("foo", Item::kTokenType), {Param("*a", 1)}}}},
     "foo;*a=1"},
    {"0x2b starting a parameterised list key", "foo; +a=1", 9, std::nullopt,
     nullptr},
    {"0x2c starting a parameterised list key", "foo; ,a=1", 9, std::nullopt,
     nullptr},
    {"0x2d starting a parameterised list key", "foo; -a=1", 9, std::nullopt,
     nullptr},
    {"0x2e starting a parameterised list key", "foo; .a=1", 9, std::nullopt,
     nullptr},
    {"0x2f starting a parameterised list key", "foo; /a=1", 9, std::nullopt,
     nullptr},
    {"0x30 starting a parameterised list key", "foo; 0a=1", 9, std::nullopt,
     nullptr},
    {"0x31 starting a parameterised list key", "foo; 1a=1", 9, std::nullopt,
     nullptr},
    {"0x32 starting a parameterised list key", "foo; 2a=1", 9, std::nullopt,
     nullptr},
    {"0x33 starting a parameterised list key", "foo; 3a=1", 9, std::nullopt,
     nullptr},
    {"0x34 starting a parameterised list key", "foo; 4a=1", 9, std::nullopt,
     nullptr},
    {"0x35 starting a parameterised list key", "foo; 5a=1", 9, std::nullopt,
     nullptr},
    {"0x36 starting a parameterised list key", "foo; 6a=1", 9, std::nullopt,
     nullptr},
    {"0x37 starting a parameterised list key", "foo; 7a=1", 9, std::nullopt,
     nullptr},
    {"0x38 starting a parameterised list key", "foo; 8a=1", 9, std::nullopt,
     nullptr},
    {"0x39 starting a parameterised list key", "foo; 9a=1", 9, std::nullopt,
     nullptr},
    {"0x3a starting a parameterised list key", "foo; :a=1", 9, std::nullopt,
     nullptr},
    {"0x3b starting a parameterised list key", "foo; ;a=1", 9, std::nullopt,
     nullptr},
    {"0x3c starting a parameterised list key", "foo; <a=1", 9, std::nullopt,
     nullptr},
    {"0x3d starting a parameterised list key", "foo; =a=1", 9, std::nullopt,
     nullptr},
    {"0x3e starting a parameterised list key", "foo; >a=1", 9, std::nullopt,
     nullptr},
    {"0x3f starting a parameterised list key", "foo; ?a=1", 9, std::nullopt,
     nullptr},
    {"0x40 starting a parameterised list key", "foo; @a=1", 9, std::nullopt,
     nullptr},
    {"0x41 starting a parameterised list key", "foo; Aa=1", 9, std::nullopt,
     nullptr},
    {"0x42 starting a parameterised list key", "foo; Ba=1", 9, std::nullopt,
     nullptr},
    {"0x43 starting a parameterised list key", "foo; Ca=1", 9, std::nullopt,
     nullptr},
    {"0x44 starting a parameterised list key", "foo; Da=1", 9, std::nullopt,
     nullptr},
    {"0x45 starting a parameterised list key", "foo; Ea=1", 9, std::nullopt,
     nullptr},
    {"0x46 starting a parameterised list key", "foo; Fa=1", 9, std::nullopt,
     nullptr},
    {"0x47 starting a parameterised list key", "foo; Ga=1", 9, std::nullopt,
     nullptr},
    {"0x48 starting a parameterised list key", "foo; Ha=1", 9, std::nullopt,
     nullptr},
    {"0x49 starting a parameterised list key", "foo; Ia=1", 9, std::nullopt,
     nullptr},
    {"0x4a starting a parameterised list key", "foo; Ja=1", 9, std::nullopt,
     nullptr},
    {"0x4b starting a parameterised list key", "foo; Ka=1", 9, std::nullopt,
     nullptr},
    {"0x4c starting a parameterised list key", "foo; La=1", 9, std::nullopt,
     nullptr},
    {"0x4d starting a parameterised list key", "foo; Ma=1", 9, std::nullopt,
     nullptr},
    {"0x4e starting a parameterised list key", "foo; Na=1", 9, std::nullopt,
     nullptr},
    {"0x4f starting a parameterised list key", "foo; Oa=1", 9, std::nullopt,
     nullptr},
    {"0x50 starting a parameterised list key", "foo; Pa=1", 9, std::nullopt,
     nullptr},
    {"0x51 starting a parameterised list key", "foo; Qa=1", 9, std::nullopt,
     nullptr},
    {"0x52 starting a parameterised list key", "foo; Ra=1", 9, std::nullopt,
     nullptr},
    {"0x53 starting a parameterised list key", "foo; Sa=1", 9, std::nullopt,
     nullptr},
    {"0x54 starting a parameterised list key", "foo; Ta=1", 9, std::nullopt,
     nullptr},
    {"0x55 starting a parameterised list key", "foo; Ua=1", 9, std::nullopt,
     nullptr},
    {"0x56 starting a parameterised list key", "foo; Va=1", 9, std::nullopt,
     nullptr},
    {"0x57 starting a parameterised list key", "foo; Wa=1", 9, std::nullopt,
     nullptr},
    {"0x58 starting a parameterised list key", "foo; Xa=1", 9, std::nullopt,
     nullptr},
    {"0x59 starting a parameterised list key", "foo; Ya=1", 9, std::nullopt,
     nullptr},
    {"0x5a starting a parameterised list key", "foo; Za=1", 9, std::nullopt,
     nullptr},
    {"0x5b starting a parameterised list key", "foo; [a=1", 9, std::nullopt,
     nullptr},
    {"0x5c starting a parameterised list key", "foo; \\a=1", 9, std::nullopt,
     nullptr},
    {"0x5d starting a parameterised list key", "foo; ]a=1", 9, std::nullopt,
     nullptr},
    {"0x5e starting a parameterised list key", "foo; ^a=1", 9, std::nullopt,
     nullptr},
    {"0x5f starting a parameterised list key", "foo; _a=1", 9, std::nullopt,
     nullptr},
    {"0x60 starting a parameterised list key", "foo; `a=1", 9, std::nullopt,
     nullptr},
    {"0x61 starting a parameterised list key",
     "foo; aa=1",
     9,
     {{{Item("foo", Item::kTokenType), {Param("aa", 1)}}}},
     "foo;aa=1"},
    {"0x62 starting a parameterised list key",
     "foo; ba=1",
     9,
     {{{Item("foo", Item::kTokenType), {Param("ba", 1)}}}},
     "foo;ba=1"},
    {"0x63 starting a parameterised list key",
     "foo; ca=1",
     9,
     {{{Item("foo", Item::kTokenType), {Param("ca", 1)}}}},
     "foo;ca=1"},
    {"0x64 starting a parameterised list key",
     "foo; da=1",
     9,
     {{{Item("foo", Item::kTokenType), {Param("da", 1)}}}},
     "foo;da=1"},
    {"0x65 starting a parameterised list key",
     "foo; ea=1",
     9,
     {{{Item("foo", Item::kTokenType), {Param("ea", 1)}}}},
     "foo;ea=1"},
    {"0x66 starting a parameterised list key",
     "foo; fa=1",
     9,
     {{{Item("foo", Item::kTokenType), {Param("fa", 1)}}}},
     "foo;fa=1"},
    {"0x67 starting a parameterised list key",
     "foo; ga=1",
     9,
     {{{Item("foo", Item::kTokenType), {Param("ga", 1)}}}},
     "foo;ga=1"},
    {"0x68 starting a parameterised list key",
     "foo; ha=1",
     9,
     {{{Item("foo", Item::kTokenType), {Param("ha", 1)}}}},
     "foo;ha=1"},
    {"0x69 starting a parameterised list key",
     "foo; ia=1",
     9,
     {{{Item("foo", Item::kTokenType), {Param("ia", 1)}}}},
     "foo;ia=1"},
    {"0x6a starting a parameterised list key",
     "foo; ja=1",
     9,
     {{{Item("foo", Item::kTokenType), {Param("ja", 1)}}}},
     "foo;ja=1"},
    {"0x6b starting a parameterised list key",
     "foo; ka=1",
     9,
     {{{Item("foo", Item::kTokenType), {Param("ka", 1)}}}},
     "foo;ka=1"},
    {"0x6c starting a parameterised list key",
     "foo; la=1",
     9,
     {{{Item("foo", Item::kTokenType), {Param("la", 1)}}}},
     "foo;la=1"},
    {"0x6d starting a parameterised list key",
     "foo; ma=1",
     9,
     {{{Item("foo", Item::kTokenType), {Param("ma", 1)}}}},
     "foo;ma=1"},
    {"0x6e starting a parameterised list key",
     "foo; na=1",
     9,
     {{{Item("foo", Item::kTokenType), {Param("na", 1)}}}},
     "foo;na=1"},
    {"0x6f starting a parameterised list key",
     "foo; oa=1",
     9,
     {{{Item("foo", Item::kTokenType), {Param("oa", 1)}}}},
     "foo;oa=1"},
    {"0x70 starting a parameterised list key",
     "foo; pa=1",
     9,
     {{{Item("foo", Item::kTokenType), {Param("pa", 1)}}}},
     "foo;pa=1"},
    {"0x71 starting a parameterised list key",
     "foo; qa=1",
     9,
     {{{Item("foo", Item::kTokenType), {Param("qa", 1)}}}},
     "foo;qa=1"},
    {"0x72 starting a parameterised list key",
     "foo; ra=1",
     9,
     {{{Item("foo", Item::kTokenType), {Param("ra", 1)}}}},
     "foo;ra=1"},
    {"0x73 starting a parameterised list key",
     "foo; sa=1",
     9,
     {{{Item("foo", Item::kTokenType), {Param("sa", 1)}}}},
     "foo;sa=1"},
    {"0x74 starting a parameterised list key",
     "foo; ta=1",
     9,
     {{{Item("foo", Item::kTokenType), {Param("ta", 1)}}}},
     "foo;ta=1"},
    {"0x75 starting a parameterised list key",
     "foo; ua=1",
     9,
     {{{Item("foo", Item::kTokenType), {Param("ua", 1)}}}},
     "foo;ua=1"},
    {"0x76 starting a parameterised list key",
     "foo; va=1",
     9,
     {{{Item("foo", Item::kTokenType), {Param("va", 1)}}}},
     "foo;va=1"},
    {"0x77 starting a parameterised list key",
     "foo; wa=1",
     9,
     {{{Item("foo", Item::kTokenType), {Param("wa", 1)}}}},
     "foo;wa=1"},
    {"0x78 starting a parameterised list key",
     "foo; xa=1",
     9,
     {{{Item("foo", Item::kTokenType), {Param("xa", 1)}}}},
     "foo;xa=1"},
    {"0x79 starting a parameterised list key",
     "foo; ya=1",
     9,
     {{{Item("foo", Item::kTokenType), {Param("ya", 1)}}}},
     "foo;ya=1"},
    {"0x7a starting a parameterised list key",
     "foo; za=1",
     9,
     {{{Item("foo", Item::kTokenType), {Param("za", 1)}}}},
     "foo;za=1"},
    {"0x7b starting a parameterised list key", "foo; {a=1", 9, std::nullopt,
     nullptr},
    {"0x7c starting a parameterised list key", "foo; |a=1", 9, std::nullopt,
     nullptr},
    {"0x7d starting a parameterised list key", "foo; }a=1", 9, std::nullopt,
     nullptr},
    {"0x7e starting a parameterised list key", "foo; ~a=1", 9, std::nullopt,
     nullptr},
    {"0x7f starting a parameterised list key", "foo; \177a=1", 9, std::nullopt,
     nullptr},
    // list.json
    {"basic list",
     "1, 42",
     5,
     {{{Integer(1), {}}, {Integer(42), {}}}},
     nullptr},
    {"empty list", "", 0, {List()}, nullptr},
    {"leading SP list",
     "  42, 43",
     8,
     {{{Integer(42), {}}, {Integer(43), {}}}},
     "42, 43"},
    {"single item list", "42", 2, {{{Integer(42), {}}}}, nullptr},
    {"no whitespace list",
     "1,42",
     4,
     {{{Integer(1), {}}, {Integer(42), {}}}},
     "1, 42"},
    {"extra whitespace list",
     "1 , 42",
     6,
     {{{Integer(1), {}}, {Integer(42), {}}}},
     "1, 42"},
    {"tab separated list",
     "1\t,\t42",
     6,
     {{{Integer(1), {}}, {Integer(42), {}}}},
     "1, 42"},
    {"two line list",
     "1, 42",
     5,
     {{{Integer(1), {}}, {Integer(42), {}}}},
     "1, 42"},
    {"trailing comma list", "1, 42,", 6, std::nullopt, nullptr},
    {"empty item list", "1,,42", 5, std::nullopt, nullptr},
    {"empty item list (multiple field lines)", "1, , 42", 7, std::nullopt,
     nullptr},
    // listlist.json
    {"basic list of lists",
     "(1 2), (42 43)",
     14,
     {{{{{Integer(1), {}}, {Integer(2), {}}}, {}},
       {{{Integer(42), {}}, {Integer(43), {}}}, {}}}},
     nullptr},
    {"single item list of lists",
     "(42)",
     4,
     {{{{{Integer(42), {}}}, {}}}},
     nullptr},
    {"empty item list of lists",
     "()",
     2,
     {{{std::vector<ParameterizedItem>(), {}}}},
     nullptr},
    {"empty middle item list of lists",
     "(1),(),(42)",
     11,
     {{{{{Integer(1), {}}}, {}},
       {std::vector<ParameterizedItem>(), {}},
       {{{Integer(42), {}}}, {}}}},
     "(1), (), (42)"},
    {"extra whitespace list of lists",
     "(  1  42  )",
     11,
     {{{{{Integer(1), {}}, {Integer(42), {}}}, {}}}},
     "(1 42)"},
    {"wrong whitespace list of lists", "(1\t 42)", 7, std::nullopt, nullptr},
    {"no trailing parenthesis list of lists", "(1 42", 5, std::nullopt,
     nullptr},
    {"no trailing parenthesis middle list of lists", "(1 2, (42 43)", 13,
     std::nullopt, nullptr},
    {"no spaces in inner-list", "(abc\"def\"?0123*dXZ3*xyz)", 24, std::nullopt,
     nullptr},
    {"no closing parenthesis", "(", 1, std::nullopt, nullptr},
    // param-list.json
    {"basic parameterised list",
     "abc_123;a=1;b=2; cdef_456, ghi;q=9;r=\"+w\"",
     41,
     {{{Item("abc_123", Item::kTokenType),
        {Param("a", 1), Param("b", 2), BooleanParam("cdef_456", true)}},
       {Item("ghi", Item::kTokenType), {Param("q", 9), Param("r", "+w")}}}},
     "abc_123;a=1;b=2;cdef_456, ghi;q=9;r=\"+w\""},
    {"single item parameterised list",
     "text/html;q=1.0",
     15,
     {{{Item("text/html", Item::kTokenType), {DoubleParam("q", 1.000000)}}}},
     nullptr},
    {"missing parameter value parameterised list",
     "text/html;a;q=1.0",
     17,
     {{{Item("text/html", Item::kTokenType),
        {BooleanParam("a", true), DoubleParam("q", 1.000000)}}}},
     nullptr},
    {"missing terminal parameter value parameterised list",
     "text/html;q=1.0;a",
     17,
     {{{Item("text/html", Item::kTokenType),
        {DoubleParam("q", 1.000000), BooleanParam("a", true)}}}},
     nullptr},
    {"no whitespace parameterised list",
     "text/html,text/plain;q=0.5",
     26,
     {{{Item("text/html", Item::kTokenType), {}},
       {Item("text/plain", Item::kTokenType), {DoubleParam("q", 0.500000)}}}},
     "text/html, text/plain;q=0.5"},
    {"whitespace before = parameterised list", "text/html, text/plain;q =0.5",
     28, std::nullopt, nullptr},
    {"whitespace after = parameterised list", "text/html, text/plain;q= 0.5",
     28, std::nullopt, nullptr},
    {"whitespace before ; parameterised list", "text/html, text/plain ;q=0.5",
     28, std::nullopt, nullptr},
    {"whitespace after ; parameterised list",
     "text/html, text/plain; q=0.5",
     28,
     {{{Item("text/html", Item::kTokenType), {}},
       {Item("text/plain", Item::kTokenType), {DoubleParam("q", 0.500000)}}}},
     "text/html, text/plain;q=0.5"},
    {"extra whitespace parameterised list",
     "text/html  ,  text/plain;  q=0.5;  charset=utf-8",
     48,
     {{{Item("text/html", Item::kTokenType), {}},
       {Item("text/plain", Item::kTokenType),
        {DoubleParam("q", 0.500000), TokenParam("charset", "utf-8")}}}},
     "text/html, text/plain;q=0.5;charset=utf-8"},
    {"two lines parameterised list",
     "text/html, text/plain;q=0.5",
     27,
     {{{Item("text/html", Item::kTokenType), {}},
       {Item("text/plain", Item::kTokenType), {DoubleParam("q", 0.500000)}}}},
     "text/html, text/plain;q=0.5"},
    {"trailing comma parameterised list", "text/html,text/plain;q=0.5,", 27,
     std::nullopt, nullptr},
    {"empty item parameterised list", "text/html,,text/plain;q=0.5,", 28,
     std::nullopt, nullptr},
    // param-listlist.json
    {"parameterised inner list",
     "(abc_123);a=1;b=2, cdef_456",
     27,
     {{{{{Item("abc_123", Item::kTokenType), {}}},
        {Param("a", 1), Param("b", 2)}},
       {Item("cdef_456", Item::kTokenType), {}}}},
     nullptr},
    {"parameterised inner list item",
     "(abc_123;a=1;b=2;cdef_456)",
     26,
     {{{{{Item("abc_123", Item::kTokenType),
          {Param("a", 1), Param("b", 2), BooleanParam("cdef_456", true)}}},
        {}}}},
     nullptr},
    {"parameterised inner list with parameterised item",
     "(abc_123;a=1;b=2);cdef_456",
     26,
     {{{{{Item("abc_123", Item::kTokenType), {Param("a", 1), Param("b", 2)}}},
        {BooleanParam("cdef_456", true)}}}},
     nullptr},
    // token.json
    {"basic token - list",
     "a_b-c3/*",
     8,
     {{{Item("a_b-c3/*", Item::kTokenType), {}}}},
     nullptr},
    {"token with capitals - list",
     "fooBar",
     6,
     {{{Item("fooBar", Item::kTokenType), {}}}},
     nullptr},
    {"token starting with capitals - list",
     "FooBar",
     6,
     {{{Item("FooBar", Item::kTokenType), {}}}},
     nullptr},
};

const struct DictionaryTestCase {
  const char* name;
  const char* raw;
  size_t raw_len;
  const std::optional<Dictionary>
      expected;           // nullopt if parse error is expected.
  const char* canonical;  // nullptr if parse error is expected, or if canonical
                          // format is identical to raw.
} dictionary_test_cases[] = {
    // dictionary.json
    {"basic dictionary",
     "en=\"Applepie\", da=:w4ZibGV0w6ZydGUK:",
     36,
     {Dictionary{
         {{"en", {Item("Applepie"), {}}},
          {"da",
           {Item("\303\206blet\303\246rte\n", Item::kByteSequenceType), {}}}}}},
     nullptr},
    {"empty dictionary", "", 0, {Dictionary{{}}}, nullptr},
    {"single item dictionary",
     "a=1",
     3,
     {Dictionary{{{"a", {Integer(1), {}}}}}},
     nullptr},
    {"list item dictionary",
     "a=(1 2)",
     7,
     {Dictionary{{{"a", {{{Integer(1), {}}, {Integer(2), {}}}, {}}}}}},
     nullptr},
    {"single list item dictionary",
     "a=(1)",
     5,
     {Dictionary{{{"a", {{{Integer(1), {}}}, {}}}}}},
     nullptr},
    {"empty list item dictionary",
     "a=()",
     4,
     {Dictionary{{{"a", {std::vector<ParameterizedItem>(), {}}}}}},
     nullptr},
    {"no whitespace dictionary",
     "a=1,b=2",
     7,
     {Dictionary{{{"a", {Integer(1), {}}}, {"b", {Integer(2), {}}}}}},
     "a=1, b=2"},
    {"extra whitespace dictionary",
     "a=1 ,  b=2",
     10,
     {Dictionary{{{"a", {Integer(1), {}}}, {"b", {Integer(2), {}}}}}},
     "a=1, b=2"},
    {"tab separated dictionary",
     "a=1\t,\tb=2",
     9,
     {Dictionary{{{"a", {Integer(1), {}}}, {"b", {Integer(2), {}}}}}},
     "a=1, b=2"},
    {"leading whitespace dictionary",
     "     a=1 ,  b=2",
     15,
     {Dictionary{{{"a", {Integer(1), {}}}, {"b", {Integer(2), {}}}}}},
     "a=1, b=2"},
    {"whitespace before = dictionary", "a =1, b=2", 9, std::nullopt, nullptr},
    {"whitespace after = dictionary", "a=1, b= 2", 9, std::nullopt, nullptr},
    {"two lines dictionary",
     "a=1, b=2",
     8,
     {Dictionary{{{"a", {Integer(1), {}}}, {"b", {Integer(2), {}}}}}},
     "a=1, b=2"},
    {"missing value dictionary",
     "a=1, b, c=3",
     11,
     {Dictionary{{{"a", {Integer(1), {}}},
                  {"b", {Item(true), {}}},
                  {"c", {Integer(3), {}}}}}},
     nullptr},
    {"all missing value dictionary",
     "a, b, c",
     7,
     {Dictionary{{{"a", {Item(true), {}}},
                  {"b", {Item(true), {}}},
                  {"c", {Item(true), {}}}}}},
     nullptr},
    {"start missing value dictionary",
     "a, b=2",
     6,
     {Dictionary{{{"a", {Item(true), {}}}, {"b", {Integer(2), {}}}}}},
     nullptr},
    {"end missing value dictionary",
     "a=1, b",
     6,
     {Dictionary{{{"a", {Integer(1), {}}}, {"b", {Item(true), {}}}}}},
     nullptr},
    {"missing value with params dictionary",
     "a=1, b;foo=9, c=3",
     17,
     {Dictionary{{{"a", {Integer(1), {}}},
                  {"b", {Item(true), {Param("foo", 9)}}},
                  {"c", {Integer(3), {}}}}}},
     nullptr},
    {"explicit true value with params dictionary",
     "a=1, b=?1;foo=9, c=3",
     20,
     {Dictionary{{{"a", {Integer(1), {}}},
                  {"b", {Item(true), {Param("foo", 9)}}},
                  {"c", {Integer(3), {}}}}}},
     "a=1, b;foo=9, c=3"},
    {"trailing comma dictionary", "a=1, b=2,", 9, std::nullopt, nullptr},
    {"empty item dictionary", "a=1,,b=2,", 9, std::nullopt, nullptr},
    {"duplicate key dictionary",
     "a=1,b=2,a=3",
     11,
     {Dictionary{{{"a", {Integer(3), {}}}, {"b", {Integer(2), {}}}}}},
     "a=3, b=2"},
    {"numeric key dictionary", "a=1,1b=2,a=1", 12, std::nullopt, nullptr},
    {"uppercase key dictionary", "a=1,B=2,a=1", 11, std::nullopt, nullptr},
    {"bad key dictionary", "a=1,b!=2,a=1", 12, std::nullopt, nullptr},
    // examples.json
    {"Example-DictHeader",
     "en=\"Applepie\", da=:w4ZibGV0w6ZydGU=:",
     36,
     {Dictionary{
         {{"en", {Item("Applepie"), {}}},
          {"da",
           {Item("\303\206blet\303\246rte", Item::kByteSequenceType), {}}}}}},
     nullptr},
    {"Example-DictHeader (boolean values)",
     "a=?0, b, c; foo=bar",
     19,
     {Dictionary{{{"a", {Item(false), {}}},
                  {"b", {Item(true), {}}},
                  {"c", {Item(true), {TokenParam("foo", "bar")}}}}}},
     "a=?0, b, c;foo=bar"},
    {"Example-DictListHeader",
     "rating=1.5, feelings=(joy sadness)",
     34,
     {Dictionary{{{"rating", {Item(1.500000), {}}},
                  {"feelings",
                   {{{Item("joy", Item::kTokenType), {}},
                     {Item("sadness", Item::kTokenType), {}}},
                    {}}}}}},
     nullptr},
    {"Example-MixDict",
     "a=(1 2), b=3, c=4;aa=bb, d=(5 6);valid",
     38,
     {Dictionary{{{"a", {{{Integer(1), {}}, {Integer(2), {}}}, {}}},
                  {"b", {Integer(3), {}}},
                  {"c", {Integer(4), {TokenParam("aa", "bb")}}},
                  {"d",
                   {{{Integer(5), {}}, {Integer(6), {}}},
                    {BooleanParam("valid", true)}}}}}},
     "a=(1 2), b=3, c=4;aa=bb, d=(5 6);valid"},
    {"Example-Hdr (dictionary on one line)",
     "foo=1, bar=2",
     12,
     {Dictionary{{{"foo", {Integer(1), {}}}, {"bar", {Integer(2), {}}}}}},
     nullptr},
    {"Example-Hdr (dictionary on two lines)",
     "foo=1, bar=2",
     12,
     {Dictionary{{{"foo", {Integer(1), {}}}, {"bar", {Integer(2), {}}}}}},
     "foo=1, bar=2"},
    // key-generated.json
    {"0x00 as a single-character dictionary key", "\000=1", 3, std::nullopt,
     nullptr},
    {"0x01 as a single-character dictionary key", "\001=1", 3, std::nullopt,
     nullptr},
    {"0x02 as a single-character dictionary key", "\002=1", 3, std::nullopt,
     nullptr},
    {"0x03 as a single-character dictionary key", "\003=1", 3, std::nullopt,
     nullptr},
    {"0x04 as a single-character dictionary key", "\004=1", 3, std::nullopt,
     nullptr},
    {"0x05 as a single-character dictionary key", "\005=1", 3, std::nullopt,
     nullptr},
    {"0x06 as a single-character dictionary key", "\006=1", 3, std::nullopt,
     nullptr},
    {"0x07 as a single-character dictionary key", "\a=1", 3, std::nullopt,
     nullptr},
    {"0x08 as a single-character dictionary key", "\b=1", 3, std::nullopt,
     nullptr},
    {"0x09 as a single-character dictionary key", "\t=1", 3, std::nullopt,
     nullptr},
    {"0x0a as a single-character dictionary key", "\n=1", 3, std::nullopt,
     nullptr},
    {"0x0b as a single-character dictionary key", "\v=1", 3, std::nullopt,
     nullptr},
    {"0x0c as a single-character dictionary key", "\f=1", 3, std::nullopt,
     nullptr},
    {"0x0d as a single-character dictionary key", "\r=1", 3, std::nullopt,
     nullptr},
    {"0x0e as a single-character dictionary key", "\016=1", 3, std::nullopt,
     nullptr},
    {"0x0f as a single-character dictionary key", "\017=1", 3, std::nullopt,
     nullptr},
    {"0x10 as a single-character dictionary key", "\020=1", 3, std::nullopt,
     nullptr},
    {"0x11 as a single-character dictionary key", "\021=1", 3, std::nullopt,
     nullptr},
    {"0x12 as a single-character dictionary key", "\022=1", 3, std::nullopt,
     nullptr},
    {"0x13 as a single-character dictionary key", "\023=1", 3, std::nullopt,
     nullptr},
    {"0x14 as a single-character dictionary key", "\024=1", 3, std::nullopt,
     nullptr},
    {"0x15 as a single-character dictionary key", "\025=1", 3, std::nullopt,
     nullptr},
    {"0x16 as a single-character dictionary key", "\026=1", 3, std::nullopt,
     nullptr},
    {"0x17 as a single-character dictionary key", "\027=1", 3, std::nullopt,
     nullptr},
    {"0x18 as a single-character dictionary key", "\030=1", 3, std::nullopt,
     nullptr},
    {"0x19 as a single-character dictionary key", "\031=1", 3, std::nullopt,
     nullptr},
    {"0x1a as a single-character dictionary key", "\032=1", 3, std::nullopt,
     nullptr},
    {"0x1b as a single-character dictionary key", "\033=1", 3, std::nullopt,
     nullptr},
    {"0x1c as a single-character dictionary key", "\034=1", 3, std::nullopt,
     nullptr},
    {"0x1d as a single-character dictionary key", "\035=1", 3, std::nullopt,
     nullptr},
    {"0x1e as a single-character dictionary key", "\036=1", 3, std::nullopt,
     nullptr},
    {"0x1f as a single-character dictionary key", "\037=1", 3, std::nullopt,
     nullptr},
    {"0x20 as a single-character dictionary key", "=1", 2, std::nullopt,
     nullptr},
    {"0x21 as a single-character dictionary key", "!=1", 3, std::nullopt,
     nullptr},
    {"0x22 as a single-character dictionary key", "\"=1", 3, std::nullopt,
     nullptr},
    {"0x23 as a single-character dictionary key", "#=1", 3, std::nullopt,
     nullptr},
    {"0x24 as a single-character dictionary key", "$=1", 3, std::nullopt,
     nullptr},
    {"0x25 as a single-character dictionary key", "%=1", 3, std::nullopt,
     nullptr},
    {"0x26 as a single-character dictionary key", "&=1", 3, std::nullopt,
     nullptr},
    {"0x27 as a single-character dictionary key", "'=1", 3, std::nullopt,
     nullptr},
    {"0x28 as a single-character dictionary key", "(=1", 3, std::nullopt,
     nullptr},
    {"0x29 as a single-character dictionary key", ")=1", 3, std::nullopt,
     nullptr},
    {"0x2a as a single-character dictionary key",
     "*=1",
     3,
     {Dictionary{{{"*", {Integer(1), {}}}}}},
     nullptr},
    {"0x2b as a single-character dictionary key", "+=1", 3, std::nullopt,
     nullptr},
    {"0x2c as a single-character dictionary key", ",=1", 3, std::nullopt,
     nullptr},
    {"0x2d as a single-character dictionary key", "-=1", 3, std::nullopt,
     nullptr},
    {"0x2e as a single-character dictionary key", ".=1", 3, std::nullopt,
     nullptr},
    {"0x2f as a single-character dictionary key", "/=1", 3, std::nullopt,
     nullptr},
    {"0x30 as a single-character dictionary key", "0=1", 3, std::nullopt,
     nullptr},
    {"0x31 as a single-character dictionary key", "1=1", 3, std::nullopt,
     nullptr},
    {"0x32 as a single-character dictionary key", "2=1", 3, std::nullopt,
     nullptr},
    {"0x33 as a single-character dictionary key", "3=1", 3, std::nullopt,
     nullptr},
    {"0x34 as a single-character dictionary key", "4=1", 3, std::nullopt,
     nullptr},
    {"0x35 as a single-character dictionary key", "5=1", 3, std::nullopt,
     nullptr},
    {"0x36 as a single-character dictionary key", "6=1", 3, std::nullopt,
     nullptr},
    {"0x37 as a single-character dictionary key", "7=1", 3, std::nullopt,
     nullptr},
    {"0x38 as a single-character dictionary key", "8=1", 3, std::nullopt,
     nullptr},
    {"0x39 as a single-character dictionary key", "9=1", 3, std::nullopt,
     nullptr},
    {"0x3a as a single-character dictionary key", ":=1", 3, std::nullopt,
     nullptr},
    {"0x3b as a single-character dictionary key", ";=1", 3, std::nullopt,
     nullptr},
    {"0x3c as a single-character dictionary key", "<=1", 3, std::nullopt,
     nullptr},
    {"0x3d as a single-character dictionary key", "==1", 3, std::nullopt,
     nullptr},
    {"0x3e as a single-character dictionary key", ">=1", 3, std::nullopt,
     nullptr},
    {"0x3f as a single-character dictionary key", "?=1", 3, std::nullopt,
     nullptr},
    {"0x40 as a single-character dictionary key", "@=1", 3, std::nullopt,
     nullptr},
    {"0x41 as a single-character dictionary key", "A=1", 3, std::nullopt,
     nullptr},
    {"0x42 as a single-character dictionary key", "B=1", 3, std::nullopt,
     nullptr},
    {"0x43 as a single-character dictionary key", "C=1", 3, std::nullopt,
     nullptr},
    {"0x44 as a single-character dictionary key", "D=1", 3, std::nullopt,
     nullptr},
    {"0x45 as a single-character dictionary key", "E=1", 3, std::nullopt,
     nullptr},
    {"0x46 as a single-character dictionary key", "F=1", 3, std::nullopt,
     nullptr},
    {"0x47 as a single-character dictionary key", "G=1", 3, std::nullopt,
     nullptr},
    {"0x48 as a single-character dictionary key", "H=1", 3, std::nullopt,
     nullptr},
    {"0x49 as a single-character dictionary key", "I=1", 3, std::nullopt,
     nullptr},
    {"0x4a as a single-character dictionary key", "J=1", 3, std::nullopt,
     nullptr},
    {"0x4b as a single-character dictionary key", "K=1", 3, std::nullopt,
     nullptr},
    {"0x4c as a single-character dictionary key", "L=1", 3, std::nullopt,
     nullptr},
    {"0x4d as a single-character dictionary key", "M=1", 3, std::nullopt,
     nullptr},
    {"0x4e as a single-character dictionary key", "N=1", 3, std::nullopt,
     nullptr},
    {"0x4f as a single-character dictionary key", "O=1", 3, std::nullopt,
     nullptr},
    {"0x50 as a single-character dictionary key", "P=1", 3, std::nullopt,
     nullptr},
    {"0x51 as a single-character dictionary key", "Q=1", 3, std::nullopt,
     nullptr},
    {"0x52 as a single-character dictionary key", "R=1", 3, std::nullopt,
     nullptr},
    {"0x53 as a single-character dictionary key", "S=1", 3, std::nullopt,
     nullptr},
    {"0x54 as a single-character dictionary key", "T=1", 3, std::nullopt,
     nullptr},
    {"0x55 as a single-character dictionary key", "U=1", 3, std::nullopt,
     nullptr},
    {"0x56 as a single-character dictionary key", "V=1", 3, std::nullopt,
     nullptr},
    {"0x57 as a single-character dictionary key", "W=1", 3, std::nullopt,
     nullptr},
    {"0x58 as a single-character dictionary key", "X=1", 3, std::nullopt,
     nullptr},
    {"0x59 as a single-character dictionary key", "Y=1", 3, std::nullopt,
     nullptr},
    {"0x5a as a single-character dictionary key", "Z=1", 3, std::nullopt,
     nullptr},
    {"0x5b as a single-character dictionary key", "[=1", 3, std::nullopt,
     nullptr},
    {"0x5c as a single-character dictionary key", "\\=1", 3, std::nullopt,
     nullptr},
    {"0x5d as a single-character dictionary key", "]=1", 3, std::nullopt,
     nullptr},
    {"0x5e as a single-character dictionary key", "^=1", 3, std::nullopt,
     nullptr},
    {"0x5f as a single-character dictionary key", "_=1", 3, std::nullopt,
     nullptr},
    {"0x60 as a single-character dictionary key", "`=1", 3, std::nullopt,
     nullptr},
    {"0x61 as a single-character dictionary key",
     "a=1",
     3,
     {Dictionary{{{"a", {Integer(1), {}}}}}},
     nullptr},
    {"0x62 as a single-character dictionary key",
     "b=1",
     3,
     {Dictionary{{{"b", {Integer(1), {}}}}}},
     nullptr},
    {"0x63 as a single-character dictionary key",
     "c=1",
     3,
     {Dictionary{{{"c", {Integer(1), {}}}}}},
     nullptr},
    {"0x64 as a single-character dictionary key",
     "d=1",
     3,
     {Dictionary{{{"d", {Integer(1), {}}}}}},
     nullptr},
    {"0x65 as a single-character dictionary key",
     "e=1",
     3,
     {Dictionary{{{"e", {Integer(1), {}}}}}},
     nullptr},
    {"0x66 as a single-character dictionary key",
     "f=1",
     3,
     {Dictionary{{{"f", {Integer(1), {}}}}}},
     nullptr},
    {"0x67 as a single-character dictionary key",
     "g=1",
     3,
     {Dictionary{{{"g", {Integer(1), {}}}}}},
     nullptr},
    {"0x68 as a single-character dictionary key",
     "h=1",
     3,
     {Dictionary{{{"h", {Integer(1), {}}}}}},
     nullptr},
    {"0x69 as a single-character dictionary key",
     "i=1",
     3,
     {Dictionary{{{"i", {Integer(1), {}}}}}},
     nullptr},
    {"0x6a as a single-character dictionary key",
     "j=1",
     3,
     {Dictionary{{{"j", {Integer(1), {}}}}}},
     nullptr},
    {"0x6b as a single-character dictionary key",
     "k=1",
     3,
     {Dictionary{{{"k", {Integer(1), {}}}}}},
     nullptr},
    {"0x6c as a single-character dictionary key",
     "l=1",
     3,
     {Dictionary{{{"l", {Integer(1), {}}}}}},
     nullptr},
    {"0x6d as a single-character dictionary key",
     "m=1",
     3,
     {Dictionary{{{"m", {Integer(1), {}}}}}},
     nullptr},
    {"0x6e as a single-character dictionary key",
     "n=1",
     3,
     {Dictionary{{{"n", {Integer(1), {}}}}}},
     nullptr},
    {"0x6f as a single-character dictionary key",
     "o=1",
     3,
     {Dictionary{{{"o", {Integer(1), {}}}}}},
     nullptr},
    {"0x70 as a single-character dictionary key",
     "p=1",
     3,
     {Dictionary{{{"p", {Integer(1), {}}}}}},
     nullptr},
    {"0x71 as a single-character dictionary key",
     "q=1",
     3,
     {Dictionary{{{"q", {Integer(1), {}}}}}},
     nullptr},
    {"0x72 as a single-character dictionary key",
     "r=1",
     3,
     {Dictionary{{{"r", {Integer(1), {}}}}}},
     nullptr},
    {"0x73 as a single-character dictionary key",
     "s=1",
     3,
     {Dictionary{{{"s", {Integer(1), {}}}}}},
     nullptr},
    {"0x74 as a single-character dictionary key",
     "t=1",
     3,
     {Dictionary{{{"t", {Integer(1), {}}}}}},
     nullptr},
    {"0x75 as a single-character dictionary key",
     "u=1",
     3,
     {Dictionary{{{"u", {Integer(1), {}}}}}},
     nullptr},
    {"0x76 as a single-character dictionary key",
     "v=1",
     3,
     {Dictionary{{{"v", {Integer(1), {}}}}}},
     nullptr},
    {"0x77 as a single-character dictionary key",
     "w=1",
     3,
     {Dictionary{{{"w", {Integer(1), {}}}}}},
     nullptr},
    {"0x78 as a single-character dictionary key",
     "x=1",
     3,
     {Dictionary{{{"x", {Integer(1), {}}}}}},
     nullptr},
    {"0x79 as a single-character dictionary key",
     "y=1",
     3,
     {Dictionary{{{"y", {Integer(1), {}}}}}},
     nullptr},
    {"0x7a as a single-character dictionary key",
     "z=1",
     3,
     {Dictionary{{{"z", {Integer(1), {}}}}}},
     nullptr},
    {"0x7b as a single-character dictionary key", "{=1", 3, std::nullopt,
     nullptr},
    {"0x7c as a single-character dictionary key", "|=1", 3, std::nullopt,
     nullptr},
    {"0x7d as a single-character dictionary key", "}=1", 3, std::nullopt,
     nullptr},
    {"0x7e as a single-character dictionary key", "~=1", 3, std::nullopt,
     nullptr},
    {"0x7f as a single-character dictionary key", "\177=1", 3, std::nullopt,
     nullptr},
    {"0x00 in dictionary key", "a\000a=1", 5, std::nullopt, nullptr},
    {"0x01 in dictionary key", "a\001a=1", 5, std::nullopt, nullptr},
    {"0x02 in dictionary key", "a\002a=1", 5, std::nullopt, nullptr},
    {"0x03 in dictionary key", "a\003a=1", 5, std::nullopt, nullptr},
    {"0x04 in dictionary key", "a\004a=1", 5, std::nullopt, nullptr},
    {"0x05 in dictionary key", "a\005a=1", 5, std::nullopt, nullptr},
    {"0x06 in dictionary key", "a\006a=1", 5, std::nullopt, nullptr},
    {"0x07 in dictionary key", "a\aa=1", 5, std::nullopt, nullptr},
    {"0x08 in dictionary key", "a\ba=1", 5, std::nullopt, nullptr},
    {"0x09 in dictionary key", "a\ta=1", 5, std::nullopt, nullptr},
    {"0x0a in dictionary key", "a\na=1", 5, std::nullopt, nullptr},
    {"0x0b in dictionary key", "a\va=1", 5, std::nullopt, nullptr},
    {"0x0c in dictionary key", "a\fa=1", 5, std::nullopt, nullptr},
    {"0x0d in dictionary key", "a\ra=1", 5, std::nullopt, nullptr},
    {"0x0e in dictionary key", "a\016a=1", 5, std::nullopt, nullptr},
    {"0x0f in dictionary key", "a\017a=1", 5, std::nullopt, nullptr},
    {"0x10 in dictionary key", "a\020a=1", 5, std::nullopt, nullptr},
    {"0x11 in dictionary key", "a\021a=1", 5, std::nullopt, nullptr},
    {"0x12 in dictionary key", "a\022a=1", 5, std::nullopt, nullptr},
    {"0x13 in dictionary key", "a\023a=1", 5, std::nullopt, nullptr},
    {"0x14 in dictionary key", "a\024a=1", 5, std::nullopt, nullptr},
    {"0x15 in dictionary key", "a\025a=1", 5, std::nullopt, nullptr},
    {"0x16 in dictionary key", "a\026a=1", 5, std::nullopt, nullptr},
    {"0x17 in dictionary key", "a\027a=1", 5, std::nullopt, nullptr},
    {"0x18 in dictionary key", "a\030a=1", 5, std::nullopt, nullptr},
    {"0x19 in dictionary key", "a\031a=1", 5, std::nullopt, nullptr},
    {"0x1a in dictionary key", "a\032a=1", 5, std::nullopt, nullptr},
    {"0x1b in dictionary key", "a\033a=1", 5, std::nullopt, nullptr},
    {"0x1c in dictionary key", "a\034a=1", 5, std::nullopt, nullptr},
    {"0x1d in dictionary key", "a\035a=1", 5, std::nullopt, nullptr},
    {"0x1e in dictionary key", "a\036a=1", 5, std::nullopt, nullptr},
    {"0x1f in dictionary key", "a\037a=1", 5, std::nullopt, nullptr},
    {"0x20 in dictionary key", "a a=1", 5, std::nullopt, nullptr},
    {"0x21 in dictionary key", "a!a=1", 5, std::nullopt, nullptr},
    {"0x22 in dictionary key", "a\"a=1", 5, std::nullopt, nullptr},
    {"0x23 in dictionary key", "a#a=1", 5, std::nullopt, nullptr},
    {"0x24 in dictionary key", "a$a=1", 5, std::nullopt, nullptr},
    {"0x25 in dictionary key", "a%a=1", 5, std::nullopt, nullptr},
    {"0x26 in dictionary key", "a&a=1", 5, std::nullopt, nullptr},
    {"0x27 in dictionary key", "a'a=1", 5, std::nullopt, nullptr},
    {"0x28 in dictionary key", "a(a=1", 5, std::nullopt, nullptr},
    {"0x29 in dictionary key", "a)a=1", 5, std::nullopt, nullptr},
    {"0x2a in dictionary key",
     "a*a=1",
     5,
     {Dictionary{{{"a*a", {Integer(1), {}}}}}},
     nullptr},
    {"0x2b in dictionary key", "a+a=1", 5, std::nullopt, nullptr},
    {"0x2c in dictionary key",
     "a,a=1",
     5,
     {Dictionary{{{"a", {Integer(1), {}}}}}},
     "a=1"},
    {"0x2d in dictionary key",
     "a-a=1",
     5,
     {Dictionary{{{"a-a", {Integer(1), {}}}}}},
     nullptr},
    {"0x2e in dictionary key",
     "a.a=1",
     5,
     {Dictionary{{{"a.a", {Integer(1), {}}}}}},
     nullptr},
    {"0x2f in dictionary key", "a/a=1", 5, std::nullopt, nullptr},
    {"0x30 in dictionary key",
     "a0a=1",
     5,
     {Dictionary{{{"a0a", {Integer(1), {}}}}}},
     nullptr},
    {"0x31 in dictionary key",
     "a1a=1",
     5,
     {Dictionary{{{"a1a", {Integer(1), {}}}}}},
     nullptr},
    {"0x32 in dictionary key",
     "a2a=1",
     5,
     {Dictionary{{{"a2a", {Integer(1), {}}}}}},
     nullptr},
    {"0x33 in dictionary key",
     "a3a=1",
     5,
     {Dictionary{{{"a3a", {Integer(1), {}}}}}},
     nullptr},
    {"0x34 in dictionary key",
     "a4a=1",
     5,
     {Dictionary{{{"a4a", {Integer(1), {}}}}}},
     nullptr},
    {"0x35 in dictionary key",
     "a5a=1",
     5,
     {Dictionary{{{"a5a", {Integer(1), {}}}}}},
     nullptr},
    {"0x36 in dictionary key",
     "a6a=1",
     5,
     {Dictionary{{{"a6a", {Integer(1), {}}}}}},
     nullptr},
    {"0x37 in dictionary key",
     "a7a=1",
     5,
     {Dictionary{{{"a7a", {Integer(1), {}}}}}},
     nullptr},
    {"0x38 in dictionary key",
     "a8a=1",
     5,
     {Dictionary{{{"a8a", {Integer(1), {}}}}}},
     nullptr},
    {"0x39 in dictionary key",
     "a9a=1",
     5,
     {Dictionary{{{"a9a", {Integer(1), {}}}}}},
     nullptr},
    {"0x3a in dictionary key", "a:a=1", 5, std::nullopt, nullptr},
    {"0x3b in dictionary key",
     "a;a=1",
     5,
     {Dictionary{{{"a", {Item(true), {Param("a", 1)}}}}}},
     nullptr},
    {"0x3c in dictionary key", "a<a=1", 5, std::nullopt, nullptr},
    {"0x3d in dictionary key", "a=a=1", 5, std::nullopt, nullptr},
    {"0x3e in dictionary key", "a>a=1", 5, std::nullopt, nullptr},
    {"0x3f in dictionary key", "a?a=1", 5, std::nullopt, nullptr},
    {"0x40 in dictionary key", "a@a=1", 5, std::nullopt, nullptr},
    {"0x41 in dictionary key", "aAa=1", 5, std::nullopt, nullptr},
    {"0x42 in dictionary key", "aBa=1", 5, std::nullopt, nullptr},
    {"0x43 in dictionary key", "aCa=1", 5, std::nullopt, nullptr},
    {"0x44 in dictionary key", "aDa=1", 5, std::nullopt, nullptr},
    {"0x45 in dictionary key", "aEa=1", 5, std::nullopt, nullptr},
    {"0x46 in dictionary key", "aFa=1", 5, std::nullopt, nullptr},
    {"0x47 in dictionary key", "aGa=1", 5, std::nullopt, nullptr},
    {"0x48 in dictionary key", "aHa=1", 5, std::nullopt, nullptr},
    {"0x49 in dictionary key", "aIa=1", 5, std::nullopt, nullptr},
    {"0x4a in dictionary key", "aJa=1", 5, std::nullopt, nullptr},
    {"0x4b in dictionary key", "aKa=1", 5, std::nullopt, nullptr},
    {"0x4c in dictionary key", "aLa=1", 5, std::nullopt, nullptr},
    {"0x4d in dictionary key", "aMa=1", 5, std::nullopt, nullptr},
    {"0x4e in dictionary key", "aNa=1", 5, std::nullopt, nullptr},
    {"0x4f in dictionary key", "aOa=1", 5, std::nullopt, nullptr},
    {"0x50 in dictionary key", "aPa=1", 5, std::nullopt, nullptr},
    {"0x51 in dictionary key", "aQa=1", 5, std::nullopt, nullptr},
    {"0x52 in dictionary key", "aRa=1", 5, std::nullopt, nullptr},
    {"0x53 in dictionary key", "aSa=1", 5, std::nullopt, nullptr},
    {"0x54 in dictionary key", "aTa=1", 5, std::nullopt, nullptr},
    {"0x55 in dictionary key", "aUa=1", 5, std::nullopt, nullptr},
    {"0x56 in dictionary key", "aVa=1", 5, std::nullopt, nullptr},
    {"0x57 in dictionary key", "aWa=1", 5, std::nullopt, nullptr},
    {"0x58 in dictionary key", "aXa=1", 5, std::nullopt, nullptr},
    {"0x59 in dictionary key", "aYa=1", 5, std::nullopt, nullptr},
    {"0x5a in dictionary key", "aZa=1", 5, std::nullopt, nullptr},
    {"0x5b in dictionary key", "a[a=1", 5, std::nullopt, nullptr},
    {"0x5c in dictionary key", "a\\a=1", 5, std::nullopt, nullptr},
    {"0x5d in dictionary key", "a]a=1", 5, std::nullopt, nullptr},
    {"0x5e in dictionary key", "a^a=1", 5, std::nullopt, nullptr},
    {"0x5f in dictionary key",
     "a_a=1",
     5,
     {Dictionary{{{"a_a", {Integer(1), {}}}}}},
     nullptr},
    {"0x60 in dictionary key", "a`a=1", 5, std::nullopt, nullptr},
    {"0x61 in dictionary key",
     "aaa=1",
     5,
     {Dictionary{{{"aaa", {Integer(1), {}}}}}},
     nullptr},
    {"0x62 in dictionary key",
     "aba=1",
     5,
     {Dictionary{{{"aba", {Integer(1), {}}}}}},
     nullptr},
    {"0x63 in dictionary key",
     "aca=1",
     5,
     {Dictionary{{{"aca", {Integer(1), {}}}}}},
     nullptr},
    {"0x64 in dictionary key",
     "ada=1",
     5,
     {Dictionary{{{"ada", {Integer(1), {}}}}}},
     nullptr},
    {"0x65 in dictionary key",
     "aea=1",
     5,
     {Dictionary{{{"aea", {Integer(1), {}}}}}},
     nullptr},
    {"0x66 in dictionary key",
     "afa=1",
     5,
     {Dictionary{{{"afa", {Integer(1), {}}}}}},
     nullptr},
    {"0x67 in dictionary key",
     "aga=1",
     5,
     {Dictionary{{{"aga", {Integer(1), {}}}}}},
     nullptr},
    {"0x68 in dictionary key",
     "aha=1",
     5,
     {Dictionary{{{"aha", {Integer(1), {}}}}}},
     nullptr},
    {"0x69 in dictionary key",
     "aia=1",
     5,
     {Dictionary{{{"aia", {Integer(1), {}}}}}},
     nullptr},
    {"0x6a in dictionary key",
     "aja=1",
     5,
     {Dictionary{{{"aja", {Integer(1), {}}}}}},
     nullptr},
    {"0x6b in dictionary key",
     "aka=1",
     5,
     {Dictionary{{{"aka", {Integer(1), {}}}}}},
     nullptr},
    {"0x6c in dictionary key",
     "ala=1",
     5,
     {Dictionary{{{"ala", {Integer(1), {}}}}}},
     nullptr},
    {"0x6d in dictionary key",
     "ama=1",
     5,
     {Dictionary{{{"ama", {Integer(1), {}}}}}},
     nullptr},
    {"0x6e in dictionary key",
     "ana=1",
     5,
     {Dictionary{{{"ana", {Integer(1), {}}}}}},
     nullptr},
    {"0x6f in dictionary key",
     "aoa=1",
     5,
     {Dictionary{{{"aoa", {Integer(1), {}}}}}},
     nullptr},
    {"0x70 in dictionary key",
     "apa=1",
     5,
     {Dictionary{{{"apa", {Integer(1), {}}}}}},
     nullptr},
    {"0x71 in dictionary key",
     "aqa=1",
     5,
     {Dictionary{{{"aqa", {Integer(1), {}}}}}},
     nullptr},
    {"0x72 in dictionary key",
     "ara=1",
     5,
     {Dictionary{{{"ara", {Integer(1), {}}}}}},
     nullptr},
    {"0x73 in dictionary key",
     "asa=1",
     5,
     {Dictionary{{{"asa", {Integer(1), {}}}}}},
     nullptr},
    {"0x74 in dictionary key",
     "ata=1",
     5,
     {Dictionary{{{"ata", {Integer(1), {}}}}}},
     nullptr},
    {"0x75 in dictionary key",
     "aua=1",
     5,
     {Dictionary{{{"aua", {Integer(1), {}}}}}},
     nullptr},
    {"0x76 in dictionary key",
     "ava=1",
     5,
     {Dictionary{{{"ava", {Integer(1), {}}}}}},
     nullptr},
    {"0x77 in dictionary key",
     "awa=1",
     5,
     {Dictionary{{{"awa", {Integer(1), {}}}}}},
     nullptr},
    {"0x78 in dictionary key",
     "axa=1",
     5,
     {Dictionary{{{"axa", {Integer(1), {}}}}}},
     nullptr},
    {"0x79 in dictionary key",
     "aya=1",
     5,
     {Dictionary{{{"aya", {Integer(1), {}}}}}},
     nullptr},
    {"0x7a in dictionary key",
     "aza=1",
     5,
     {Dictionary{{{"aza", {Integer(1), {}}}}}},
     nullptr},
    {"0x7b in dictionary key", "a{a=1", 5, std::nullopt, nullptr},
    {"0x7c in dictionary key", "a|a=1", 5, std::nullopt, nullptr},
    {"0x7d in dictionary key", "a}a=1", 5, std::nullopt, nullptr},
    {"0x7e in dictionary key", "a~a=1", 5, std::nullopt, nullptr},
    {"0x7f in dictionary key", "a\177a=1", 5, std::nullopt, nullptr},
    {"0x00 starting an dictionary key", "\000a=1", 4, std::nullopt, nullptr},
    {"0x01 starting an dictionary key", "\001a=1", 4, std::nullopt, nullptr},
    {"0x02 starting an dictionary key", "\002a=1", 4, std::nullopt, nullptr},
    {"0x03 starting an dictionary key", "\003a=1", 4, std::nullopt, nullptr},
    {"0x04 starting an dictionary key", "\004a=1", 4, std::nullopt, nullptr},
    {"0x05 starting an dictionary key", "\005a=1", 4, std::nullopt, nullptr},
    {"0x06 starting an dictionary key", "\006a=1", 4, std::nullopt, nullptr},
    {"0x07 starting an dictionary key", "\aa=1", 4, std::nullopt, nullptr},
    {"0x08 starting an dictionary key", "\ba=1", 4, std::nullopt, nullptr},
    {"0x09 starting an dictionary key", "\ta=1", 4, std::nullopt, nullptr},
    {"0x0a starting an dictionary key", "\na=1", 4, std::nullopt, nullptr},
    {"0x0b starting an dictionary key", "\va=1", 4, std::nullopt, nullptr},
    {"0x0c starting an dictionary key", "\fa=1", 4, std::nullopt, nullptr},
    {"0x0d starting an dictionary key", "\ra=1", 4, std::nullopt, nullptr},
    {"0x0e starting an dictionary key", "\016a=1", 4, std::nullopt, nullptr},
    {"0x0f starting an dictionary key", "\017a=1", 4, std::nullopt, nullptr},
    {"0x10 starting an dictionary key", "\020a=1", 4, std::nullopt, nullptr},
    {"0x11 starting an dictionary key", "\021a=1", 4, std::nullopt, nullptr},
    {"0x12 starting an dictionary key", "\022a=1", 4, std::nullopt, nullptr},
    {"0x13 starting an dictionary key", "\023a=1", 4, std::nullopt, nullptr},
    {"0x14 starting an dictionary key", "\024a=1", 4, std::nullopt, nullptr},
    {"0x15 starting an dictionary key", "\025a=1", 4, std::nullopt, nullptr},
    {"0x16 starting an dictionary key", "\026a=1", 4, std::nullopt, nullptr},
    {"0x17 starting an dictionary key", "\027a=1", 4, std::nullopt, nullptr},
    {"0x18 starting an dictionary key", "\030a=1", 4, std::nullopt, nullptr},
    {"0x19 starting an dictionary key", "\031a=1", 4, std::nullopt, nullptr},
    {"0x1a starting an dictionary key", "\032a=1", 4, std::nullopt, nullptr},
    {"0x1b starting an dictionary key", "\033a=1", 4, std::nullopt, nullptr},
    {"0x1c starting an dictionary key", "\034a=1", 4, std::nullopt, nullptr},
    {"0x1d starting an dictionary key", "\035a=1", 4, std::nullopt, nullptr},
    {"0x1e starting an dictionary key", "\036a=1", 4, std::nullopt, nullptr},
    {"0x1f starting an dictionary key", "\037a=1", 4, std::nullopt, nullptr},
    {"0x20 starting an dictionary key",
     " a=1",
     4,
     {Dictionary{{{"a", {Integer(1), {}}}}}},
     "a=1"},
    {"0x21 starting an dictionary key", "!a=1", 4, std::nullopt, nullptr},
    {"0x22 starting an dictionary key", "\"a=1", 4, std::nullopt, nullptr},
    {"0x23 starting an dictionary key", "#a=1", 4, std::nullopt, nullptr},
    {"0x24 starting an dictionary key", "$a=1", 4, std::nullopt, nullptr},
    {"0x25 starting an dictionary key", "%a=1", 4, std::nullopt, nullptr},
    {"0x26 starting an dictionary key", "&a=1", 4, std::nullopt, nullptr},
    {"0x27 starting an dictionary key", "'a=1", 4, std::nullopt, nullptr},
    {"0x28 starting an dictionary key", "(a=1", 4, std::nullopt, nullptr},
    {"0x29 starting an dictionary key", ")a=1", 4, std::nullopt, nullptr},
    {"0x2a starting an dictionary key",
     "*a=1",
     4,
     {Dictionary{{{"*a", {Integer(1), {}}}}}},
     nullptr},
    {"0x2b starting an dictionary key", "+a=1", 4, std::nullopt, nullptr},
    {"0x2c starting an dictionary key", ",a=1", 4, std::nullopt, nullptr},
    {"0x2d starting an dictionary key", "-a=1", 4, std::nullopt, nullptr},
    {"0x2e starting an dictionary key", ".a=1", 4, std::nullopt, nullptr},
    {"0x2f starting an dictionary key", "/a=1", 4, std::nullopt, nullptr},
    {"0x30 starting an dictionary key", "0a=1", 4, std::nullopt, nullptr},
    {"0x31 starting an dictionary key", "1a=1", 4, std::nullopt, nullptr},
    {"0x32 starting an dictionary key", "2a=1", 4, std::nullopt, nullptr},
    {"0x33 starting an dictionary key", "3a=1", 4, std::nullopt, nullptr},
    {"0x34 starting an dictionary key", "4a=1", 4, std::nullopt, nullptr},
    {"0x35 starting an dictionary key", "5a=1", 4, std::nullopt, nullptr},
    {"0x36 starting an dictionary key", "6a=1", 4, std::nullopt, nullptr},
    {"0x37 starting an dictionary key", "7a=1", 4, std::nullopt, nullptr},
    {"0x38 starting an dictionary key", "8a=1", 4, std::nullopt, nullptr},
    {"0x39 starting an dictionary key", "9a=1", 4, std::nullopt, nullptr},
    {"0x3a starting an dictionary key", ":a=1", 4, std::nullopt, nullptr},
    {"0x3b starting an dictionary key", ";a=1", 4, std::nullopt, nullptr},
    {"0x3c starting an dictionary key", "<a=1", 4, std::nullopt, nullptr},
    {"0x3d starting an dictionary key", "=a=1", 4, std::nullopt, nullptr},
    {"0x3e starting an dictionary key", ">a=1", 4, std::nullopt, nullptr},
    {"0x3f starting an dictionary key", "?a=1", 4, std::nullopt, nullptr},
    {"0x40 starting an dictionary key", "@a=1", 4, std::nullopt, nullptr},
    {"0x41 starting an dictionary key", "Aa=1", 4, std::nullopt, nullptr},
    {"0x42 starting an dictionary key", "Ba=1", 4, std::nullopt, nullptr},
    {"0x43 starting an dictionary key", "Ca=1", 4, std::nullopt, nullptr},
    {"0x44 starting an dictionary key", "Da=1", 4, std::nullopt, nullptr},
    {"0x45 starting an dictionary key", "Ea=1", 4, std::nullopt, nullptr},
    {"0x46 starting an dictionary key", "Fa=1", 4, std::nullopt, nullptr},
    {"0x47 starting an dictionary key", "Ga=1", 4, std::nullopt, nullptr},
    {"0x48 starting an dictionary key", "Ha=1", 4, std::nullopt, nullptr},
    {"0x49 starting an dictionary key", "Ia=1", 4, std::nullopt, nullptr},
    {"0x4a starting an dictionary key", "Ja=1", 4, std::nullopt, nullptr},
    {"0x4b starting an dictionary key", "Ka=1", 4, std::nullopt, nullptr},
    {"0x4c starting an dictionary key", "La=1", 4, std::nullopt, nullptr},
    {"0x4d starting an dictionary key", "Ma=1", 4, std::nullopt, nullptr},
    {"0x4e starting an dictionary key", "Na=1", 4, std::nullopt, nullptr},
    {"0x4f starting an dictionary key", "Oa=1", 4, std::nullopt, nullptr},
    {"0x50 starting an dictionary key", "Pa=1", 4, std::nullopt, nullptr},
    {"0x51 starting an dictionary key", "Qa=1", 4, std::nullopt, nullptr},
    {"0x52 starting an dictionary key", "Ra=1", 4, std::nullopt, nullptr},
    {"0x53 starting an dictionary key", "Sa=1", 4, std::nullopt, nullptr},
    {"0x54 starting an dictionary key", "Ta=1", 4, std::nullopt, nullptr},
    {"0x55 starting an dictionary key", "Ua=1", 4, std::nullopt, nullptr},
    {"0x56 starting an dictionary key", "Va=1", 4, std::nullopt, nullptr},
    {"0x57 starting an dictionary key", "Wa=1", 4, std::nullopt, nullptr},
    {"0x58 starting an dictionary key", "Xa=1", 4, std::nullopt, nullptr},
    {"0x59 starting an dictionary key", "Ya=1", 4, std::nullopt, nullptr},
    {"0x5a starting an dictionary key", "Za=1", 4, std::nullopt, nullptr},
    {"0x5b starting an dictionary key", "[a=1", 4, std::nullopt, nullptr},
    {"0x5c starting an dictionary key", "\\a=1", 4, std::nullopt, nullptr},
    {"0x5d starting an dictionary key", "]a=1", 4, std::nullopt, nullptr},
    {"0x5e starting an dictionary key", "^a=1", 4, std::nullopt, nullptr},
    {"0x5f starting an dictionary key", "_a=1", 4, std::nullopt, nullptr},
    {"0x60 starting an dictionary key", "`a=1", 4, std::nullopt, nullptr},
    {"0x61 starting an dictionary key",
     "aa=1",
     4,
     {Dictionary{{{"aa", {Integer(1), {}}}}}},
     nullptr},
    {"0x62 starting an dictionary key",
     "ba=1",
     4,
     {Dictionary{{{"ba", {Integer(1), {}}}}}},
     nullptr},
    {"0x63 starting an dictionary key",
     "ca=1",
     4,
     {Dictionary{{{"ca", {Integer(1), {}}}}}},
     nullptr},
    {"0x64 starting an dictionary key",
     "da=1",
     4,
     {Dictionary{{{"da", {Integer(1), {}}}}}},
     nullptr},
    {"0x65 starting an dictionary key",
     "ea=1",
     4,
     {Dictionary{{{"ea", {Integer(1), {}}}}}},
     nullptr},
    {"0x66 starting an dictionary key",
     "fa=1",
     4,
     {Dictionary{{{"fa", {Integer(1), {}}}}}},
     nullptr},
    {"0x67 starting an dictionary key",
     "ga=1",
     4,
     {Dictionary{{{"ga", {Integer(1), {}}}}}},
     nullptr},
    {"0x68 starting an dictionary key",
     "ha=1",
     4,
     {Dictionary{{{"ha", {Integer(1), {}}}}}},
     nullptr},
    {"0x69 starting an dictionary key",
     "ia=1",
     4,
     {Dictionary{{{"ia", {Integer(1), {}}}}}},
     nullptr},
    {"0x6a starting an dictionary key",
     "ja=1",
     4,
     {Dictionary{{{"ja", {Integer(1), {}}}}}},
     nullptr},
    {"0x6b starting an dictionary key",
     "ka=1",
     4,
     {Dictionary{{{"ka", {Integer(1), {}}}}}},
     nullptr},
    {"0x6c starting an dictionary key",
     "la=1",
     4,
     {Dictionary{{{"la", {Integer(1), {}}}}}},
     nullptr},
    {"0x6d starting an dictionary key",
     "ma=1",
     4,
     {Dictionary{{{"ma", {Integer(1), {}}}}}},
     nullptr},
    {"0x6e starting an dictionary key",
     "na=1",
     4,
     {Dictionary{{{"na", {Integer(1), {}}}}}},
     nullptr},
    {"0x6f starting an dictionary key",
     "oa=1",
     4,
     {Dictionary{{{"oa", {Integer(1), {}}}}}},
     nullptr},
    {"0x70 starting an dictionary key",
     "pa=1",
     4,
     {Dictionary{{{"pa", {Integer(1), {}}}}}},
     nullptr},
    {"0x71 starting an dictionary key",
     "qa=1",
     4,
     {Dictionary{{{"qa", {Integer(1), {}}}}}},
     nullptr},
    {"0x72 starting an dictionary key",
     "ra=1",
     4,
     {Dictionary{{{"ra", {Integer(1), {}}}}}},
     nullptr},
    {"0x73 starting an dictionary key",
     "sa=1",
     4,
     {Dictionary{{{"sa", {Integer(1), {}}}}}},
     nullptr},
    {"0x74 starting an dictionary key",
     "ta=1",
     4,
     {Dictionary{{{"ta", {Integer(1), {}}}}}},
     nullptr},
    {"0x75 starting an dictionary key",
     "ua=1",
     4,
     {Dictionary{{{"ua", {Integer(1), {}}}}}},
     nullptr},
    {"0x76 starting an dictionary key",
     "va=1",
     4,
     {Dictionary{{{"va", {Integer(1), {}}}}}},
     nullptr},
    {"0x77 starting an dictionary key",
     "wa=1",
     4,
     {Dictionary{{{"wa", {Integer(1), {}}}}}},
     nullptr},
    {"0x78 starting an dictionary key",
     "xa=1",
     4,
     {Dictionary{{{"xa", {Integer(1), {}}}}}},
     nullptr},
    {"0x79 starting an dictionary key",
     "ya=1",
     4,
     {Dictionary{{{"ya", {Integer(1), {}}}}}},
     nullptr},
    {"0x7a starting an dictionary key",
     "za=1",
     4,
     {Dictionary{{{"za", {Integer(1), {}}}}}},
     nullptr},
    {"0x7b starting an dictionary key", "{a=1", 4, std::nullopt, nullptr},
    {"0x7c starting an dictionary key", "|a=1", 4, std::nullopt, nullptr},
    {"0x7d starting an dictionary key", "}a=1", 4, std::nullopt, nullptr},
    {"0x7e starting an dictionary key", "~a=1", 4, std::nullopt, nullptr},
    {"0x7f starting an dictionary key", "\177a=1", 4, std::nullopt, nullptr},
    // param-dict.json
    {"basic parameterised dict",
     "abc=123;a=1;b=2, def=456, ghi=789;q=9;r=\"+w\"",
     44,
     {Dictionary{{{"abc", {Integer(123), {Param("a", 1), Param("b", 2)}}},
                  {"def", {Integer(456), {}}},
                  {"ghi", {Integer(789), {Param("q", 9), Param("r", "+w")}}}}}},
     nullptr},
    {"single item parameterised dict",
     "a=b; q=1.0",
     10,
     {Dictionary{
         {{"a", {Item("b", Item::kTokenType), {DoubleParam("q", 1.000000)}}}}}},
     "a=b;q=1.0"},
    {"list item parameterised dictionary",
     "a=(1 2); q=1.0",
     14,
     {Dictionary{{{"a",
                   {{{Integer(1), {}}, {Integer(2), {}}},
                    {DoubleParam("q", 1.000000)}}}}}},
     "a=(1 2);q=1.0"},
    {"missing parameter value parameterised dict",
     "a=3;c;d=5",
     9,
     {Dictionary{
         {{"a", {Integer(3), {BooleanParam("c", true), Param("d", 5)}}}}}},
     nullptr},
    {"terminal missing parameter value parameterised dict",
     "a=3;c=5;d",
     9,
     {Dictionary{
         {{"a", {Integer(3), {Param("c", 5), BooleanParam("d", true)}}}}}},
     nullptr},
    {"no whitespace parameterised dict",
     "a=b;c=1,d=e;f=2",
     15,
     {Dictionary{{{"a", {Item("b", Item::kTokenType), {Param("c", 1)}}},
                  {"d", {Item("e", Item::kTokenType), {Param("f", 2)}}}}}},
     "a=b;c=1, d=e;f=2"},
    {"whitespace before = parameterised dict", "a=b;q =0.5", 10, std::nullopt,
     nullptr},
    {"whitespace after = parameterised dict", "a=b;q= 0.5", 10, std::nullopt,
     nullptr},
    {"whitespace before ; parameterised dict", "a=b ;q=0.5", 10, std::nullopt,
     nullptr},
    {"whitespace after ; parameterised dict",
     "a=b; q=0.5",
     10,
     {Dictionary{
         {{"a", {Item("b", Item::kTokenType), {DoubleParam("q", 0.500000)}}}}}},
     "a=b;q=0.5"},
    {"extra whitespace parameterised dict",
     "a=b;  c=1  ,  d=e; f=2; g=3",
     27,
     {Dictionary{
         {{"a", {Item("b", Item::kTokenType), {Param("c", 1)}}},
          {"d",
           {Item("e", Item::kTokenType), {Param("f", 2), Param("g", 3)}}}}}},
     "a=b;c=1, d=e;f=2;g=3"},
    {"two lines parameterised list",
     "a=b;c=1, d=e;f=2",
     16,
     {Dictionary{{{"a", {Item("b", Item::kTokenType), {Param("c", 1)}}},
                  {"d", {Item("e", Item::kTokenType), {Param("f", 2)}}}}}},
     "a=b;c=1, d=e;f=2"},
    {"trailing comma parameterised list", "a=b; q=1.0,", 11, std::nullopt,
     nullptr},
    {"empty item parameterised list", "a=b; q=1.0,,c=d", 15, std::nullopt,
     nullptr},
};

}  // namespace

TEST(StructuredHeaderGeneratedTest, ParseItem) {
  for (const auto& c : parameterized_item_test_cases) {
    if (c.raw) {
      SCOPED_TRACE(c.name);
      std::string raw{c.raw, c.raw_len};
      std::optional<ParameterizedItem> result = ParseItem(raw);
      EXPECT_EQ(result, c.expected);
    }
  }
}

TEST(StructuredHeaderGeneratedTest, ParseList) {
  for (const auto& c : list_test_cases) {
    if (c.raw) {
      SCOPED_TRACE(c.name);
      std::string raw{c.raw, c.raw_len};
      std::optional<List> result = ParseList(raw);
      EXPECT_EQ(result, c.expected);
    }
  }
}

TEST(StructuredHeaderGeneratedTest, ParseDictionary) {
  for (const auto& c : dictionary_test_cases) {
    if (c.raw) {
      SCOPED_TRACE(c.name);
      std::string raw{c.raw, c.raw_len};
      std::optional<Dictionary> result = ParseDictionary(raw);
      EXPECT_EQ(result, c.expected);
    }
  }
}

TEST(StructuredHeaderGeneratedTest, SerializeItem) {
  for (const auto& c : parameterized_item_test_cases) {
    SCOPED_TRACE(c.name);
    if (c.expected) {
      std::optional<std::string> result = SerializeItem(*c.expected);
      if (c.raw || c.canonical) {
        EXPECT_TRUE(result.has_value());
        EXPECT_EQ(result.value(),
                  std::string(c.canonical ? c.canonical : c.raw));
      } else {
        EXPECT_FALSE(result.has_value());
      }
    }
  }
}

TEST(StructuredHeaderGeneratedTest, SerializeList) {
  for (const auto& c : list_test_cases) {
    SCOPED_TRACE(c.name);
    if (c.expected) {
      std::optional<std::string> result = SerializeList(*c.expected);
      if (c.raw || c.canonical) {
        EXPECT_TRUE(result.has_value());
        EXPECT_EQ(result.value(),
                  std::string(c.canonical ? c.canonical : c.raw));
      } else {
        EXPECT_FALSE(result.has_value());
      }
    }
  }
}

TEST(StructuredHeaderGeneratedTest, SerializeDictionary) {
  for (const auto& c : dictionary_test_cases) {
    SCOPED_TRACE(c.name);
    if (c.expected) {
      std::optional<std::string> result = SerializeDictionary(*c.expected);
      if (c.raw || c.canonical) {
        EXPECT_TRUE(result.has_value());
        EXPECT_EQ(result.value(),
                  std::string(c.canonical ? c.canonical : c.raw));
      } else {
        EXPECT_FALSE(result.has_value());
      }
    }
  }
}

}  // namespace structured_headers
}  // namespace quiche
