// SPDX-License-Identifier: GPL-2.0 OR CDDL-1.0
/*
 * CDDL HEADER START
 *
 * The contents of this file are subject to the terms of the
 * Common Development and Distribution License (the "License").
 * You may not use this file except in compliance with the License.
 *
 * You can obtain a copy of the license at usr/src/OPENSOLARIS.LICENSE
 * or https://opensource.org/licenses/CDDL-1.0.
 * See the License for the specific language governing permissions
 * and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL HEADER in each
 * file and include the License file at usr/src/OPENSOLARIS.LICENSE.
 * If applicable, add the following below this CDDL HEADER, with the
 * fields enclosed by brackets "[]" replaced with your own identifying
 * information: Portions Copyright [yyyy] [name of copyright owner]
 *
 * CDDL HEADER END
 */

/* Copyright (c) 2019-2025 Chilledheart  */

#ifndef H_CRYPTO_ENCRYPTER
#define H_CRYPTO_ENCRYPTER

#include "crypto/crypter.hpp"

#include <stddef.h>
#include <stdint.h>
#include <memory>
#include <string>

namespace crypto {

class Encrypter : public Crypter {
 public:
  virtual ~Encrypter();

  static std::unique_ptr<Encrypter> CreateFromCipherSuite(uint32_t cipher_suite);

  // Writes encrypted |plaintext| and a MAC over |plaintext| and
  // |associated_data| into output. Sets |output_length| to the number of
  // bytes written. Returns true on success or false if there was an error.
  // |packet_number| is appended to the |nonce_prefix| value provided in
  // SetNoncePrefix() to form the nonce. |output| must not overlap with
  // |associated_data|. If |output| overlaps with |plaintext| then
  // |plaintext| must be <= |output|.
  virtual bool EncryptPacket(uint64_t packet_number,
                             const char* associated_data,
                             size_t associated_data_len,
                             const char* plaintext,
                             size_t plaintext_len,
                             char* output,
                             size_t* output_length,
                             size_t max_output_length) = 0;

#if 0
  // Takes a |sample| of ciphertext and uses the header protection key to
  // generate a mask to use for header protection, and returns that mask. On
  // success, the mask will be at least 5 bytes long; on failure the string will
  // be empty.
  virtual std::string GenerateHeaderProtectionMask(QuicStringPiece sample) = 0;
#endif

  // Returns the maximum length of plaintext that can be encrypted
  // to ciphertext no larger than |ciphertext_size|.
  virtual size_t GetMaxPlaintextSize(size_t ciphertext_size) const = 0;

  // Returns the length of the ciphertext that would be generated by encrypting
  // to plaintext of size |plaintext_size|.
  virtual size_t GetCiphertextSize(size_t plaintext_size) const = 0;

  // The ID of the cipher. Return 0x03000000 ORed with the 'cryptographic suite
  // selector'.
  virtual uint32_t cipher_id() const = 0;

  // For use by unit tests only.
  virtual const uint8_t* GetKey() const = 0;
  virtual const uint8_t* GetIV() const = 0;
  virtual const uint8_t* GetNoncePrefix() const = 0;
};

}  // namespace crypto

#endif  // H_CRYPTO_ENCRYPTER
