//! NIST P-384 tests and test vectors

use crate::{ecdsa_tests, signature::TestVector};
use core::convert::TryFrom;
use ring_compat::signature::{
    ecdsa::p384::{Signature, SigningKey, VerifyingKey},
    Signer, Verifier,
};

ecdsa_tests!(SigningKey, VerifyingKey, TEST_VECTORS);

/// ECDSA test vectors for the NIST P-384 elliptic curve
///
/// Sourced from NIST's CAVP web site (FIPS 186-4 ECDSA Test Vectors):
///
/// <https://csrc.nist.gov/Projects/Cryptographic-Algorithm-Validation-Program/Digital-Signatures>
const TEST_VECTORS: &[TestVector] = &[
    TestVector {
        sk: b"\x20\x1b\x43\x2d\x8d\xf1\x43\x24\x18\x2d\x62\x61\xdb\x3e\x4b\x3f\x46\xa8\x28\x44\x82\xd5\x2e\x37\x0d\xa4\x1e\x6c\xbd\xf4\x5e\xc2\x95\x2f\x5d\xb7\xcc\xbc\xe3\xbc\x29\x44\x9f\x4f\xb0\x80\xac\x97",
        pk: b"\xc2\xb4\x79\x44\xfb\x5d\xe3\x42\xd0\x32\x85\x88\x01\x77\xca\x5f\x7d\x0f\x2f\xca\xd7\x67\x8c\xce\x42\x29\xd6\xe1\x93\x2f\xca\xc1\x1b\xfc\x3c\x3e\x97\xd9\x42\xa3\xc5\x6b\xf3\x41\x23\x01\x3d\xbf\x37\x25\x79\x06\xa8\x22\x38\x66\xed\xa0\x74\x3c\x51\x96\x16\xa7\x6a\x75\x8a\xe5\x8a\xee\x81\xc5\xfd\x35\xfb\xf3\xa8\x55\xb7\x75\x4a\x36\xd4\xa0\x67\x2d\xf9\x5d\x6c\x44\xa8\x1c\xf7\x62\x0c\x2d",
        msg: b"\x6b\x45\xd8\x80\x37\x39\x2e\x13\x71\xd9\xfd\x1c\xd1\x74\xe9\xc1\x83\x8d\x11\xc3\xd6\x13\x3d\xc1\x7e\x65\xfa\x0c\x48\x5d\xcc\xa9\xf5\x2d\x41\xb6\x01\x61\x24\x60\x39\xe4\x2e\xc7\x84\xd4\x94\x00\xbf\xfd\xb5\x14\x59\xf5\xde\x65\x40\x91\x30\x1a\x09\x37\x8f\x93\x46\x4d\x52\x11\x8b\x48\xd4\x4b\x30\xd7\x81\xeb\x1d\xbe\xd0\x9d\xa1\x1f\xb4\xc8\x18\xdb\xd4\x42\xd1\x61\xab\xa4\xb9\xed\xc7\x9f\x05\xe4\xb7\xe4\x01\x65\x13\x95\xb5\x3b\xd8\xb5\xbd\x3f\x2a\xaa\x6a\x00\x87\x7f\xa9\xb4\x5c\xad\xb8\xe6\x48\x55\x0b\x4c\x6c\xbe",
        sig: b"\x50\x83\x5a\x92\x51\xba\xd0\x08\x10\x61\x77\xef\x00\x4b\x09\x1a\x1e\x42\x35\xcd\x0d\xa8\x4f\xff\x54\x54\x2b\x0e\xd7\x55\xc1\xd6\xf2\x51\x60\x9d\x14\xec\xf1\x8f\x9e\x1d\xdf\xe6\x9b\x94\x6e\x32\x04\x75\xf3\xd3\x0c\x64\x63\xb6\x46\xe8\xd3\xbf\x24\x55\x83\x03\x14\x61\x1c\xbd\xe4\x04\xbe\x51\x8b\x14\x46\x4f\xdb\x19\x5f\xdc\xc9\x2e\xb2\x22\xe6\x1f\x42\x6a\x4a\x59\x2c\x00\xa6\xa8\x97\x21",
    },
    TestVector {
        sk: b"\x23\xd9\xf4\xea\x6d\x87\xb7\xd6\x16\x3d\x64\x25\x6e\x34\x49\x25\x5d\xb1\x47\x86\x40\x1a\x51\xda\xa7\x84\x71\x61\xbf\x56\xd4\x94\x32\x5a\xd2\xac\x8b\xa9\x28\x39\x4e\x01\x06\x1d\x88\x2c\x35\x28",
        pk: b"\x5d\x42\xd6\x30\x1c\x54\xa4\x38\xf6\x59\x70\xba\xe2\xa0\x98\xcb\xc5\x67\xe9\x88\x40\x00\x6e\x35\x62\x21\x96\x6c\x86\xd8\x2e\x8e\xca\x51\x5b\xca\x85\x0e\xaa\x3c\xd4\x1f\x17\x5f\x03\xa0\xcb\xfd\x4a\xef\x5a\x0c\xee\xce\x95\xd3\x82\xbd\x70\xab\x5c\xe1\xcb\x77\x40\x8b\xae\x42\xb5\x1a\x08\x81\x6d\x5e\x5e\x1d\x3d\xa8\xc1\x8f\xcc\x95\x56\x4a\x75\x27\x30\xb0\xaa\xbe\xa9\x83\xcc\xea\x4e\x2e",
        msg: b"\xd7\x68\xf4\x1e\x6e\x8e\xc2\x12\x5d\x6c\xf5\x78\x6d\x1b\xa9\x66\x68\xac\x65\x66\xc5\xcd\xbb\xe4\x07\xf7\xf2\x05\x1f\x3a\xd6\xb1\xac\xdb\xfe\x13\xed\xf0\xd0\xa8\x6f\xa1\x10\xf4\x05\x40\x6b\x69\x08\x52\x19\xb5\xa2\x34\xeb\xdb\x93\x15\x32\x41\xf7\x85\xd4\x58\x11\xb3\x54\x0d\x1c\x37\x42\x4c\xc7\x19\x44\x24\x78\x7a\x51\xb7\x96\x79\x26\x64\x84\xc7\x87\xfb\x1d\xed\x6d\x1a\x26\xb9\x56\x7d\x5e\xa6\x8f\x04\xbe\x41\x6c\xaf\x3b\xe9\xbd\x2c\xaf\xa2\x08\xfe\x2a\x9e\x23\x4d\x3a\xe5\x57\xc6\x5d\x3f\xe6\xda\x4c\xb4\x8d\xa4",
        sig: b"\xfb\x31\x8f\x4c\xb1\x27\x62\x82\xbb\x43\xf7\x33\xa7\xfb\x7c\x56\x7c\xe9\x4f\x4d\x02\x92\x4f\xc7\x58\x63\x5a\xb2\xd1\x10\x71\x08\xbf\x15\x9b\x85\xdb\x08\x0c\xdc\x3b\x30\xfb\xb5\x40\x00\x16\xf3\x58\x8e\x3d\x7a\xf5\xda\x03\xea\xe2\x55\xec\xb1\x81\x31\x00\xd9\x5e\xdc\x24\x34\x76\xb7\x24\xb2\x2d\xb8\xe8\x53\x77\x66\x0d\x76\x45\xdd\xc1\xc2\xc2\xee\x4e\xae\xa8\xb6\x83\xdb\xe2\x2f\x86\xca",
    },
    TestVector {
        sk: b"\xb5\xf6\x70\xe9\x8d\x8b\xef\xc4\x6f\x6f\x51\xfb\x29\x97\x06\x95\x50\xc2\xa5\x2e\xbf\xb4\xe5\xe2\x5d\xd9\x05\x35\x2d\x9e\xf8\x9e\xed\x5c\x2e\xcd\x16\x52\x18\x53\xaa\xdb\x1b\x52\xb8\xc4\x2a\xe6",
        pk: b"\x44\xff\xb2\xa3\xa9\x5e\x12\xd8\x7c\x72\xb5\xea\x0a\x8a\x7c\xb8\x9f\x56\xb3\xbd\x46\x34\x2b\x23\x03\x60\x8d\x72\x16\x30\x1c\x21\xb5\xd2\x92\x1d\x80\xb6\x62\x8d\xc5\x12\xcc\xb8\x4e\x2f\xc2\x78\xe4\xc1\x00\x2f\x18\x28\xab\xae\xc7\x68\xca\xdc\xb7\xcf\x42\xfb\xf9\x3b\x17\x09\xcc\xae\x6d\xf5\xb1\x34\xc4\x1f\xae\x2b\x9a\x18\x8b\xfb\xe1\xec\xcf\xf0\xbd\x34\x85\x17\xd7\x22\x7f\x20\x71\xa6",
        msg: b"\x6a\xf6\x65\x2e\x92\xa1\x7b\x78\x98\xe4\x0b\x67\x76\xfa\xba\xf0\xd7\x4c\xf8\x8d\x8f\x0e\xbf\xa6\x08\x83\x09\xcb\xe0\x9f\xac\x47\x2e\xea\xc2\xaa\x8e\xa9\x6b\x8c\x12\xe9\x93\xd1\x4c\x93\xf8\xef\x4e\x8b\x54\x7a\xfe\x7a\xe5\xe4\xf3\x97\x31\x70\xb3\x5d\xeb\x32\x39\x89\x89\x18\xc7\x0c\x10\x56\x33\x2c\x3f\x89\x4c\xd6\x43\xd2\xd9\xb9\x3c\x25\x61\xaa\xc0\x69\x57\x7b\xba\xb4\x58\x03\x25\x0a\x31\xcd\x62\x22\x6c\xab\x94\xd8\xcb\xa7\x26\x1d\xce\x9f\xe8\x8c\x21\x0c\x21\x2b\x54\x32\x9d\x76\xa2\x73\x52\x2c\x8b\xa9\x1d\xdf",
        sig: b"\xb1\x1d\xb5\x92\xe4\xeb\xc7\x5b\x64\x72\xb8\x79\xb1\xd8\xce\x57\x45\x2c\x61\x5a\xef\x20\xf6\x7a\x28\x0f\x8b\xca\x9b\x11\xa3\x0a\xd4\xac\x9d\x69\x54\x12\x58\xc7\xdd\x5d\x0b\x4a\xb8\xdd\x7d\x49\x4e\xb5\x1d\xb8\x00\x4e\x46\xd4\x38\x35\x9a\xbf\x06\x0a\x94\x44\x61\x6c\xb4\x6b\x4f\x99\xc9\xa0\x5b\x53\xba\x6d\xf0\x2e\x91\x4c\x9c\x0b\x6c\xc3\xa9\x79\x1d\x80\x4d\x2e\x4c\x09\x84\xda\xb1\xcc",
    },
    TestVector {
        sk: b"\xde\x59\x75\xd8\x93\x25\x33\xf0\x92\xe7\x62\x95\xed\x6b\x23\xf1\x0f\xc5\xfb\xa4\x8b\xfb\x82\xc6\xcc\x71\x48\x26\xba\xf0\x12\x68\x13\x24\x7f\x8b\xd5\x1d\x57\x38\x50\x36\x54\xab\x22\x45\x99\x76",
        pk: b"\xf1\xfa\xba\xfc\x01\xfe\xc7\xe9\x6d\x98\x25\x28\xd9\xef\x3a\x2a\x18\xb7\xfe\x8a\xe0\xfa\x06\x73\x97\x73\x41\xc7\xae\x4a\xe8\xd8\xd3\xd6\x74\x20\x34\x3d\x01\x3a\x98\x4f\x5f\x61\xda\x29\xae\x38\x1a\x31\xcf\x90\x2c\x46\x34\x3d\x01\xb2\xeb\xb6\x14\xbc\x78\x9c\x31\x3b\x5f\x91\xf9\x30\x2a\xd9\x41\x8e\x9c\x79\x75\x63\xe2\xfa\x3d\x44\x50\x0f\x47\xb4\xe2\x6a\xd8\xfd\xec\x1a\x81\x6d\x1d\xcf",
        msg: b"\xb9\x6d\x74\xb2\x26\x5d\xd8\x95\xd9\x4e\x25\x09\x2f\xb9\x26\x2d\xc4\xf2\xf7\xa3\x28\xa3\xc0\xc3\xda\x13\x4b\x2d\x0a\x4e\x20\x58\xca\x99\x4e\x34\x45\xc5\xff\x4f\x81\x27\x38\xe1\xb0\xc0\xf7\xa1\x26\x48\x69\x42\xa1\x2e\x67\x4a\x21\xf2\x2d\x08\x86\xd6\x8d\xf2\x37\x5f\x41\x68\x5d\x69\x4d\x48\x7a\x71\x80\x24\x93\x3a\x7c\x43\x06\xf3\x3f\x1a\x42\x67\xd4\x69\xc5\x30\xb0\xfe\xd4\xe7\xde\xa5\x20\xa1\x9d\xd6\x8b\xf0\x20\x3c\xc8\x7c\xad\x65\x22\x60\xed\x43\xb7\xb2\x3f\x6e\xd1\x40\xd3\x08\x58\x75\x19\x01\x91\xa0\x38\x1a",
        sig: b"\xc2\xfb\xdd\x6a\x56\x78\x90\x24\x08\x21\x73\x72\x5d\x79\x7e\xf9\xfd\x6a\xcc\xb6\xae\x66\x4b\x72\x60\xf9\xe8\x3c\xb8\xab\x24\x90\x42\x8c\x8b\x9c\x52\xe1\x53\x61\x22\x95\x43\x2f\xec\x4d\x59\xcd\x80\x56\xc5\xbb\x57\xf4\x1f\x73\x08\x28\x88\xb2\x34\xfc\xda\x32\x0a\x33\x25\x0b\x5d\xa0\x12\xba\x1f\xdb\x49\x24\x35\x5a\xe6\x79\x01\x2d\x81\xd2\xc0\x8f\xc0\xf8\x63\x4c\x70\x8a\x48\x33\x23\x2f",
    },
    TestVector {
        sk: b"\x11\xe0\xd4\x70\xdc\x31\xfa\xb0\xf5\x72\x2f\x87\xb7\x4a\x6c\x8d\x74\x14\x11\x5e\x58\xce\xb3\x8b\xfc\xdc\xed\x36\x7b\xea\xc3\xad\xbf\x1f\xe9\xba\x5a\x04\xf7\x2e\x97\x8b\x1e\xb5\x45\x97\xea\xbc",
        pk: b"\x19\x50\x16\x69\x89\x16\x4c\xbf\xd9\x79\x68\xc7\xe8\xad\xb6\xfb\xca\x18\x73\xeb\xef\x81\x1e\xa2\x59\xeb\x48\xb7\xd5\x84\x62\x7f\x0e\x6d\x6c\x64\xde\xfe\x23\xcb\xc9\x52\x36\x50\x5a\x25\x2a\xa1\x41\xef\x42\x4b\x5c\xb0\x76\xd4\xe3\x2a\xcc\xd9\x25\x0e\xa7\x5f\xcf\x4f\xfd\x81\x81\x40\x40\xc0\x50\xd5\x8c\x0a\x29\xb0\x6b\xe1\x1e\xdf\x67\xc9\x11\xb4\x03\xe4\x18\xb7\x27\x74\x17\xe5\x29\x06",
        msg: b"\x7c\xec\x74\x80\xa0\x37\xff\x40\xc2\x32\xc1\xd2\xd6\xe8\xcd\x4c\x08\x0b\xbe\xec\xda\xf3\x88\x6f\xcc\xc9\xf1\x29\xbb\x6d\x20\x2c\x31\x6e\xca\x76\xc8\xad\x4e\x76\x07\x9a\xfe\x62\x2f\x83\x3a\x16\xf4\x90\x7e\x81\x72\x60\xc1\xfa\x68\xb1\x0c\x7a\x15\x1a\x37\xeb\x8c\x03\x6b\x05\x7e\xd4\x65\x2c\x35\x3d\xb4\xb4\xa3\x4b\x37\xc9\xa2\xb3\x00\xfb\x5f\x5f\xcf\xb8\xaa\x8a\xda\xe1\x3d\xb3\x59\x16\x0f\x70\xa9\x24\x15\x46\x14\x0e\x55\x0a\xf0\x07\x34\x68\x68\x33\x77\xe6\x77\x1b\x65\x08\x32\x74\x08\xc2\x45\xd7\x89\x11\xc2\xcc",
        sig: b"\xc3\x8e\xf3\x0f\x55\x62\x4e\x89\x35\x68\x0c\x29\xf8\xc2\x48\x24\x87\x7c\xf4\x8f\xfc\x0e\xf0\x15\xe6\x2d\xe1\x06\x88\x93\x35\x30\x30\xd1\x19\x3b\xf9\xd3\x42\x37\xd7\xce\x6b\xa9\x2c\x98\xb0\xfe\x65\x1b\x8c\x3d\x5c\x9d\x5b\x93\x6d\x30\x08\x02\xa0\x6d\x82\xad\x54\xf7\xb1\xba\x43\x27\xb2\xf0\x31\xc0\xc5\xb0\xcb\x21\x5a\xd4\x35\x4e\xdc\x7f\x93\x2d\x93\x4e\x87\x7d\xfa\x1c\xf5\x1b\x13\xfe",
    },
    TestVector {
        sk: b"\x5c\x6b\xbf\x9f\xbc\xbb\x7b\x97\xc9\x53\x5f\x57\xb4\x31\xed\x1c\xca\xe1\x94\x5b\x7e\x8a\x4f\x1b\x03\x20\x16\xb0\x78\x10\xbd\x24\xa9\xe2\x00\x55\xc0\xe9\x30\x66\x50\xdf\x59\xef\x7e\x2c\xd8\xc2",
        pk: b"\x2e\x01\xc5\xb5\x9e\x61\x9e\x00\xb7\x90\x60\xa1\xe8\xef\x69\x54\x72\xe2\x3b\xf9\xa5\x11\xfc\x3d\x5e\xd7\x7a\x33\x4a\x24\x25\x57\x09\x8e\x40\x97\x27\x13\x73\x2c\x52\x91\xc9\x7a\xdf\x9c\xf2\xcf\x56\x3e\x3f\xe4\xad\x80\x7e\x80\x3b\x9e\x96\x1b\x08\xda\x4d\xde\x4c\xea\x89\x25\x64\x9d\xa0\xd9\x32\x21\xce\x4c\xdc\xea\xbc\x6a\x1d\xb7\x61\x21\x80\xa8\xc6\xbe\xf3\x57\x9c\x65\x53\x9b\x97\xe9",
        msg: b"\x00\xce\x97\x86\x03\x22\x97\x10\x34\x5c\x9a\xd7\xc1\xc2\xdb\xa3\x59\x6b\x19\x65\x28\xee\xa2\x5b\xd8\x22\xd4\x3c\xa8\xf7\x6a\x02\x4e\x29\x21\x77\x03\xdd\x06\x52\xc8\xa6\x15\x28\x4f\xc3\xed\xcc\x1c\x5a\xd1\xc8\xd5\xa8\x52\x1c\x8e\x10\x4c\x01\x6a\x24\xe5\x0c\x2e\x25\x06\x6d\xcb\x56\x59\x6f\x91\x3b\x87\x27\x67\xe3\x62\x7a\xa3\xe5\x5e\xc8\x12\xe9\xfd\xac\x7c\x2f\x1b\xea\xde\x83\xae\xf0\x93\xe2\x4c\x9c\x95\x39\x82\xad\xf4\x31\xa7\x76\x88\x0a\xe4\x58\x3b\xe1\x58\xe1\x1c\xda\xb1\xcb\xca\x3a\xd3\xa6\x69\x00\x21\x3d",
        sig: b"\xab\xab\x65\x30\x8f\x0b\x79\xc4\xf3\xa9\xff\x28\xdd\x49\x0a\xcb\x0c\x32\x04\x34\x09\x4c\xef\x93\xe7\x5a\xdf\xe1\x7e\x58\x20\xdc\x1f\x77\x54\x4c\xfa\xaa\xcd\xc8\xcf\x9a\xc8\xb3\x8e\x17\x4b\xef\x11\xb7\x83\xd8\x79\xa6\xde\x05\x4b\x31\x6a\xf7\xd5\x6e\x52\x6c\x3d\xce\x96\xc8\x52\x89\x12\x2e\x3a\xd9\x27\xcf\xa7\x7b\xfc\x50\xb4\xa9\x6c\x97\xf8\x5b\x1b\x82\x21\xbe\x2d\xf0\x83\xff\x58\xfb",
    },
    TestVector {
        sk: b"\xff\xc7\xde\xde\xff\x83\x43\x72\x1f\x72\x04\x6b\xc3\xc1\x26\x62\x6c\x17\x7b\x0e\x48\xe2\x47\xf4\x4f\xd6\x1f\x84\x69\xd4\xd5\xf0\xa7\x41\x47\xfa\xba\xa3\x34\x49\x5c\xc1\xf9\x86\xeb\xc5\xf0\xb1",
        pk: b"\x51\xc7\x8c\x97\x94\x52\xed\xd5\x3b\x56\x3f\x63\xeb\x3e\x85\x4a\x5b\x23\xe8\x7f\x1b\x21\x03\x94\x2b\x65\xf7\x7d\x02\x44\x71\xf7\x5c\x8c\xe1\xcc\x0d\xfe\xf8\x32\x92\xb3\x68\x11\x2a\xa5\x12\x6e\x31\x3e\x6a\xaf\x09\xca\xa3\xba\x30\xf1\x30\x72\xb2\x13\x48\x78\xf1\x4a\x4a\x01\xee\x86\x32\x6c\xcc\xbf\xf3\xd0\x79\xb4\xdf\x09\x7d\xc5\x79\x85\xe8\xc8\xc8\x34\xa1\x0c\xb9\xd7\x66\x16\x93\x66",
        msg: b"\x54\xa2\x55\xc1\x86\x92\xc6\x16\x2a\x46\xad\xd1\x76\xa0\xae\x83\x61\xdc\xb8\x94\x8f\x09\x2d\x8d\x7b\xac\x83\xe1\x60\x43\x17\x94\xd3\xb9\x81\x28\x49\xbf\x19\x94\xbc\xdc\xfb\xa5\x6e\x85\x40\xc8\xa9\xee\x5b\x93\x41\x45\x48\xf2\xa6\x53\x19\x1b\x6b\xb2\x8b\xda\x8d\xc7\x0d\x45\xcc\x1b\x92\xa4\x89\xf5\x8a\x2d\x54\xf8\x57\x66\xcb\x3c\x90\xde\x7d\xd8\x8e\x69\x0d\x8e\xbc\x9a\x79\x98\x7e\xee\x19\x89\xdf\x35\xaf\x5e\x35\x52\x2f\x83\xd8\x5c\x48\xdd\xa8\x98\x63\x17\x1c\x8b\x0b\xf4\x85\x3a\xe2\x8c\x2a\xc4\x5c\x76\x44\x16",
        sig: b"\xf4\xf4\x77\x85\x58\x19\xad\x8b\x17\x63\xf5\x36\x91\xb7\x6a\xfb\xc4\xa3\x1a\x63\x8b\x1e\x08\xc2\x93\xf9\xbc\xd5\x5d\xec\xf7\x97\xf9\x91\x3c\xa1\x28\xd4\xb4\x5b\x2e\x2e\xa3\xe8\x2c\x6c\xf5\x65\x7c\x26\xbe\x29\x56\x9e\xf9\x54\x80\xa6\xd0\xc1\xaf\x49\xdc\x10\xa5\x1a\x0a\x89\x31\x34\x5e\x48\xc0\xc3\x94\x98\xbf\xb9\x4d\x62\x96\x29\x80\xb5\x61\x43\xa7\xb4\x1a\x2f\xdd\xc8\x79\x4c\x1b\x7f",
    },
    TestVector {
        sk: b"\xad\xca\x36\x4e\xf1\x44\xa2\x1d\xf6\x4b\x16\x36\x15\xe8\x34\x9c\xf7\x4e\xe9\xdb\xf7\x28\x10\x42\x15\xc5\x32\x07\x3a\x7f\x74\xe2\xf6\x73\x85\x77\x9f\x7f\x74\xab\x34\x4c\xc3\xc7\xda\x06\x1c\xf6",
        pk: b"\xef\x94\x8d\xaa\xe6\x82\x42\x33\x0a\x73\x58\xef\x73\xf2\x3b\x56\xc0\x7e\x37\x12\x62\x66\xdb\x3f\xa6\xee\xa2\x33\xa0\x4a\x9b\x3e\x49\x15\x23\x3d\xd6\x75\x44\x27\xcd\x4b\x71\xb7\x58\x54\x07\x7d\x00\x94\x53\xef\x18\x28\xea\xff\x9e\x17\xc8\x56\xd4\xfc\x18\x95\xab\x60\x05\x13\x12\xc3\xe1\xdb\x1e\x37\x66\x56\x64\x38\xb2\x99\x0c\xbf\x99\x45\xc2\x54\x56\x19\xe3\xe0\x14\x5b\xc6\xa7\x90\x04",
        msg: b"\x69\x2a\x78\xf9\x0d\x4f\x9d\x5a\xee\x5d\xa5\x36\x31\x4a\x78\xd6\x8c\x1f\xea\xbb\xfe\x5d\x1c\xce\xa7\xf6\x05\x9a\x66\xc4\xb3\x10\xf8\x05\x1c\x41\x1c\x40\x9c\xcf\x6e\x19\xa0\xcb\xd8\xb8\xe1\x00\xc4\x83\x17\xfe\x8c\x6d\x4f\x8a\x63\x8b\x95\x51\xce\x7e\xe1\x78\x02\x0f\x04\xf7\xda\x30\x01\xa0\xe6\x85\x52\x25\xfb\x3c\x9b\x37\x5e\x4e\xd9\x64\x58\x8a\x1a\x41\xa0\x95\xf3\xf4\x76\xc4\x2d\x52\xff\xd2\x3c\xe1\x70\x2c\x93\xb5\x6d\x44\x25\xd3\xbe\xfc\xf7\x5d\x09\x51\xb6\xfd\x5c\x05\xb0\x54\x55\xbd\xaf\x20\x5f\xe7\x0c\xa2",
        sig: b"\xdd\xa9\x94\xb9\xc4\x28\xb5\x7e\x9f\x8b\xba\xeb\xba\x0d\x68\x2e\x3a\xac\x6e\xd8\x28\xe3\xa1\xe9\x9a\x7f\xc4\xc8\x04\xbf\xf8\xdf\x15\x11\x37\xf5\x39\xc7\x38\x9d\x80\xe2\x3d\x9f\x3e\xe4\x97\xbf\xa0\xd6\xb1\x0c\xef\xfd\x0e\x1b\x29\xcf\x78\x44\x76\xf9\x17\x3b\xa6\xec\xd2\xcf\xc7\x92\x97\x25\xf2\xd6\xe2\x4e\x0d\xb5\xa4\x72\x16\x83\x64\x0e\xaa\x2b\xbe\x15\x1f\xb5\x75\x60\xf9\xce\x59\x4b",
    },
    TestVector {
        sk: b"\x39\xbe\xa0\x08\xec\x8a\x21\x78\x66\xdc\xbd\xb1\xb9\x3d\xa3\x4d\x1d\x3e\x85\x1d\x01\x1d\xf9\xef\x44\xb7\x82\x8b\x34\x53\xa5\x4a\xa7\x0f\x1d\xf9\x93\x21\x70\x80\x4e\xac\xd2\x07\xe4\xf7\xe9\x1d",
        pk: b"\x57\x09\xec\x43\x05\xa9\xc3\x27\x1c\x30\x4f\xac\xe6\xc1\x48\x14\x24\x90\xb8\x27\xa7\x3a\x4c\x17\xaf\xfc\xfd\x01\xff\xfd\x7e\xaa\x65\xd2\xfd\xed\xfa\x24\x19\xfc\x64\xed\x91\x08\x23\x51\x3f\xaf\xb0\x83\xcd\xa1\xcf\x3b\xe6\x37\x1b\x6c\x06\xe7\x29\xea\x62\x99\x21\x34\x28\xdb\x57\x11\x93\x47\x24\x7e\xc1\xfc\xd4\x42\x04\x38\x6c\xc0\xbc\xa3\xf4\x52\xd9\xd8\x64\xb3\x9e\xfb\xfc\x89\xd6\xb2",
        msg: b"\x3b\x30\x9b\xb9\x12\xab\x2a\x51\x68\x14\x51\xed\x18\xad\x79\xe9\x5d\x96\x8a\xbc\x35\x42\x3a\x67\x03\x6a\x02\xaf\x92\xf5\x75\xa0\xc8\x9f\x1b\x66\x8a\xfe\x22\xc7\x03\x7a\xd1\x19\x9e\x75\x7a\x8f\x06\xb2\x81\xc3\x3e\x9a\x40\xba\xb6\x9c\x98\x74\xe0\xbb\x68\x0b\x90\x5d\x90\x9b\x9d\xc2\x4a\x9f\xe8\x9b\xb3\xd7\xf7\xd4\x70\x82\xb2\x50\x93\xc5\x97\x54\xf8\xc1\x9d\x1f\x81\xf3\x03\x34\xa8\xcd\xd5\x0a\x3c\xb7\x2f\x96\xd4\xb3\xc3\x05\xe6\x0a\x43\x9a\x7e\x93\xae\xb6\x40\xdd\x3c\x8d\xe3\x7d\x63\xc6\x0f\xb4\x69\xc2\xd3\xed",
        sig: b"\xd1\x36\x46\x89\x5a\xfb\x1b\xfd\x19\x53\x55\x1b\xb9\x22\x80\x9c\x95\xad\x65\xd6\xab\xe9\x4e\xb3\x71\x9c\x89\x9a\xa1\xf6\xdb\xa6\xb0\x12\x22\xc7\xf2\x83\x90\x0f\xe9\x86\x28\xb7\x59\x7b\x6e\xa6\x4a\x9a\x38\xaf\xda\x04\xc0\xa6\xb0\x05\x89\x43\xb6\x79\xbd\x02\x20\x5b\x14\xd0\xf3\xd4\x9b\x8f\x31\xaa\xc2\x89\x12\x97\x80\xcd\xb1\xc5\x55\xde\xf8\xc3\xf9\x10\x6b\x47\x87\x29\xe0\xc7\xef\xaa",
    },
    TestVector {
        sk: b"\xe8\x49\xcf\x94\x8b\x24\x13\x62\xe3\xe2\x0c\x45\x8b\x52\xdf\x04\x4f\x2a\x72\xde\xb0\xf4\x1c\x1b\xb0\x67\x3e\x7c\x04\xcd\xd7\x08\x11\x21\x50\x59\x03\x2b\x5c\xa3\xcc\x69\xc3\x45\xdc\xce\x4c\xf7",
        pk: b"\x06\xc0\x37\xa0\xcb\xf4\x3f\xdf\x33\x5d\xff\x33\xde\x06\xd3\x43\x48\x40\x53\x53\xf9\xfd\xf2\xce\x13\x61\xef\xba\x30\xfb\x20\x4a\xea\x9d\xbd\x2e\x30\xda\x0a\x10\xfd\x2d\x87\x61\x88\x37\x1b\xe6\x36\x0d\x38\xf3\x94\x0e\x34\x67\x92\x04\xb9\x8f\xbf\x70\xb8\xa4\xd9\x7f\x25\x44\x3e\x46\xd0\x80\x7a\xb6\x34\xed\x58\x91\xad\x86\x4d\xd7\x70\x35\x57\xaa\x93\x3c\xd3\x80\xe2\x6e\xea\x66\x2a\x43",
        msg: b"\xf0\x72\xb7\x2b\x87\x83\x28\x94\x63\xda\x11\x86\x13\xc4\x38\x24\xd1\x14\x41\xdb\xa3\x64\xc2\x89\xde\x03\xff\x5f\xab\x3a\x6f\x60\xe8\x59\x57\xd8\xff\x21\x1f\x1c\xb6\x2f\xa9\x02\x16\xfb\x72\x71\x06\xf6\x92\xe5\xae\x08\x44\xb1\x1b\x71\x0e\x5a\x12\xc6\x9d\xf3\xed\x89\x5b\x94\xe8\x76\x9e\xcd\x15\xff\x43\x37\x62\xd6\xe8\xe9\x4d\x8e\x6a\x72\x64\x5b\x21\x3b\x02\x31\x34\x4e\x2c\x96\x80\x56\x76\x6c\x5d\xd6\xb5\xa5\xdf\x41\x97\x18\x58\xb8\x5e\x99\xaf\xbf\x85\x94\x00\xf8\x39\xb4\x2c\xd1\x29\x06\x8e\xfa\xbe\xea\x4a\x26",
        sig: b"\x58\x86\x07\x8d\x34\x95\x76\x7e\x33\x0c\x75\x07\xb7\xca\x0f\xa0\x7a\x50\xe5\x99\x12\xa4\x16\xd8\x9f\x0a\xb1\xaa\x4e\x88\x15\x3d\x6e\xaf\x00\x88\x2d\x1b\x4a\xa6\x41\x53\x15\x33\x52\xd8\x53\xb5\x2c\xc1\x00\x23\xbf\x1b\xf8\xcc\xfd\x14\xb0\x6b\x82\xcc\x21\x14\x44\x9a\x35\x23\x89\xc8\xff\x9f\x6f\x78\xcd\xc4\xe3\x2b\xde\x69\xf3\x86\x9d\xa0\xe1\x7f\x69\x1b\x32\x96\x82\xae\x7a\x36\xe1\xaa",
    },
    TestVector {
        sk: b"\xd8\x96\x07\x47\x5d\x50\x9e\xf2\x3d\xc9\xf4\x76\xea\xe4\x28\x0c\x98\x6d\xe7\x41\xb6\x35\x60\x67\x0f\xa2\xbd\x60\x5f\x50\x49\xf1\x97\x27\x92\xc0\x41\x3a\x5b\x3b\x4b\x34\xe7\xa3\x8b\x70\xb7\xca",
        pk: b"\x49\xa1\xc6\x31\xf3\x1c\xf5\xc4\x5b\x26\x76\xb1\xf1\x30\xcb\xf9\xbe\x68\x3d\x0a\x50\xdf\xfa\xe0\xd1\x47\xc1\xe9\x91\x3a\xb1\x09\x0c\x65\x29\xa8\x4f\x47\xdd\xc7\xcf\x02\x59\x21\xb7\x71\x35\x5a\x1e\x20\x7e\xec\xe6\x2f\x2b\xcc\x6b\xda\xbc\x11\x13\x15\x81\x45\x17\x0b\xe9\x74\x69\xa2\x90\x4e\xaa\xa9\x3a\xad\x85\xb8\x6a\x19\x71\x92\x07\xf3\xe4\x23\x05\x1f\x5b\x9c\xbb\xe2\x75\x4e\xef\xcb",
        msg: b"\xcf\x49\x45\x35\x0b\xe8\x13\x3b\x57\x5c\x4a\xd6\xc9\x58\x5e\x0b\x83\xff\x1e\xd1\x79\x89\xb6\xcd\x6c\x71\xb4\x1b\x52\x64\xe8\x28\xb4\xe1\x15\x99\x5b\x1a\xe7\x75\x28\xe7\xe9\x00\x2a\xc1\xb5\x66\x90\x64\x44\x26\x45\x92\x9f\x9d\x7d\xd7\x09\x27\xcb\x93\xf9\x5e\xde\xb7\x3e\x86\x24\xf4\xbc\x89\x7e\xc4\xc2\xc7\x58\x1c\xb6\x26\x91\x6f\x29\xb2\xd6\xe6\xc2\xfb\xa8\xc5\x9a\x71\xe3\x07\x54\xb4\x59\xd8\x1b\x91\x2a\x12\x79\x81\x82\xbc\xff\x40\x19\xc7\xbd\xfe\x92\x9c\xc7\x69\xbc\xc2\x41\x4b\xef\xe7\xd2\x90\x6a\xdd\x42\x71",
        sig: b"\x66\xf9\x2b\x39\xaa\x3f\x4a\xeb\x9e\x2d\xc0\x3a\xc3\x85\x54\x06\xfa\x3e\xbb\xab\x0a\x6c\x88\xa7\x8d\x7a\x03\x48\x2f\x0c\x98\x68\xd7\xb7\x8b\xc0\x81\xed\xe0\x94\x7c\x7f\x37\xbf\x19\x30\x74\xba\xe5\xc6\x4e\xd9\x8d\x7f\x37\x01\x19\x3f\x25\xdd\x23\x7d\x59\xc9\x1c\x0d\xa6\xe2\x62\x15\xe0\x88\x9d\x82\xe6\xd3\xe4\x16\x69\x3f\x8d\x58\x84\x3c\xf3\x0a\xb1\x0a\xb8\xd0\xed\xd9\x17\x0b\x53\xad",
    },
    TestVector {
        sk: b"\x08\x3e\x71\x52\x73\x4a\xdf\x34\x25\x20\xae\x37\x70\x87\xa2\x23\x68\x8d\xe2\x89\x9b\x10\xcf\xcb\x34\xa0\xb3\x6b\xca\x50\x0a\x4d\xfa\x53\x0e\x23\x43\xe6\xa3\x9d\xa7\xae\x1e\xb0\x86\x2b\x4a\x0d",
        pk: b"\x70\xa0\xf1\x6b\x6c\x61\x17\x26\x59\xb0\x27\xed\x19\xb1\x8f\xd8\xf5\x7b\xd2\x8d\xc0\x50\x1f\x20\x7b\xd6\xb0\xbb\x06\x5b\x56\x71\xcf\x3d\xd1\xed\x13\xd3\x88\xdc\xf6\xcc\xc7\x66\x59\x7a\xa6\x04\x4f\x84\x5b\xf0\x1c\x3c\x3f\x61\x26\xa7\x36\x8c\x34\x54\xf5\x14\x25\x80\x1e\xe0\xb7\x2e\x63\xfb\x67\x99\xb4\x42\x0b\xfd\xeb\xe3\xe3\x7c\x72\x46\xdb\x62\x7c\xc8\x2c\x09\x65\x49\x79\xc7\x00\xbb",
        msg: b"\xd9\xb5\xcf\x0b\x50\x41\x65\x73\xff\x3c\x63\x13\x32\x75\xa1\x83\x94\xdd\x43\x26\xbe\x20\x41\xe8\xd9\x7e\x6e\x4e\x38\x55\xa4\xa1\x77\xe9\xd2\x6d\xfd\x22\x3f\xe8\xaa\x74\x56\x4e\xdb\x49\xbd\x72\xde\x19\x91\x6f\xb6\xf0\x01\xf4\x45\x30\xd5\xc1\x8e\x2c\x33\x2b\xce\x1b\x74\x15\xdf\x59\x27\xec\xe5\xf3\x82\x4f\x34\xd1\x74\xb9\x63\x13\x6b\x53\xae\xf1\xfb\x78\xfb\x0c\x06\xa2\x01\xa4\x0b\x2d\xb3\x8e\x4d\x82\x16\xfc\x1e\x39\x2a\x79\x8c\x8a\xb4\xb3\xa3\x14\x49\x6b\x7f\x10\x87\x80\x4e\xbf\xa8\x9b\xf9\x6e\x9c\xdb\x80\xc0",
        sig: b"\xee\x29\x23\xf9\xb9\x99\x9e\xa0\x5b\x5e\x57\xf5\x05\xbe\xd5\xc6\xba\x04\x20\xde\xf4\x2c\x6f\xa9\x0e\xef\x7a\x6e\xf7\x70\x78\x65\x25\x54\x6d\xe2\x7c\xde\xb2\xf8\x58\x6f\x8f\x29\xfb\x4e\xe6\x7c\x50\xef\x92\x3f\xb2\x17\xc4\xcf\x65\xa4\x8b\x94\x41\x2f\xda\x43\x0f\xac\x68\x5f\x0d\xa7\xbd\x57\x45\x57\xc6\xc5\x0f\x5b\x22\xe0\xc8\x35\x4d\x99\xf2\xc2\xf2\xc2\x69\x1f\x25\x2f\x93\xc7\xd8\x4a",
    },
    TestVector {
        sk: b"\x63\x57\x8d\x41\x62\x15\xaf\xf2\xcc\x78\xf9\xb9\x26\xd4\xc7\x74\x0a\x77\xc1\x42\x94\x4e\x10\x4a\xa7\x42\x2b\x19\xa6\x16\x89\x82\x62\xd4\x6a\x8a\x94\x2d\x5e\x8d\x5d\xb1\x35\xee\x8b\x09\xa3\x68",
        pk: b"\xca\xdb\xac\xef\x44\x06\x09\x93\x16\xdb\x2c\xe3\x20\x6a\xdc\x63\x6c\x2b\xb0\xa8\x35\x84\x7e\xd7\x94\x1e\xfb\x02\x86\x24\x72\xf3\x15\x03\x38\xf1\x3f\x48\x60\xd4\x7f\x39\xb7\xe0\x98\xf0\xa3\x90\x75\x2a\xd0\xf2\x2c\x9c\x26\x43\x36\xcd\xe1\x1b\xbc\x95\xd1\x81\x6e\xd4\xd1\xb1\x50\x0d\xb6\xb8\xdc\xe2\x59\xa4\x28\x32\xe6\x13\xc3\x11\x78\xc2\xc7\x99\x52\x06\xa6\x2e\x20\x1b\xa1\x08\xf5\x70",
        msg: b"\x9e\x40\x42\xd8\x43\x8a\x40\x54\x75\xb7\xda\xb1\xcd\x78\x3e\xb6\xce\x1d\x1b\xff\xa4\x6a\xc9\xdf\xda\x62\x2b\x23\xac\x31\x05\x7b\x92\x2e\xce\xd8\xe2\xed\x7b\x32\x41\xef\xea\xfd\x7c\x9a\xb3\x72\xbf\x16\x23\x0f\x71\x34\x64\x7f\x29\x56\xfb\x79\x39\x89\xd3\xc8\x85\xa5\xae\x06\x4e\x85\xed\x97\x1b\x64\xf5\xf5\x61\xe7\xdd\xb7\x9d\x49\xaa\x6e\xbe\x72\x7c\x67\x1c\x67\x87\x9b\x79\x45\x54\xc0\x4d\xe0\xe0\x5d\x68\x26\x48\x55\x74\x5e\xf3\xc9\x56\x7b\xd6\x46\xd5\xc5\xf8\x72\x8b\x79\x7c\x18\x1b\x6b\x6a\x87\x6e\x16\x76\x63",
        sig: b"\xdb\x05\x4a\xdd\xb6\x16\x1e\xe4\x9c\x6c\xe2\xe4\xd6\x46\xd7\x67\x07\x54\x74\x7b\x67\x37\xca\x85\x16\xe9\xd1\xe8\x78\x59\x93\x7c\x3e\xf9\xb1\xd2\x66\x3e\x10\xd7\xe4\xbd\x00\xec\x85\xb7\xa9\x7a\xfc\xc5\x04\xe0\xf0\x0e\xf2\x95\x87\xe4\xbc\x22\xfa\xad\xa4\xdb\x30\xe2\xcb\x1a\xc5\x52\x68\x0a\x65\x78\x5a\xe8\x7b\xeb\x66\x6c\x79\x25\x13\xf2\xbe\x7a\x31\x80\xfc\x54\x42\x96\x84\x1a\x0e\x27",
    },
    TestVector {
        sk: b"\xed\x4d\xf1\x99\x71\x65\x8b\x74\x86\x88\x00\xb3\xb8\x1b\xc8\x77\x80\x77\x43\xb2\x5c\x65\x74\x0f\x1d\x63\x77\x54\x2a\xfe\x2c\x64\x27\x61\x2c\x84\x0a\xda\x31\xa8\xeb\x79\x47\x18\xf3\x7c\x72\x83",
        pk: b"\x33\x09\x3a\x05\x68\x75\x7e\x8b\x58\xdf\x5b\x72\xea\x5f\xe5\xbf\x26\xe6\xf7\xae\xb5\x41\xb4\xc6\xa8\xc1\x89\xc9\x37\x21\x74\x9b\xca\xce\xcc\xf2\x98\x2a\x2f\x07\x02\x58\x6a\x9f\x81\x2f\xc6\x6f\xeb\xe3\x20\xd0\x9e\x1f\x06\x62\x18\x9d\x50\xb8\x5a\x20\x40\x3b\x82\x1a\xc0\xd0\x00\xaf\xdb\xf6\x6a\x0a\x33\xf3\x04\x72\x6c\x69\xe3\x54\xd8\x1c\x50\xb9\x4b\xa3\xa5\x25\x0e\xfc\x31\x31\x9c\xd1",
        msg: b"\x0b\x14\xa7\x48\x4a\x40\xb6\x8a\x3c\xe1\x27\x3b\x8a\x48\xb8\xfd\xb6\x5b\xa9\x00\xd9\x85\x41\xc4\xbb\xd0\x7b\x97\xe3\x1b\xcc\x4c\x85\x54\x5a\x03\xe9\xde\xab\x3c\x56\x3f\x47\xa0\x36\xff\x60\xd0\x36\x16\x84\xba\x24\x1b\x5a\xa6\x8b\xb4\x6f\x44\x0d\xa2\x21\x81\xee\x32\x8a\x01\x1d\xe9\x8e\xff\x34\xba\x23\x5e\xc1\x06\x12\xb0\x7b\xdf\xa6\xb3\xdc\x4c\xcc\x5e\x82\xd3\xa8\xd0\x57\xe1\x86\x2f\xef\x3d\xef\x5a\x18\x04\x69\x6f\x84\x69\x9f\xda\x2e\xc4\x17\x5a\x54\xa4\xd0\x8b\xcb\x4f\x04\x06\xfd\xac\x4e\xdd\xad\xf5\xe2\x9b",
        sig: b"\x00\x9c\x74\x06\x3e\x20\x6a\x42\x59\xb5\x3d\xec\xff\x54\x45\x68\x3a\x03\xf4\x4f\xa6\x72\x52\xb7\x6b\xd3\x58\x10\x81\xc7\x14\xf8\x82\xf8\x82\xdf\x91\x5e\x97\xdb\xea\xb0\x61\xfa\x8b\x3c\xc4\xe7\xd4\x0e\x09\xd3\x46\x8b\x46\x69\x99\x48\x00\x7e\x8f\x59\x84\x57\x66\xdb\xf6\x94\xb9\xc6\x20\x66\x89\x0d\xd0\x55\xc0\xcb\x9a\x0c\xaf\x0a\xa6\x11\xfb\x9f\x46\x6a\xd0\xbb\xb0\x0d\xbe\x29\xd7\xeb",
    },
    TestVector {
        sk: b"\xe9\xc7\xe9\xa7\x96\x18\xd6\xff\x32\x74\xda\x1a\xbd\x0f\xf3\xed\x0e\xc1\xae\x3b\x54\xc3\xa4\xfd\x8d\x68\xd9\x8f\xb0\x43\x26\xb7\x63\x3f\xc6\x37\xe0\xb1\x95\x22\x8d\x0e\xdb\xa6\xbb\x14\x68\xfb",
        pk: b"\xa3\x9a\xc3\x53\xca\x78\x79\x82\xc5\x77\xaf\xf1\xe8\x60\x1c\xe1\x92\xaa\x90\xfd\x0d\xe4\xc0\xed\x62\x7f\x66\xa8\xb6\xf0\x2a\xe5\x13\x15\x54\x3f\x72\xff\xc1\xc4\x8a\x72\x69\xb2\x5e\x7c\x28\x9a\x90\x64\xa5\x07\xb6\x6b\x34\x0b\x6e\x0e\x0d\x5f\xfa\xa6\x7d\xd2\x0e\x6d\xaf\xc0\xea\x6a\x6f\xae\xe1\x63\x51\x77\xaf\x25\x6f\x91\x08\xa2\x2e\x9e\xdf\x73\x6a\xb4\xae\x8e\x96\xdc\x20\x7b\x1f\xa9",
        msg: b"\x0e\x64\x6c\x6c\x3c\xc0\xf9\xfd\xed\xef\x93\x4b\x71\x95\xfe\x38\x37\x83\x6a\x9f\x6f\x26\x39\x68\xaf\x95\xef\x84\xcd\x03\x57\x50\xf3\xcd\xb6\x49\xde\x74\x5c\x87\x4a\x6e\xf6\x6b\x3d\xd8\x3b\x66\x06\x8b\x43\x35\xbc\x0a\x97\x18\x41\x82\xe3\x96\x5c\x72\x2b\x3b\x1a\xee\x48\x8c\x36\x20\xad\xb8\x35\xa8\x14\x0e\x19\x9f\x4f\xc8\x3a\x88\xb0\x28\x81\x81\x6b\x36\x6a\x09\x31\x6e\x25\x68\x52\x17\xf9\x22\x11\x57\xfc\x05\xb2\xd8\xd2\xbc\x85\x53\x72\x18\x3d\xa7\xaf\x3f\x0a\x14\x14\x8a\x09\xde\xf3\x7a\x33\x2f\x8e\xb4\x0d\xc9",
        sig: b"\xee\x82\xc0\xf9\x05\x01\x13\x6e\xb0\xdc\x0e\x45\x9a\xd1\x7b\xf3\xbe\x1b\x1c\x8b\x8d\x05\xc6\x00\x68\xa9\x30\x6a\x34\x63\x26\xff\x73\x44\x77\x6a\x95\xf1\xf7\xe2\xe2\xcf\x94\x77\x13\x0e\x73\x5c\xaf\x10\xb9\x0f\x20\x3a\xf2\x3b\x75\x00\xe0\x70\x53\x6e\x64\x62\x9b\xa1\x92\x45\xd6\xef\x39\xaa\xb5\x7f\xcd\xb1\xb7\x3c\x4c\x6b\xf7\x07\x0c\x62\x63\x54\x46\x33\xd3\xd3\x58\xc1\x2a\x17\x81\x38",
    },
];
