// SPDX-License-Identifier: MIT

use netlink_packet_utils::{nla::DefaultNla, Emitable, Parseable};

use crate::link::link_flag::LinkFlags;
use crate::link::{
    AfSpecInet, AfSpecInet6, AfSpecUnspec, Icmp6Stats, Inet6CacheInfo,
    Inet6DevConf, Inet6IfaceFlags, Inet6Stats, InetDevConf, LinkAttribute,
    LinkHeader, LinkLayerType, LinkMessage, LinkMessageBuffer, LinkXdp, Map,
    Prop, State, Stats, Stats64, XdpAttached,
};
use crate::AddressFamily;

// tshark capture of command `ip -s link show ens3`
#[test]
fn test_parsing_link_statistics_on_kernel_4_18() {
    let raw = vec![
        0x00, 0x00, 0x01, 0x00, 0x02, 0x00, 0x00, 0x00, 0x43, 0x10, 0x01, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x09, 0x00, 0x03, 0x00, 0x65, 0x6e, 0x73, 0x33,
        0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x0d, 0x00, 0xe8, 0x03, 0x00, 0x00,
        0x05, 0x00, 0x10, 0x00, 0x06, 0x00, 0x00, 0x00, 0x05, 0x00, 0x11, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x04, 0x00, 0xdc, 0x05, 0x00, 0x00,
        0x08, 0x00, 0x32, 0x00, 0x44, 0x00, 0x00, 0x00, 0x08, 0x00, 0x33, 0x00,
        0xff, 0xff, 0x00, 0x00, 0x08, 0x00, 0x1b, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x08, 0x00, 0x1e, 0x00, 0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x1f, 0x00,
        0x01, 0x00, 0x00, 0x00, 0x08, 0x00, 0x28, 0x00, 0xff, 0xff, 0x00, 0x00,
        0x08, 0x00, 0x29, 0x00, 0x00, 0x00, 0x01, 0x00, 0x08, 0x00, 0x20, 0x00,
        0x01, 0x00, 0x00, 0x00, 0x05, 0x00, 0x21, 0x00, 0x01, 0x00, 0x00, 0x00,
        0x0d, 0x00, 0x06, 0x00, 0x66, 0x71, 0x5f, 0x63, 0x6f, 0x64, 0x65, 0x6c,
        0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x23, 0x00, 0x02, 0x00, 0x00, 0x00,
        0x05, 0x00, 0x27, 0x00, 0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x2f, 0x00,
        0x01, 0x00, 0x00, 0x00, 0x08, 0x00, 0x30, 0x00, 0x01, 0x00, 0x00, 0x00,
        0x24, 0x00, 0x0e, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x0a, 0x00, 0x01, 0x00, 0x52, 0x54, 0x00, 0xf3, 0xc5, 0xcf, 0x00, 0x00,
        0x0a, 0x00, 0x02, 0x00, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00,
        0xc4, 0x00, 0x17, 0x00, 0x8b, 0x6d, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0xf6, 0x0d, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x91, 0xb0, 0x6a, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x2d, 0xd6, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x06, 0x50, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x64, 0x00, 0x07, 0x00, 0x8b, 0x6d, 0x00, 0x00,
        0xf6, 0x0d, 0x00, 0x00, 0x91, 0xb0, 0x6a, 0x00, 0x2d, 0xd6, 0x04, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x06, 0x50, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0c, 0x00, 0x2b, 0x00,
        0x05, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0a, 0x00, 0x36, 0x00,
        0x52, 0x54, 0x00, 0xf3, 0xc5, 0xcf, 0x00, 0x00, 0xf8, 0x02, 0x1a, 0x00,
        0x88, 0x00, 0x02, 0x00, 0x84, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x10, 0x27, 0x00, 0x00, 0xe8, 0x03, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x6c, 0x02, 0x0a, 0x00, 0x08, 0x00, 0x01, 0x00,
        0x00, 0x00, 0x00, 0x80, 0x14, 0x00, 0x05, 0x00, 0xff, 0xff, 0x00, 0x00,
        0xce, 0x72, 0x3e, 0x00, 0x2c, 0x70, 0x00, 0x00, 0xe8, 0x03, 0x00, 0x00,
        0xd0, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00,
        0xdc, 0x05, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff,
        0xa0, 0x0f, 0x00, 0x00, 0xe8, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x80, 0x3a, 0x09, 0x00, 0x80, 0x51, 0x01, 0x00, 0x03, 0x00, 0x00, 0x00,
        0x58, 0x02, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
        0x60, 0xea, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x10, 0x27, 0x00, 0x00, 0xe8, 0x03, 0x00, 0x00,
        0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0xee, 0x36, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
        0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x2c, 0x01, 0x03, 0x00, 0x25, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0xf4, 0x05, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x56, 0xca, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x06, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x10, 0x02, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0xec, 0x05, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x06, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xf2, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x10, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x34, 0x00, 0x06, 0x00, 0x06, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x06, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x14, 0x00, 0x07, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x05, 0x00, 0x08, 0x00, 0x01, 0x00, 0x00, 0x00,
        0x10, 0x00, 0x34, 0x80, 0x0b, 0x00, 0x35, 0x00, 0x65, 0x6e, 0x70, 0x30,
        0x73, 0x33, 0x00, 0x00, 0x0c, 0x00, 0x38, 0x00, 0x76, 0x69, 0x72, 0x74,
        0x69, 0x6f, 0x30, 0x00, 0x0b, 0x00, 0x39, 0x00, 0x76, 0x69, 0x72, 0x74,
        0x69, 0x6f, 0x00, 0x00,
    ];

    let expected = LinkMessage {
        header: LinkHeader {
            interface_family: AddressFamily::Unspec,
            index: 2,
            link_layer_type: LinkLayerType::Ether,
            flags: LinkFlags::Broadcast
                | LinkFlags::LowerUp
                | LinkFlags::Multicast
                | LinkFlags::Running
                | LinkFlags::Up,
            change_mask: LinkFlags::empty(),
        },
        attributes: vec![
            LinkAttribute::IfName("ens3".into()),
            LinkAttribute::TxQueueLen(1000),
            LinkAttribute::OperState(State::Up),
            LinkAttribute::Mode(0),
            LinkAttribute::Mtu(1500),
            LinkAttribute::MinMtu(68),
            LinkAttribute::MaxMtu(65535),
            LinkAttribute::Group(0),
            LinkAttribute::Promiscuity(0),
            LinkAttribute::NumTxQueues(1),
            LinkAttribute::GsoMaxSegs(65535),
            LinkAttribute::GsoMaxSize(65536),
            LinkAttribute::NumRxQueues(1),
            LinkAttribute::Carrier(1),
            LinkAttribute::Qdisc("fq_codel".into()),
            LinkAttribute::CarrierChanges(2),
            LinkAttribute::ProtoDown(0),
            LinkAttribute::CarrierUpCount(1),
            LinkAttribute::CarrierDownCount(1),
            LinkAttribute::Map(Map {
                memory_start: 0,
                memory_end: 0,
                base_address: 0,
                irq: 0,
                dma: 0,
                port: 0,
            }),
            LinkAttribute::Address(vec![82, 84, 0, 243, 197, 207]),
            LinkAttribute::Broadcast(vec![255, 255, 255, 255, 255, 255]),
            LinkAttribute::Stats64(Stats64 {
                rx_packets: 28043,
                tx_packets: 3574,
                rx_bytes: 6992017,
                tx_bytes: 316973,
                rx_errors: 0,
                tx_errors: 0,
                rx_dropped: 20486,
                tx_dropped: 0,
                multicast: 0,
                collisions: 0,
                rx_length_errors: 0,
                rx_over_errors: 0,
                rx_crc_errors: 0,
                rx_frame_errors: 0,
                rx_fifo_errors: 0,
                rx_missed_errors: 0,
                tx_aborted_errors: 0,
                tx_carrier_errors: 0,
                tx_fifo_errors: 0,
                tx_heartbeat_errors: 0,
                tx_window_errors: 0,
                rx_compressed: 0,
                tx_compressed: 0,
                rx_nohandler: 0,
                rx_otherhost_dropped: 0,
            }),
            LinkAttribute::Stats(Stats {
                rx_packets: 28043,
                tx_packets: 3574,
                rx_bytes: 6992017,
                tx_bytes: 316973,
                rx_errors: 0,
                tx_errors: 0,
                rx_dropped: 20486,
                tx_dropped: 0,
                multicast: 0,
                collisions: 0,
                rx_length_errors: 0,
                rx_over_errors: 0,
                rx_crc_errors: 0,
                rx_frame_errors: 0,
                rx_fifo_errors: 0,
                rx_missed_errors: 0,
                tx_aborted_errors: 0,
                tx_carrier_errors: 0,
                tx_fifo_errors: 0,
                tx_heartbeat_errors: 0,
                tx_window_errors: 0,
                rx_compressed: 0,
                tx_compressed: 0,
                rx_nohandler: 0,
            }),
            LinkAttribute::Xdp(vec![LinkXdp::Attached(XdpAttached::None)]),
            LinkAttribute::PermAddress(vec![82, 84, 0, 243, 197, 207]),
            LinkAttribute::AfSpecUnspec(vec![
                AfSpecUnspec::Inet(vec![AfSpecInet::DevConf(InetDevConf {
                    forwarding: 0,
                    mc_forwarding: 0,
                    proxy_arp: 0,
                    accept_redirects: 0,
                    secure_redirects: 1,
                    send_redirects: 0,
                    shared_media: 1,
                    rp_filter: 0,
                    accept_source_route: 1,
                    bootp_relay: 0,
                    log_martians: 0,
                    tag: 0,
                    arpfilter: 0,
                    medium_id: 0,
                    noxfrm: 0,
                    nopolicy: 0,
                    force_igmp_version: 0,
                    arp_announce: 0,
                    arp_ignore: 0,
                    promote_secondaries: 0,
                    arp_accept: 0,
                    arp_notify: 0,
                    accept_local: 0,
                    src_vmark: 0,
                    proxy_arp_pvlan: 0,
                    route_localnet: 0,
                    igmpv2_unsolicited_report_interval: 10000,
                    igmpv3_unsolicited_report_interval: 1000,
                    ignore_routes_with_linkdown: 0,
                    drop_unicast_in_l2_multicast: 0,
                    drop_gratuitous_arp: 0,
                    bc_forwarding: 0,
                    arp_evict_nocarrier: 0,
                })]),
                AfSpecUnspec::Inet6(vec![
                    AfSpecInet6::Flags(Inet6IfaceFlags::Ready),
                    AfSpecInet6::CacheInfo(Inet6CacheInfo {
                        max_reasm_len: 65535,
                        tstamp: 4092622,
                        reachable_time: 28716,
                        retrans_time: 1000,
                    }),
                    AfSpecInet6::DevConf(Inet6DevConf {
                        forwarding: 0,
                        hoplimit: 64,
                        mtu6: 1500,
                        accept_ra: 0,
                        accept_redirects: 0,
                        autoconf: 1,
                        dad_transmits: 1,
                        rtr_solicits: -1,
                        rtr_solicit_interval: 4000,
                        rtr_solicit_delay: 1000,
                        use_tempaddr: 0,
                        temp_valid_lft: 604800,
                        temp_prefered_lft: 86400,
                        regen_max_retry: 3,
                        max_desync_factor: 600,
                        max_addresses: 16,
                        force_mld_version: 0,
                        accept_ra_defrtr: 1,
                        accept_ra_pinfo: 1,
                        accept_ra_rtr_pref: 1,
                        rtr_probe_interval: 60000,
                        accept_ra_rt_info_max_plen: 0,
                        proxy_ndp: 0,
                        optimistic_dad: 0,
                        accept_source_route: 0,
                        mc_forwarding: 0,
                        disable_ipv6: 0,
                        accept_dad: 1,
                        force_tllao: 0,
                        ndisc_notify: 0,
                        mldv1_unsolicited_report_interval: 10000,
                        mldv2_unsolicited_report_interval: 1000,
                        suppress_frag_ndisc: 1,
                        accept_ra_from_local: 0,
                        use_optimistic: 0,
                        accept_ra_mtu: 1,
                        stable_secret: 0,
                        use_oif_addrs_only: 0,
                        accept_ra_min_hop_limit: 1,
                        ignore_routes_with_linkdown: 0,
                        drop_unicast_in_l2_multicast: 0,
                        drop_unsolicited_na: 0,
                        keep_addr_on_down: 0,
                        rtr_solicit_max_interval: 3600000,
                        seg6_enabled: 0,
                        seg6_require_hmac: 0,
                        enhanced_dad: 1,
                        addr_gen_mode: 1,
                        disable_policy: 0,
                        accept_ra_rt_info_min_plen: 0,
                        ndisc_tclass: 0,
                        rpl_seg_enabled: 0,
                        ra_defrtr_metric: 0,
                        ioam6_enabled: 0,
                        ioam6_id: 0,
                        ioam6_id_wide: 0,
                        ndisc_evict_nocarrier: 0,
                        accept_untracked_na: 0,
                        accept_ra_min_lft: 0,
                    }),
                    AfSpecInet6::Stats(Inet6Stats {
                        num: 37,
                        in_pkts: 1524,
                        in_octets: 182870,
                        in_delivers: 0,
                        out_forw_datagrams: 0,
                        out_requests: 6,
                        out_octets: 528,
                        in_hdr_errors: 0,
                        in_too_big_errors: 0,
                        in_no_routes: 0,
                        in_addr_errors: 0,
                        in_unknown_protos: 0,
                        in_truncated_pkts: 0,
                        in_discards: 1516,
                        out_discards: 0,
                        out_no_routes: 0,
                        reasm_timeout: 0,
                        reasm_reqds: 0,
                        reasm_oks: 0,
                        reasm_fails: 0,
                        frag_oks: 0,
                        frag_fails: 0,
                        frag_creates: 0,
                        in_mcast_pkts: 2,
                        out_mcast_pkts: 6,
                        in_bcast_pkts: 0,
                        out_bcast_pkts: 0,
                        in_mcast_octets: 242,
                        out_mcast_octets: 528,
                        in_bcast_octets: 0,
                        out_bcast_octets: 0,
                        in_csum_errors: 0,
                        in_no_ect_pkts: 8,
                        in_ect1_pkts: 0,
                        in_ect0_pkts: 0,
                        in_ce_pkts: 0,
                        reasm_overlaps: 0,
                        out_pkts: 0,
                    }),
                    AfSpecInet6::Icmp6Stats(Icmp6Stats {
                        num: 6,
                        in_msgs: 0,
                        in_errors: 0,
                        out_msgs: 6,
                        out_errors: 0,
                        csum_errors: 0,
                        rate_limit_host: 0,
                    }),
                    AfSpecInet6::Token(std::net::Ipv6Addr::UNSPECIFIED),
                    AfSpecInet6::AddrGenMode(1),
                ]),
            ]),
            LinkAttribute::PropList(vec![Prop::AltIfName("enp0s3".into())]),
            LinkAttribute::Other(DefaultNla::new(
                56,
                vec![118, 105, 114, 116, 105, 111, 48, 0],
            )),
            LinkAttribute::Other(DefaultNla::new(
                57,
                vec![118, 105, 114, 116, 105, 111, 0],
            )),
        ],
    };

    assert_eq!(
        expected,
        LinkMessage::parse(&LinkMessageBuffer::new(&raw)).unwrap()
    );
    // We do not test emit here as old kernel has smaller buffer size of stats
}

// tshark capture of command `ip -s link show wlan0` on kernel 6.6
#[test]
fn test_parsing_link_statistics() {
    let raw = vec![
        0x00, 0x00, 0x01, 0x00, 0x03, 0x00, 0x00, 0x00, 0x43, 0x10, 0x01, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x0a, 0x00, 0x03, 0x00, 0x77, 0x6c, 0x61, 0x6e,
        0x30, 0x00, 0x00, 0x00, 0x08, 0x00, 0x0d, 0x00, 0xe8, 0x03, 0x00, 0x00,
        0x05, 0x00, 0x10, 0x00, 0x06, 0x00, 0x00, 0x00, 0x05, 0x00, 0x11, 0x00,
        0x01, 0x00, 0x00, 0x00, 0x08, 0x00, 0x04, 0x00, 0xdc, 0x05, 0x00, 0x00,
        0x08, 0x00, 0x32, 0x00, 0x00, 0x01, 0x00, 0x00, 0x08, 0x00, 0x33, 0x00,
        0x00, 0x09, 0x00, 0x00, 0x08, 0x00, 0x1b, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x08, 0x00, 0x1e, 0x00, 0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x3d, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x1f, 0x00, 0x01, 0x00, 0x00, 0x00,
        0x08, 0x00, 0x28, 0x00, 0xff, 0xff, 0x00, 0x00, 0x08, 0x00, 0x29, 0x00,
        0x00, 0x00, 0x01, 0x00, 0x08, 0x00, 0x3a, 0x00, 0x00, 0x00, 0x01, 0x00,
        0x08, 0x00, 0x3f, 0x00, 0x00, 0x00, 0x01, 0x00, 0x08, 0x00, 0x40, 0x00,
        0x00, 0x00, 0x01, 0x00, 0x08, 0x00, 0x3b, 0x00, 0x00, 0x00, 0x01, 0x00,
        0x08, 0x00, 0x3c, 0x00, 0xff, 0xff, 0x00, 0x00, 0x08, 0x00, 0x20, 0x00,
        0x01, 0x00, 0x00, 0x00, 0x05, 0x00, 0x21, 0x00, 0x01, 0x00, 0x00, 0x00,
        0x0c, 0x00, 0x06, 0x00, 0x6e, 0x6f, 0x71, 0x75, 0x65, 0x75, 0x65, 0x00,
        0x08, 0x00, 0x23, 0x00, 0x02, 0x00, 0x00, 0x00, 0x08, 0x00, 0x2f, 0x00,
        0x01, 0x00, 0x00, 0x00, 0x08, 0x00, 0x30, 0x00, 0x01, 0x00, 0x00, 0x00,
        0x05, 0x00, 0x27, 0x00, 0x00, 0x00, 0x00, 0x00, 0x24, 0x00, 0x0e, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0a, 0x00, 0x01, 0x00,
        0x10, 0xa5, 0x1d, 0x6f, 0xd2, 0x18, 0x00, 0x00, 0x0a, 0x00, 0x02, 0x00,
        0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0xcc, 0x00, 0x17, 0x00,
        0x50, 0xb4, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x89, 0x01, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x42, 0xc9, 0x91, 0x07, 0x00, 0x00, 0x00, 0x00,
        0x4d, 0x6e, 0x14, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x64, 0x00, 0x07, 0x00,
        0x50, 0xb4, 0x02, 0x00, 0x80, 0x89, 0x01, 0x00, 0x42, 0xc9, 0x91, 0x07,
        0x4d, 0x6e, 0x14, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x08, 0x00, 0x15, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0c, 0x00, 0x2b, 0x00,
        0x05, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0a, 0x00, 0x36, 0x00,
        0x10, 0xa5, 0x1d, 0x6f, 0xd2, 0x18, 0x00, 0x00, 0xa0, 0x01, 0x1a, 0x00,
        0x8c, 0x00, 0x02, 0x00, 0x88, 0x00, 0x01, 0x00, 0x01, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
        0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
        0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x10, 0x27, 0x00, 0x00, 0xe8, 0x03, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x10, 0x01, 0x0a, 0x00,
        0x08, 0x00, 0x01, 0x00, 0xf0, 0x00, 0x00, 0x80, 0x14, 0x00, 0x05, 0x00,
        0xff, 0xff, 0x00, 0x00, 0xea, 0x04, 0x00, 0x00, 0x1a, 0x5e, 0x00, 0x00,
        0xe8, 0x03, 0x00, 0x00, 0xf0, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x40, 0x00, 0x00, 0x00, 0xdc, 0x05, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
        0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
        0xff, 0xff, 0xff, 0xff, 0xa0, 0x0f, 0x00, 0x00, 0xe8, 0x03, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x80, 0x3a, 0x09, 0x00, 0x80, 0x51, 0x01, 0x00,
        0x03, 0x00, 0x00, 0x00, 0x58, 0x02, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
        0x01, 0x00, 0x00, 0x00, 0x60, 0xea, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x10, 0x27, 0x00, 0x00,
        0xe8, 0x03, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x80, 0xee, 0x36, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0x00, 0x00,
        0xff, 0xff, 0xff, 0xff, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x14, 0x00, 0x34, 0x80, 0x0e, 0x00, 0x35, 0x00,
        0x77, 0x6c, 0x70, 0x30, 0x73, 0x32, 0x30, 0x66, 0x33, 0x00, 0x00, 0x00,
        0x11, 0x00, 0x38, 0x00, 0x30, 0x30, 0x30, 0x30, 0x3a, 0x30, 0x30, 0x3a,
        0x31, 0x34, 0x2e, 0x33, 0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x39, 0x00,
        0x70, 0x63, 0x69, 0x00, 0x04, 0x00, 0x3e, 0x80,
    ];

    let expected = LinkMessage {
        header: LinkHeader {
            interface_family: AddressFamily::Unspec,
            index: 3,
            link_layer_type: LinkLayerType::Ether,
            flags: LinkFlags::Broadcast
                | LinkFlags::LowerUp
                | LinkFlags::Multicast
                | LinkFlags::Running
                | LinkFlags::Up,
            change_mask: LinkFlags::empty(),
        },
        attributes: vec![
            LinkAttribute::IfName("wlan0".into()),
            LinkAttribute::TxQueueLen(1000),
            LinkAttribute::OperState(State::Up),
            LinkAttribute::Mode(1),
            LinkAttribute::Mtu(1500),
            LinkAttribute::MinMtu(256),
            LinkAttribute::MaxMtu(2304),
            LinkAttribute::Group(0),
            LinkAttribute::Promiscuity(0),
            LinkAttribute::Other(DefaultNla::new(61, vec![0, 0, 0, 0])),
            LinkAttribute::NumTxQueues(1),
            LinkAttribute::GsoMaxSegs(65535),
            LinkAttribute::GsoMaxSize(65536),
            LinkAttribute::Other(DefaultNla::new(58, vec![0, 0, 1, 0])),
            LinkAttribute::Other(DefaultNla::new(63, vec![0, 0, 1, 0])),
            LinkAttribute::Other(DefaultNla::new(64, vec![0, 0, 1, 0])),
            LinkAttribute::Other(DefaultNla::new(59, vec![0, 0, 1, 0])),
            LinkAttribute::Other(DefaultNla::new(60, vec![255, 255, 0, 0])),
            LinkAttribute::NumRxQueues(1),
            LinkAttribute::Carrier(1),
            LinkAttribute::Qdisc("noqueue".into()),
            LinkAttribute::CarrierChanges(2),
            LinkAttribute::CarrierUpCount(1),
            LinkAttribute::CarrierDownCount(1),
            LinkAttribute::ProtoDown(0),
            LinkAttribute::Map(Map {
                memory_start: 0,
                memory_end: 0,
                base_address: 0,
                irq: 0,
                dma: 0,
                port: 0,
            }),
            LinkAttribute::Address(vec![16, 165, 29, 111, 210, 24]),
            LinkAttribute::Broadcast(vec![255, 255, 255, 255, 255, 255]),
            LinkAttribute::Stats64(Stats64 {
                rx_packets: 177232,
                tx_packets: 100736,
                rx_bytes: 126994754,
                tx_bytes: 18116173,
                rx_errors: 0,
                tx_errors: 0,
                rx_dropped: 0,
                tx_dropped: 0,
                multicast: 0,
                collisions: 0,
                rx_length_errors: 0,
                rx_over_errors: 0,
                rx_crc_errors: 0,
                rx_frame_errors: 0,
                rx_fifo_errors: 0,
                rx_missed_errors: 0,
                tx_aborted_errors: 0,
                tx_carrier_errors: 0,
                tx_fifo_errors: 0,
                tx_heartbeat_errors: 0,
                tx_window_errors: 0,
                rx_compressed: 0,
                tx_compressed: 0,
                rx_nohandler: 0,
                rx_otherhost_dropped: 0,
            }),
            LinkAttribute::Stats(Stats {
                rx_packets: 177232,
                tx_packets: 100736,
                rx_bytes: 126994754,
                tx_bytes: 18116173,
                rx_errors: 0,
                tx_errors: 0,
                rx_dropped: 0,
                tx_dropped: 0,
                multicast: 0,
                collisions: 0,
                rx_length_errors: 0,
                rx_over_errors: 0,
                rx_crc_errors: 0,
                rx_frame_errors: 0,
                rx_fifo_errors: 0,
                rx_missed_errors: 0,
                tx_aborted_errors: 0,
                tx_carrier_errors: 0,
                tx_fifo_errors: 0,
                tx_heartbeat_errors: 0,
                tx_window_errors: 0,
                rx_compressed: 0,
                tx_compressed: 0,
                rx_nohandler: 0,
            }),
            LinkAttribute::NumVf(0),
            LinkAttribute::Xdp(vec![LinkXdp::Attached(XdpAttached::None)]),
            LinkAttribute::PermAddress(vec![16, 165, 29, 111, 210, 24]),
            LinkAttribute::AfSpecUnspec(vec![
                AfSpecUnspec::Inet(vec![AfSpecInet::DevConf(InetDevConf {
                    forwarding: 1,
                    mc_forwarding: 0,
                    proxy_arp: 0,
                    accept_redirects: 1,
                    secure_redirects: 1,
                    send_redirects: 1,
                    shared_media: 1,
                    rp_filter: 2,
                    accept_source_route: 0,
                    bootp_relay: 0,
                    log_martians: 0,
                    tag: 0,
                    arpfilter: 0,
                    medium_id: 0,
                    noxfrm: 0,
                    nopolicy: 0,
                    force_igmp_version: 0,
                    arp_announce: 0,
                    arp_ignore: 0,
                    promote_secondaries: 1,
                    arp_accept: 0,
                    arp_notify: 0,
                    accept_local: 0,
                    src_vmark: 0,
                    proxy_arp_pvlan: 0,
                    route_localnet: 0,
                    igmpv2_unsolicited_report_interval: 10000,
                    igmpv3_unsolicited_report_interval: 1000,
                    ignore_routes_with_linkdown: 0,
                    drop_unicast_in_l2_multicast: 0,
                    drop_gratuitous_arp: 0,
                    bc_forwarding: 0,
                    arp_evict_nocarrier: 1,
                })]),
                AfSpecUnspec::Inet6(vec![
                    AfSpecInet6::Flags(
                        Inet6IfaceFlags::Otherconf
                            | Inet6IfaceFlags::RaManaged
                            | Inet6IfaceFlags::RaRcvd
                            | Inet6IfaceFlags::RsSent
                            | Inet6IfaceFlags::Ready,
                    ),
                    AfSpecInet6::CacheInfo(Inet6CacheInfo {
                        max_reasm_len: 65535,
                        tstamp: 1258,
                        reachable_time: 24090,
                        retrans_time: 1000,
                    }),
                    AfSpecInet6::DevConf(Inet6DevConf {
                        forwarding: 0,
                        hoplimit: 64,
                        mtu6: 1500,
                        accept_ra: 1,
                        accept_redirects: 1,
                        autoconf: 1,
                        dad_transmits: 1,
                        rtr_solicits: -1,
                        rtr_solicit_interval: 4000,
                        rtr_solicit_delay: 1000,
                        use_tempaddr: 0,
                        temp_valid_lft: 604800,
                        temp_prefered_lft: 86400,
                        regen_max_retry: 3,
                        max_desync_factor: 600,
                        max_addresses: 16,
                        force_mld_version: 0,
                        accept_ra_defrtr: 1,
                        accept_ra_pinfo: 1,
                        accept_ra_rtr_pref: 1,
                        rtr_probe_interval: 60000,
                        accept_ra_rt_info_max_plen: 0,
                        proxy_ndp: 0,
                        optimistic_dad: 0,
                        accept_source_route: 0,
                        mc_forwarding: 0,
                        disable_ipv6: 0,
                        accept_dad: 1,
                        force_tllao: 0,
                        ndisc_notify: 0,
                        mldv1_unsolicited_report_interval: 10000,
                        mldv2_unsolicited_report_interval: 1000,
                        suppress_frag_ndisc: 1,
                        accept_ra_from_local: 0,
                        use_optimistic: 0,
                        accept_ra_mtu: 1,
                        stable_secret: 0,
                        use_oif_addrs_only: 0,
                        accept_ra_min_hop_limit: 1,
                        ignore_routes_with_linkdown: 0,
                        drop_unicast_in_l2_multicast: 0,
                        drop_unsolicited_na: 0,
                        keep_addr_on_down: 0,
                        rtr_solicit_max_interval: 3600000,
                        seg6_enabled: 0,
                        seg6_require_hmac: 0,
                        enhanced_dad: 1,
                        addr_gen_mode: 0,
                        disable_policy: 0,
                        accept_ra_rt_info_min_plen: 0,
                        ndisc_tclass: 0,
                        rpl_seg_enabled: 0,
                        ra_defrtr_metric: 1024,
                        ioam6_enabled: 0,
                        ioam6_id: 65535,
                        ioam6_id_wide: -1,
                        ndisc_evict_nocarrier: 1,
                        accept_untracked_na: 0,
                        accept_ra_min_lft: 0,
                    }),
                ]),
            ]),
            LinkAttribute::PropList(vec![Prop::AltIfName("wlp0s20f3".into())]),
            LinkAttribute::Other(DefaultNla::new(
                56,
                vec![48, 48, 48, 48, 58, 48, 48, 58, 49, 52, 46, 51, 0],
            )),
            LinkAttribute::Other(DefaultNla::new(57, vec![112, 99, 105, 0])),
            LinkAttribute::Other(DefaultNla::new(
                32830, // NLA_F_NESTED|IFLA_DEVLINK_PORT
                vec![],
            )),
        ],
    };

    assert_eq!(
        expected,
        LinkMessage::parse(&LinkMessageBuffer::new(&raw)).unwrap()
    );

    let mut buf = vec![0; expected.buffer_len()];

    expected.emit(&mut buf);

    assert_eq!(buf, raw);
    let mut buf = vec![0; expected.buffer_len()];

    expected.emit(&mut buf);

    assert_eq!(buf, raw);
}
