// Copyright (c) 2019, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

import 'dart:io' show Directory, Platform;

import 'package:_fe_analyzer_shared/src/testing/id.dart' show ActualData, Id;
import 'package:_fe_analyzer_shared/src/testing/id_testing.dart';
import 'package:front_end/src/testing/id_testing_helper.dart';
import 'package:kernel/ast.dart';

Future<void> main(List<String> args) async {
  Directory dataDir = new Directory.fromUri(
    Platform.script.resolve(
      '../../../_fe_analyzer_shared/test/flow_analysis/nullability/data',
    ),
  );
  await runTests<String>(
    dataDir,
    args: args,
    createUriForFileName: createUriForFileName,
    onFailure: onFailure,
    runTest: runTestFor(const NullabilityDataComputer(), [defaultCfeConfig]),
  );
}

class NullabilityDataComputer extends CfeDataComputer<String> {
  const NullabilityDataComputer();

  @override
  DataInterpreter<String> get dataValidator => const StringDataInterpreter();

  /// Function that computes a data mapping for [member].
  ///
  /// Fills [actualMap] with the data.
  @override
  void computeMemberData(
    CfeTestResultData testResultData,
    Member member,
    Map<Id, ActualData<String>> actualMap, {
    bool? verbose,
  }) {
    member.accept(
      new NullabilityDataExtractor(testResultData.compilerResult, actualMap),
    );
  }
}

class NullabilityDataExtractor extends CfeDataExtractor<String> {
  NullabilityDataExtractor(
    InternalCompilerResult compilerResult,
    Map<Id, ActualData<String>> actualMap,
  ) : super(compilerResult, actualMap);

  @override
  String? computeNodeValue(Id id, TreeNode node) {
    if (node is VariableGet && node.promotedType != null) {
      if (node.variable.type.nullability != Nullability.nonNullable &&
          node.promotedType!.nullability == Nullability.nonNullable) {
        return 'nonNullable';
      }
    }
    return null;
  }
}
