/**
 * Completion data for dae language
 *
 * Contains all completion items for sections, parameters, functions, and snippets
 */

/**
 * Completion item definition for dae language
 */
export interface DaeCompletionItem {
  label: string
  kind: 'keyword' | 'function' | 'constant' | 'type' | 'variable' | 'snippet' | 'property'
  detail?: string
  documentation?: string
  insertText: string
  isSnippet?: boolean
}

/**
 * Completion items for dae language autocomplete
 */
export const DAE_COMPLETION_ITEMS: DaeCompletionItem[] = [
  // Section blocks
  {
    label: 'global',
    kind: 'keyword',
    detail: 'Global Section',
    documentation: 'Global configuration section',
    insertText: 'global {\n\t${0}\n}',
    isSnippet: true,
  },
  {
    label: 'subscription',
    kind: 'keyword',
    detail: 'Subscription Section',
    documentation: 'Define proxy subscriptions',
    insertText: 'subscription {\n\t${0}\n}',
    isSnippet: true,
  },
  {
    label: 'node',
    kind: 'keyword',
    detail: 'Node Section',
    documentation: 'Define proxy nodes',
    insertText: 'node {\n\t${0}\n}',
    isSnippet: true,
  },
  {
    label: 'dns',
    kind: 'keyword',
    detail: 'DNS Section',
    documentation: 'DNS configuration section',
    insertText: 'dns {\n\t${0}\n}',
    isSnippet: true,
  },
  {
    label: 'group',
    kind: 'keyword',
    detail: 'Group Section',
    documentation: 'Proxy group configuration',
    insertText: 'group {\n\t${0}\n}',
    isSnippet: true,
  },
  {
    label: 'routing',
    kind: 'keyword',
    detail: 'Routing Section',
    documentation: 'Routing rules configuration',
    insertText: 'routing {\n\t${0}\n}',
    isSnippet: true,
  },

  // Global section options
  {
    label: 'tproxy_port',
    kind: 'property',
    detail: 'TProxy Port',
    documentation: 'Transparent proxy listening port',
    insertText: 'tproxy_port: ${1:12345}',
    isSnippet: true,
  },
  {
    label: 'log_level',
    kind: 'property',
    detail: 'Log Level',
    documentation: 'Logging level for dae output.\n\n**Values:** `error`, `warn`, `info`, `debug`, `trace`',
    insertText: 'log_level: ${1|info,warn,error,debug,trace|}',
    isSnippet: true,
  },
  {
    label: 'tcp_check_url',
    kind: 'property',
    detail: 'TCP Check URL',
    documentation: 'URL for TCP health check',
    insertText: "tcp_check_url: '${1:http://cp.cloudflare.com}'",
    isSnippet: true,
  },
  {
    label: 'udp_check_dns',
    kind: 'property',
    detail: 'UDP Check DNS',
    documentation: 'DNS server for UDP health check',
    insertText: "udp_check_dns: '${1:dns.google:53}'",
    isSnippet: true,
  },
  {
    label: 'check_interval',
    kind: 'property',
    detail: 'Check Interval',
    documentation:
      'Interval between health checks.\n\n**Go duration format:** ns, us/µs, ms, s, m, h\n\nExamples: `30s`, `1m`, `500ms`',
    insertText: 'check_interval: ${1|30s,1m,15s,45s,2m|}',
    isSnippet: true,
  },
  {
    label: 'check_tolerance',
    kind: 'property',
    detail: 'Check Tolerance',
    documentation:
      'Group will switch node only when `new_latency <= old_latency - tolerance`.\n\n**Go duration format:** ns, us/µs, ms, s, m, h\n\nExamples: `50ms`, `100ms`, `200ms`',
    insertText: 'check_tolerance: ${1|50ms,100ms,150ms,200ms|}',
    isSnippet: true,
  },
  {
    label: 'dial_mode',
    kind: 'property',
    detail: 'Dial Mode',
    documentation:
      'Dialing mode for proxy connections.\n\n**Values:**\n- `ip`: Dial using IP directly\n- `domain`: Dial using sniffed domain\n- `domain+`: domain mode without domain reality check\n- `domain++`: domain+ with re-routing',
    insertText: 'dial_mode: ${1|domain,ip,domain\\+,domain\\+\\+|}',
    isSnippet: true,
  },
  {
    label: 'wan_interface',
    kind: 'property',
    detail: 'WAN Interface',
    documentation: 'WAN interface name',
    insertText: 'wan_interface: ${1:auto}',
    isSnippet: true,
  },
  {
    label: 'lan_interface',
    kind: 'property',
    detail: 'LAN Interface',
    documentation: 'LAN interface names',
    insertText: 'lan_interface: ${1:docker0}',
    isSnippet: true,
  },
  {
    label: 'auto_config_kernel_parameter',
    kind: 'property',
    detail: 'Auto Config Kernel',
    documentation: 'Automatically configure Linux kernel parameters like ip_forward and send_redirects.',
    insertText: 'auto_config_kernel_parameter: ${1|true,false|}',
    isSnippet: true,
  },
  {
    label: 'auto_config_firewall_rule',
    kind: 'property',
    detail: 'Auto Config Firewall',
    documentation: 'Automatically configure firewall rules.',
    insertText: 'auto_config_firewall_rule: ${1|true,false|}',
    isSnippet: true,
  },
  {
    label: 'sniffing_timeout',
    kind: 'property',
    detail: 'Sniffing Timeout',
    documentation:
      'Timeout for waiting first data for sniffing. It is always 0 if dial_mode is ip.\n\n**Go duration format:** ns, us/µs, ms, s, m, h\n\nExamples: `100ms`, `200ms`, `300ms`',
    insertText: 'sniffing_timeout: ${1|100ms,200ms,300ms,500ms|}',
    isSnippet: true,
  },
  {
    label: 'tls_implementation',
    kind: 'property',
    detail: 'TLS Implementation',
    documentation: 'TLS implementation to use',
    insertText: 'tls_implementation: ${1|tls,utls|}',
    isSnippet: true,
  },
  {
    label: 'utls_imitate',
    kind: 'property',
    detail: 'uTLS Imitate',
    documentation:
      'Browser fingerprint to imitate when tls_implementation is utls. See: https://github.com/daeuniverse/dae/blob/main/component/outbound/transport/tls/utls.go',
    insertText:
      'utls_imitate: ${1|chrome_auto,firefox_auto,safari_auto,edge_auto,ios_auto,randomized,chrome_58,chrome_62,chrome_70,chrome_72,chrome_83,chrome_87,chrome_96,chrome_100,chrome_102,firefox_55,firefox_56,firefox_63,firefox_65,firefox_99,firefox_102,firefox_105,safari_16_0,edge_85,edge_106,ios_11_1,ios_12_1,ios_13,ios_14,android_11_okhttp,360_auto,360_7_5,360_11_0,qq_auto,qq_11_1,randomizedalpn,randomizednoalpn|}',
    isSnippet: true,
  },
  {
    label: 'tproxy_port_protect',
    kind: 'property',
    detail: 'TProxy Port Protect',
    documentation: 'Protect tproxy port from unsolicited traffic',
    insertText: 'tproxy_port_protect: ${1|true,false|}',
    isSnippet: true,
  },
  {
    label: 'pprof_port',
    kind: 'property',
    detail: 'PProf Port',
    documentation: 'Set non-zero value to enable pprof for debugging',
    insertText: 'pprof_port: ${1:0}',
    isSnippet: true,
  },
  {
    label: 'so_mark_from_dae',
    kind: 'property',
    detail: 'SO_MARK from dae',
    documentation: 'SO_MARK for traffic sent from dae to avoid traffic loop',
    insertText: 'so_mark_from_dae: ${1:0}',
    isSnippet: true,
  },
  {
    label: 'disable_waiting_network',
    kind: 'property',
    detail: 'Disable Waiting Network',
    documentation: 'Disable waiting for network before pulling subscriptions',
    insertText: 'disable_waiting_network: ${1|true,false|}',
    isSnippet: true,
  },
  {
    label: 'enable_local_tcp_fast_redirect',
    kind: 'property',
    detail: 'Enable Local TCP Fast Redirect',
    documentation: 'Enable fast redirect for local TCP connections (experimental)',
    insertText: 'enable_local_tcp_fast_redirect: ${1|true,false|}',
    isSnippet: true,
  },
  {
    label: 'tcp_check_http_method',
    kind: 'property',
    detail: 'TCP Check HTTP Method',
    documentation: 'HTTP method for tcp_check_url (HEAD bypasses traffic accounting)',
    insertText: 'tcp_check_http_method: ${1|HEAD,GET|}',
    isSnippet: true,
  },
  {
    label: 'allow_insecure',
    kind: 'property',
    detail: 'Allow Insecure TLS',
    documentation: 'Allow insecure TLS certificates (not recommended)',
    insertText: 'allow_insecure: ${1|true,false|}',
    isSnippet: true,
  },
  {
    label: 'tls_fragment',
    kind: 'property',
    detail: 'TLS Fragment',
    documentation: 'Enable TLS fragmentation to bypass SNI blocking',
    insertText: 'tls_fragment: ${1|true,false|}',
    isSnippet: true,
  },
  {
    label: 'tls_fragment_length',
    kind: 'property',
    detail: 'TLS Fragment Length',
    documentation: 'TLS fragment packet length range in bytes (e.g., 50-100)',
    insertText: "tls_fragment_length: '${1:50-100}'",
    isSnippet: true,
  },
  {
    label: 'tls_fragment_interval',
    kind: 'property',
    detail: 'TLS Fragment Interval',
    documentation: 'TLS fragment packet interval in milliseconds (e.g., 10-20)',
    insertText: "tls_fragment_interval: '${1:10-20}'",
    isSnippet: true,
  },
  {
    label: 'mptcp',
    kind: 'property',
    detail: 'Multipath TCP',
    documentation: 'Enable MPTCP for load balance and failover',
    insertText: 'mptcp: ${1|true,false|}',
    isSnippet: true,
  },
  {
    label: 'bandwidth_max_tx',
    kind: 'property',
    detail: 'Max Upload Bandwidth',
    documentation: 'Maximum upload bandwidth for Hysteria2 etc. (e.g., 200 mbps)',
    insertText: "bandwidth_max_tx: '${1:200 mbps}'",
    isSnippet: true,
  },
  {
    label: 'bandwidth_max_rx',
    kind: 'property',
    detail: 'Max Download Bandwidth',
    documentation: 'Maximum download bandwidth for Hysteria2 etc. (e.g., 1 gbps)',
    insertText: "bandwidth_max_rx: '${1:1 gbps}'",
    isSnippet: true,
  },
  {
    label: 'fallback_resolver',
    kind: 'property',
    detail: 'Fallback DNS Resolver',
    documentation: 'Fallback DNS when system DNS fails (format: ip:port)',
    insertText: "fallback_resolver: '${1:8.8.8.8:53}'",
    isSnippet: true,
  },

  // DNS section parameters
  {
    label: 'ipversion_prefer',
    kind: 'property',
    detail: 'IP Version Prefer',
    documentation:
      'Prefer IPv4 (4) or IPv6 (6) responses. If set to 4 and the domain has both A and AAAA records, dae will only respond to type A queries.',
    insertText: 'ipversion_prefer: ${1|4,6|}',
    isSnippet: true,
  },
  {
    label: 'bind',
    kind: 'property',
    detail: 'DNS Bind Address',
    documentation: 'Local address for DNS queries (e.g., 127.0.0.1:5353)',
    insertText: "bind: '${1|tcp+udp,tcp,udp|}://${2:127.0.0.1}:${3:5353}'",
    isSnippet: true,
  },

  // Group section options
  {
    label: 'filter',
    kind: 'property',
    detail: 'Filter',
    documentation: 'Filter nodes in group. Use subtag() or name() to filter.',
    insertText: 'filter: ${1:subtag(my_sub)}',
    isSnippet: true,
  },
  {
    label: 'filter (with latency)',
    kind: 'property',
    detail: 'Filter with Latency Offset',
    documentation: 'Filter nodes with latency offset for failover scenarios. Negative values make nodes preferred.',
    insertText: "filter: name(${1:keyword}: '${2:pattern}') [add_latency: ${3:-500ms}]",
    isSnippet: true,
  },
  {
    label: 'policy',
    kind: 'property',
    detail: 'Policy',
    documentation: 'Load balancing policy',
    insertText: 'policy: ${1|min_moving_avg,min,min_avg10,random,fixed(0)|}',
    isSnippet: true,
  },

  // Policy functions
  {
    label: 'min_moving_avg',
    kind: 'function',
    detail: 'Min Moving Average',
    documentation: 'Select node with minimum moving average latency',
    insertText: 'min_moving_avg',
  },
  {
    label: 'min',
    kind: 'function',
    detail: 'Min Latency',
    documentation: 'Select node with minimum latency',
    insertText: 'min',
  },
  {
    label: 'min_avg10',
    kind: 'function',
    detail: 'Min Average 10',
    documentation: 'Select node with minimum average of last 10 checks',
    insertText: 'min_avg10',
  },
  {
    label: 'random',
    kind: 'function',
    detail: 'Random',
    documentation: 'Randomly select a node',
    insertText: 'random',
  },
  {
    label: 'fixed',
    kind: 'function',
    detail: 'Fixed Index',
    documentation: 'Select node at fixed index',
    insertText: 'fixed(${1:0})',
    isSnippet: true,
  },

  // Filter functions
  {
    label: 'subtag',
    kind: 'function',
    detail: 'Subscription Tag',
    documentation: 'Filter nodes by subscription tag',
    insertText: 'subtag(${1:tag})',
    isSnippet: true,
  },
  {
    label: 'name',
    kind: 'function',
    detail: 'Name Filter',
    documentation: 'Filter nodes by name pattern',
    insertText: "name(${1|keyword,regex|}: '${2:pattern}')",
    isSnippet: true,
  },

  // Rule functions (matching conditions)
  {
    label: 'dip',
    kind: 'function',
    detail: 'Destination IP',
    documentation: 'Match destination IP address',
    insertText: 'dip(${1:ip})',
    isSnippet: true,
  },
  {
    label: 'dport',
    kind: 'function',
    detail: 'Destination Port',
    documentation: 'Match destination port',
    insertText: 'dport(${1:port})',
    isSnippet: true,
  },
  {
    label: 'sip',
    kind: 'function',
    detail: 'Source IP',
    documentation: 'Match source IP address',
    insertText: 'sip(${1:ip})',
    isSnippet: true,
  },
  {
    label: 'sport',
    kind: 'function',
    detail: 'Source Port',
    documentation: 'Match source port',
    insertText: 'sport(${1:port})',
    isSnippet: true,
  },
  {
    label: 'ipversion',
    kind: 'function',
    detail: 'IP Version',
    documentation: 'Match IP version (4 or 6)',
    insertText: 'ipversion(${1|4,6|})',
    isSnippet: true,
  },
  {
    label: 'l4proto',
    kind: 'function',
    detail: 'L4 Protocol',
    documentation: 'Match Layer 4 protocol (tcp/udp)',
    insertText: 'l4proto(${1|tcp,udp|})',
    isSnippet: true,
  },
  {
    label: 'mac',
    kind: 'function',
    detail: 'MAC Address',
    documentation: 'Match MAC address',
    insertText: 'mac(${1:mac})',
    isSnippet: true,
  },
  {
    label: 'pname',
    kind: 'function',
    detail: 'Process Name',
    documentation: 'Match process name',
    insertText: 'pname(${1:name})',
    isSnippet: true,
  },
  {
    label: 'qname',
    kind: 'function',
    detail: 'Query Name',
    documentation: 'Match DNS query name',
    insertText: 'qname(${1:domain})',
    isSnippet: true,
  },
  {
    label: 'domain',
    kind: 'function',
    detail: 'Domain',
    documentation: 'Match domain (suffix match)',
    insertText: 'domain(${1:domain})',
    isSnippet: true,
  },
  {
    label: 'ip',
    kind: 'function',
    detail: 'IP',
    documentation: 'Match IP address',
    insertText: 'ip(${1:ip})',
    isSnippet: true,
  },
  {
    label: 'source',
    kind: 'function',
    detail: 'Source',
    documentation: 'Match source address',
    insertText: 'source(${1:source})',
    isSnippet: true,
  },
  {
    label: 'port',
    kind: 'function',
    detail: 'Port',
    documentation: 'Match port',
    insertText: 'port(${1:port})',
    isSnippet: true,
  },
  {
    label: 'sourcePort',
    kind: 'function',
    detail: 'Source Port',
    documentation: 'Match source port',
    insertText: 'sourcePort(${1:port})',
    isSnippet: true,
  },
  {
    label: 'inboundTag',
    kind: 'function',
    detail: 'Inbound Tag',
    documentation: 'Match inbound tag',
    insertText: 'inboundTag(${1:tag})',
    isSnippet: true,
  },
  {
    label: 'network',
    kind: 'function',
    detail: 'Network',
    documentation: 'Match network type',
    insertText: 'network(${1:network})',
    isSnippet: true,
  },
  {
    label: 'protocol',
    kind: 'function',
    detail: 'Protocol',
    documentation: 'Match protocol',
    insertText: 'protocol(${1:protocol})',
    isSnippet: true,
  },
  {
    label: 'user',
    kind: 'function',
    detail: 'User',
    documentation: 'Match user',
    insertText: 'user(${1:user})',
    isSnippet: true,
  },

  // Domain matching types
  {
    label: 'full',
    kind: 'type',
    detail: 'Full Match',
    documentation: 'Full domain match',
    insertText: 'full:${1:domain}',
    isSnippet: true,
  },
  {
    label: 'contains',
    kind: 'type',
    detail: 'Contains',
    documentation: 'Domain contains keyword',
    insertText: 'contains:${1:keyword}',
    isSnippet: true,
  },
  {
    label: 'regexp',
    kind: 'type',
    detail: 'Regular Expression',
    documentation: 'Match using regular expression',
    insertText: 'regexp:${1:pattern}',
    isSnippet: true,
  },
  {
    label: 'geosite',
    kind: 'type',
    detail: 'Geosite',
    documentation: 'Match using geosite database',
    insertText: 'geosite:${1:tag}',
    isSnippet: true,
  },
  {
    label: 'geoip',
    kind: 'type',
    detail: 'Geoip',
    documentation: 'Match using geoip database',
    insertText: 'geoip:${1:tag}',
    isSnippet: true,
  },
  {
    label: 'ext',
    kind: 'type',
    detail: 'External File',
    documentation: 'Match using external file',
    insertText: 'ext:"${1:file.dat}:${2:tag}"',
    isSnippet: true,
  },

  // Built-in outbounds
  {
    label: 'proxy',
    kind: 'constant',
    detail: 'Proxy',
    documentation: 'Use proxy outbound',
    insertText: 'proxy',
  },
  {
    label: 'block',
    kind: 'constant',
    detail: 'Block',
    documentation: 'Block the connection',
    insertText: 'block',
  },
  {
    label: 'direct',
    kind: 'constant',
    detail: 'Direct',
    documentation: 'Direct connection',
    insertText: 'direct',
  },
  {
    label: 'must_direct',
    kind: 'constant',
    detail: 'Must Direct',
    documentation: 'Force direct connection',
    insertText: 'must_direct',
  },
  {
    label: 'must_proxy',
    kind: 'constant',
    detail: 'Must Proxy',
    documentation: 'Force proxy connection',
    insertText: 'must_proxy',
  },

  // Declaration keywords
  {
    label: 'routing',
    kind: 'keyword',
    detail: 'Routing Block',
    documentation: 'Define routing rules block',
    insertText: 'routing {\n\t${0}\n}',
    isSnippet: true,
  },

  // dae-specific keywords
  {
    label: 'fallback',
    kind: 'keyword',
    detail: 'Fallback',
    documentation: 'Fallback outbound',
    insertText: 'fallback: ${1:outbound}',
    isSnippet: true,
  },
  {
    label: 'must_rules',
    kind: 'keyword',
    detail: 'Must Rules',
    documentation: 'Mandatory rules that cannot be overridden',
    insertText: 'must_rules',
  },
  {
    label: 'request',
    kind: 'keyword',
    detail: 'Request',
    documentation: 'Request routing rules',
    insertText: 'request',
  },
  {
    label: 'response',
    kind: 'keyword',
    detail: 'Response',
    documentation: 'Response routing rules',
    insertText: 'response',
  },
  {
    label: 'upstream',
    kind: 'keyword',
    detail: 'Upstream',
    documentation: 'Upstream server',
    insertText: 'upstream',
  },
  {
    label: 'tcp',
    kind: 'keyword',
    detail: 'TCP',
    documentation: 'TCP protocol',
    insertText: 'tcp',
  },
  {
    label: 'udp',
    kind: 'keyword',
    detail: 'UDP',
    documentation: 'UDP protocol',
    insertText: 'udp',
  },

  // Snippets for common patterns
  {
    label: 'rule-domain',
    kind: 'snippet',
    detail: 'Domain Rule',
    documentation: 'Create a domain matching rule',
    insertText: 'domain(${1:domain}) -> ${2|proxy,direct,block|}',
    isSnippet: true,
  },
  {
    label: 'rule-ip',
    kind: 'snippet',
    detail: 'IP Rule',
    documentation: 'Create an IP matching rule',
    insertText: 'dip(${1:ip}) -> ${2|proxy,direct,block|}',
    isSnippet: true,
  },
  {
    label: 'rule-geosite',
    kind: 'snippet',
    detail: 'Geosite Rule',
    documentation: 'Create a geosite matching rule',
    insertText: 'domain(geosite:${1:cn}) -> ${2|direct,proxy,block|}',
    isSnippet: true,
  },
  {
    label: 'rule-geoip',
    kind: 'snippet',
    detail: 'Geoip Rule',
    documentation: 'Create a geoip matching rule',
    insertText: 'dip(geoip:${1:cn}) -> ${2|direct,proxy,block|}',
    isSnippet: true,
  },
  {
    label: 'rule-process',
    kind: 'snippet',
    detail: 'Process Rule',
    documentation: 'Create a process matching rule',
    insertText: 'pname(${1:process}) -> ${2|proxy,direct,block|}',
    isSnippet: true,
  },
  {
    label: 'rule-port',
    kind: 'snippet',
    detail: 'Port Rule',
    documentation: 'Create a port matching rule',
    insertText: 'dport(${1:port}) -> ${2|proxy,direct,block|}',
    isSnippet: true,
  },
]

/**
 * Property value options for context-aware completion
 */
export const PROPERTY_VALUES: Record<string, { values: string[]; type: 'boolean' | 'enum' | 'duration' }> = {
  // Boolean properties
  auto_config_kernel_parameter: { values: ['true', 'false'], type: 'boolean' },
  auto_config_firewall_rule: { values: ['true', 'false'], type: 'boolean' },
  tproxy_port_protect: { values: ['true', 'false'], type: 'boolean' },
  disable_waiting_network: { values: ['true', 'false'], type: 'boolean' },
  enable_local_tcp_fast_redirect: { values: ['true', 'false'], type: 'boolean' },
  allow_insecure: { values: ['true', 'false'], type: 'boolean' },
  tls_fragment: { values: ['true', 'false'], type: 'boolean' },
  mptcp: { values: ['true', 'false'], type: 'boolean' },

  // Enum properties
  log_level: { values: ['info', 'warn', 'error', 'debug', 'trace'], type: 'enum' },
  dial_mode: { values: ['domain', 'ip', 'domain+', 'domain++'], type: 'enum' },
  tls_implementation: { values: ['tls', 'utls'], type: 'enum' },
  tcp_check_http_method: { values: ['HEAD', 'GET'], type: 'enum' },
  ipversion_prefer: { values: ['4', '6'], type: 'enum' },
  utls_imitate: {
    values: [
      'chrome_auto',
      'firefox_auto',
      'safari_auto',
      'edge_auto',
      'ios_auto',
      'randomized',
      'chrome_58',
      'chrome_62',
      'chrome_70',
      'chrome_72',
      'chrome_83',
      'chrome_87',
      'chrome_96',
      'chrome_100',
      'chrome_102',
      'firefox_55',
      'firefox_56',
      'firefox_63',
      'firefox_65',
      'firefox_99',
      'firefox_102',
      'firefox_105',
      'safari_16_0',
      'edge_85',
      'edge_106',
      'ios_11_1',
      'ios_12_1',
      'ios_13',
      'ios_14',
      'android_11_okhttp',
      '360_auto',
      '360_7_5',
      '360_11_0',
      'qq_auto',
      'qq_11_1',
      'randomizedalpn',
      'randomizednoalpn',
    ],
    type: 'enum',
  },

  // Duration properties
  check_interval: { values: ['15s', '30s', '45s', '1m', '2m'], type: 'duration' },
  check_tolerance: { values: ['50ms', '100ms', '150ms', '200ms'], type: 'duration' },
  sniffing_timeout: { values: ['100ms', '200ms', '300ms', '500ms'], type: 'duration' },
}

/**
 * DNS routing specific completion items
 * These are shown when inside a DNS routing context (dns section or document with upstream)
 */
export const DNS_ROUTING_COMPLETIONS: DaeCompletionItem[] = [
  {
    label: 'qname',
    kind: 'function',
    detail: 'Query Name',
    documentation: 'Match DNS query name. Used in DNS routing section.\n\nExample: `qname(geosite:cn) -> alidns`',
    insertText: 'qname(${1:domain})',
    isSnippet: true,
  },
  {
    label: 'qtype',
    kind: 'function',
    detail: 'Query Type',
    documentation: 'Match DNS query type.\n\nExample: `qtype(aaaa) -> reject`',
    insertText: 'qtype(${1|a,aaaa,cname,mx,txt|})',
    isSnippet: true,
  },
  {
    label: 'fallback',
    kind: 'keyword',
    detail: 'Fallback Upstream',
    documentation: 'Default DNS upstream when no other rules match.\n\nExample: `fallback: googledns`',
    insertText: 'fallback: ${1:upstream}',
    isSnippet: true,
  },
  {
    label: 'upstream',
    kind: 'function',
    detail: 'Upstream',
    documentation: 'Match specific upstream.\n\nExample: `upstream(alidns)`',
    insertText: 'upstream(${1:name})',
    isSnippet: true,
  },
  {
    label: 'request',
    kind: 'keyword',
    detail: 'Request Section',
    documentation: 'DNS request routing rules section',
    insertText: 'request {\n\t${0}\n}',
    isSnippet: true,
  },
  {
    label: 'response',
    kind: 'keyword',
    detail: 'Response Section',
    documentation: 'DNS response routing rules section',
    insertText: 'response {\n\t${0}\n}',
    isSnippet: true,
  },
]
