// Copyright (c) 2016 Thomas Pornin <pornin@bolet.org>
// Copyright (c) 2017 Yawning Angel <yawning at schwanenlied dot me>
//
// Permission is hereby granted, free of charge, to any person obtaining
// a copy of this software and associated documentation files (the
// "Software"), to deal in the Software without restriction, including
// without limitation the rights to use, copy, modify, merge, publish,
// distribute, sublicense, and/or sell copies of the Software, and to
// permit persons to whom the Software is furnished to do so, subject to
// the following conditions:
//
// The above copyright notice and this permission notice shall be
// included in all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
// EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
// MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
// NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS
// BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
// ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
// CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
// SOFTWARE.

package ct32

func InvSbox(q *[8]uint32) {
	// AES S-box is:
	//   S(x) = A(I(x)) ^ 0x63
	// where I() is inversion in GF(256), and A() is a linear
	// transform (0 is formally defined to be its own inverse).
	// Since inversion is an involution, the inverse S-box can be
	// computed from the S-box as:
	//   iS(x) = B(S(B(x ^ 0x63)) ^ 0x63)
	// where B() is the inverse of A(). Indeed, for any y in GF(256):
	//   iS(S(y)) = B(A(I(B(A(I(y)) ^ 0x63 ^ 0x63))) ^ 0x63 ^ 0x63) = y
	//
	// Note: we reuse the implementation of the forward S-box,
	// instead of duplicating it here, so that total code size is
	// lower. By merging the B() transforms into the S-box circuit
	// we could make faster CBC decryption, but CBC decryption is
	// already quite faster than CBC encryption because we can
	// process two blocks in parallel.
	var q0, q1, q2, q3, q4, q5, q6, q7 uint32

	q0 = ^q[0]
	q1 = ^q[1]
	q2 = q[2]
	q3 = q[3]
	q4 = q[4]
	q5 = ^q[5]
	q6 = ^q[6]
	q7 = q[7]
	q[7] = q1 ^ q4 ^ q6
	q[6] = q0 ^ q3 ^ q5
	q[5] = q7 ^ q2 ^ q4
	q[4] = q6 ^ q1 ^ q3
	q[3] = q5 ^ q0 ^ q2
	q[2] = q4 ^ q7 ^ q1
	q[1] = q3 ^ q6 ^ q0
	q[0] = q2 ^ q5 ^ q7

	Sbox(q)

	q0 = ^q[0]
	q1 = ^q[1]
	q2 = q[2]
	q3 = q[3]
	q4 = q[4]
	q5 = ^q[5]
	q6 = ^q[6]
	q7 = q[7]
	q[7] = q1 ^ q4 ^ q6
	q[6] = q0 ^ q3 ^ q5
	q[5] = q7 ^ q2 ^ q4
	q[4] = q6 ^ q1 ^ q3
	q[3] = q5 ^ q0 ^ q2
	q[2] = q4 ^ q7 ^ q1
	q[1] = q3 ^ q6 ^ q0
	q[0] = q2 ^ q5 ^ q7
}

func InvShiftRows(q *[8]uint32) {
	for i := 0; i < 8; i++ {
		x := q[i]
		q[i] = (x & 0x000000FF) |
			((x & 0x00003F00) << 2) | ((x & 0x0000C000) >> 6) |
			((x & 0x000F0000) << 4) | ((x & 0x00F00000) >> 4) |
			((x & 0x03000000) << 6) | ((x & 0xFC000000) >> 2)
	}
}

func InvMixColumns(q *[8]uint32) {
	var q0, q1, q2, q3, q4, q5, q6, q7 uint32
	var r0, r1, r2, r3, r4, r5, r6, r7 uint32

	q0 = q[0]
	q1 = q[1]
	q2 = q[2]
	q3 = q[3]
	q4 = q[4]
	q5 = q[5]
	q6 = q[6]
	q7 = q[7]
	r0 = (q0 >> 8) | (q0 << 24)
	r1 = (q1 >> 8) | (q1 << 24)
	r2 = (q2 >> 8) | (q2 << 24)
	r3 = (q3 >> 8) | (q3 << 24)
	r4 = (q4 >> 8) | (q4 << 24)
	r5 = (q5 >> 8) | (q5 << 24)
	r6 = (q6 >> 8) | (q6 << 24)
	r7 = (q7 >> 8) | (q7 << 24)

	q[0] = q5 ^ q6 ^ q7 ^ r0 ^ r5 ^ r7 ^ rotr16(q0^q5^q6^r0^r5)
	q[1] = q0 ^ q5 ^ r0 ^ r1 ^ r5 ^ r6 ^ r7 ^ rotr16(q1^q5^q7^r1^r5^r6)
	q[2] = q0 ^ q1 ^ q6 ^ r1 ^ r2 ^ r6 ^ r7 ^ rotr16(q0^q2^q6^r2^r6^r7)
	q[3] = q0 ^ q1 ^ q2 ^ q5 ^ q6 ^ r0 ^ r2 ^ r3 ^ r5 ^ rotr16(q0^q1^q3^q5^q6^q7^r0^r3^r5^r7)
	q[4] = q1 ^ q2 ^ q3 ^ q5 ^ r1 ^ r3 ^ r4 ^ r5 ^ r6 ^ r7 ^ rotr16(q1^q2^q4^q5^q7^r1^r4^r5^r6)
	q[5] = q2 ^ q3 ^ q4 ^ q6 ^ r2 ^ r4 ^ r5 ^ r6 ^ r7 ^ rotr16(q2^q3^q5^q6^r2^r5^r6^r7)
	q[6] = q3 ^ q4 ^ q5 ^ q7 ^ r3 ^ r5 ^ r6 ^ r7 ^ rotr16(q3^q4^q6^q7^r3^r6^r7)
	q[7] = q4 ^ q5 ^ q6 ^ r4 ^ r6 ^ r7 ^ rotr16(q4^q5^q7^r4^r7)
}

func decrypt(numRounds int, skey []uint32, q *[8]uint32) {
	AddRoundKey(q, skey[numRounds<<3:])
	for u := numRounds - 1; u > 0; u-- {
		InvShiftRows(q)
		InvSbox(q)
		AddRoundKey(q, skey[u<<3:])
		InvMixColumns(q)
	}
	InvShiftRows(q)
	InvSbox(q)
	AddRoundKey(q, skey)
}
